;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

;;;
;;; Code for running VCODE process. 
;;; (Original code written by Timothy Freeman.  Major modifications
;;;  by Guy Blelloch.)
;;;
;;; Tested under:
;;; Lucid/Sun common lisp (with :lcl4.1 and :sun in the features list)
;;; Allegoro common lisp (version 3.1, with :allegro and :allegro-v3.1 
;;;                        in the features list)

(in-package :nesl-lisp) 

;; DEFINITION OF CONFIGURATION STRUCTURE.

(defstruct config
  memsize
  (checkargs t)
  (interp-file "vinterp")
  (munch-lines 0)
  (afs t)
  user-name
  machine-name
  rsh-command
  rcp-source-machine
  (background-function nil))

(defparameter *current-config* nil)

(defparameter *config-list* nil)

(defmacro add-configuration (name config)
  `(push (cons ',name ,config) *config-list*))

(defun nesl::configuration ()
  (let ((machine-name (config-machine-name *current-config*)))
    (format t "~%Machine Name:      ~s~
               ~%Interpreter File:  ~s~
               ~%Memory Size:       ~s~
               ~%Argument Checking: ~s"
	    machine-name
	    (config-interp-file *current-config*)
	    (config-memsize *current-config*)
	    (config-checkargs *current-config*))
    (when (not (equal machine-name "local"))
	  (format t "~%rsh Command:       ~s"
		  (config-rsh-command *current-config*))
	  (format t "~%Remote User Name:  ~s"
		  (config-user-name *current-config*)))
    (when (not (config-afs *current-config*))
	  (format t "~%No access to AFS, will be using rcp for copying."))))

(defun nesl::list-machines ()
  (format t "~%The current machine configurations are:")
  (mapcar #'(lambda (a) (format t "~%~a" (car a))) (reverse *config-list*))
  (format t "~%To use one type: use_machine(NAME).")
  nil)

(defun nesl::list_machines () (nesl::list-machines))

(defun nesl::set-memsize (size)
  (when (not (integerp size))
    (nesl-error "The size argument to SET-MEMSIZE must be an integer."))
  (setf (config-memsize *current-config*) size))

(defun nesl::set_memory_size (size) (nesl::set-memsize size))

(defun nesl::set_interpreter_file (filename)
  (setf (config-interp-file *current-config*) filename))

(defun nesl::set_username (user)
  (setf (config-user-name *current-config*) user))

(defun nesl::set_argcheck ()
  (setf (config-checkargs *current-config*)
	(not (config-checkargs *current-config*))))

(defun get-configuration (config-name)
  (or (cdr (assoc config-name *config-list*))
      (nesl-error "~%Machine ~a is not a valid configuration.~%~
                     Use list_machines() to list the configurations."
		  config-name)))

(defmacro nesl::use-machine (&optional config-name)
  (if config-name 
      `(progn
	 (setq *current-config* (get-configuration ',config-name))
	 (nesl::configuration))
    `(nesl::list-machines)))

(defmacro nesl::use_machine (&optional config-name)
  `(nesl::use-machine ,config-name))

(defun nesl::cm-finger ()
  (run-shell-line
   (format nil "~a -l ~a ~a /usr/local/bin/cmfinger"
	   (config-rsh-command *current-config*)
	   (config-user-name *current-config*)
	   (config-machine-name *current-config*))))

(defun nesl::cmfinger ()
  (nesl::cm-finger))

(defun init-machine-configuration ()
  (let ((config (or (getenv "NESL_CONFIG") "local")))
    (format t "~%; Using machine configuration ~s." config)
    (setq *current-config* 
	  (cdr (assoc (let ((*package* (find-package 'nesl)))
			(declare (special *package*))
			(lisp::read-from-string config))
		      *config-list*)))))

(defvar *username* nil)

(defun user-name ()
  (if *username* *username*
    (let ((getenv (getenv "USER")))
      (setq *username* 
	    (if getenv getenv
	      (progn
		(format t "What is your user name? ")
		(read-line)))))))

(defun uniquify-file-name (file) 
  (concatenate 'string (namestring file) (user-name)))

(defparameter *codefile* "vcode_code")
(defparameter *outfile*  "vcode_out")
(defparameter *errfile*  "vcode_err")
(defparameter *runcount* 0)

(defun temp-file (name)
  (concatenate 'string *vcode-temp-directory* name (user-name)))

(defun temp-file-num (name number)
  (temp-file (format nil "~a_~a" name (user-name) number)))

(defparameter cray-mem-add-factor 4)

(defun cray-qsub-command (filename errfile mem-size maxtime qname)
  (let ((mem-size (ceiling (+ cray-mem-add-factor 
			      (/ mem-size (float 1048576))))))
    (if (eql qname 'default)
	(format nil "qsub -e ~a -o /dev/null -lM ~aMw -lm ~aMw ~
                        -lT ~a -lt ~a -r nesl ~a"
		errfile mem-size mem-size maxtime maxtime filename)
      (format nil "qsub -e ~a -o /dev/null -lM ~aMw -lm ~aMw ~
                      -lT ~a -lt ~a -r nesl -q ~a ~a"
	      errfile mem-size mem-size maxtime maxtime 
	      (string-downcase (string qname)) filename))))

(defun cm-qsub-command (filename errfile mem-size maxtime qname)
  (declare (ignore mem-size))
  (if (eql qname 'default)
      (format nil "/usr/local/bin/qsub -e ~a -o /dev/null ~
                        -lT ~a -lt ~a -r nesl -q oneseq ~a"
	    errfile (* 2 maxtime) maxtime filename)
    (format nil "/usr/local/bin/qsub -e ~a -o /dev/null ~
                      -lT ~a -lt ~a -r nesl -q ~a ~a"
	      errfile (* 2 maxtime) maxtime 
	      (string-downcase (string qname)) filename)))

(defun remote-execute (command-line)
  (let ((machine-name (config-machine-name *current-config*))
	(user-name (config-user-name *current-config*))
	(rsh-command (config-rsh-command *current-config*)))
    (run-shell-line
     (format nil "~a -l ~a ~a \"~a\"" 
	     rsh-command user-name machine-name command-line))))

(defun vinterp-command-line (vcodefile outfile &optional (mem-size nil))
  (let* ((interp-file (concatenate 'string *vcode-interpreter-dir* 
				   (config-interp-file *current-config*)))
	 (mem-size (or mem-size (config-memsize  *current-config*)))
	 (flags (if (config-checkargs *current-config*)
		    (list "-c" "-m" mem-size)
		  (list "-m" mem-size))))
    (format nil "~a~{ ~a~} ~a > ~a" interp-file flags vcodefile outfile)))

(defun remote-copy-and-execute (vcodefile outfile)
  (let* ((interp-file (concatenate 'string *vcode-interpreter-dir* 
				   (config-interp-file *current-config*)))
	 (source-machine (config-rcp-source-machine *current-config*))
	 (mem-size (config-memsize *current-config*))
	 (copy-file (concatenate 'string *nesl-path* "src/runNESL")))
    (remote-execute 
     (format nil "rcp -p ~a:~a remote.ex;./remote.ex ~a ~a ~a ~a ~a"
	     source-machine copy-file
	     source-machine interp-file vcodefile mem-size outfile))))

(defun execute-vcode-file (vcodefile outfile)
  (cond 
   ;; FOR RUNNING VCODE LOCALLY
   ((equal (config-machine-name *current-config*) "local")
    (when *verbose* (format t "~%Starting VCODE process locally..."))
    (run-shell-line (vinterp-command-line vcodefile outfile)))

   ;; RUNNING VCODE REMOTELY ON MACHINE WITH AFS ACCESS
   ((config-afs *current-config*)
    (when *verbose* 
      (format t "~%Starting VCODE process on remote machine ~a..."
	      (string-upcase (config-machine-name *current-config*))))
    (remote-execute (vinterp-command-line vcodefile outfile)))

   ;; RUNNING VCODE REMOTELY ON MACHINE WITHOUT AFS ACCESS
   (t
    (when *verbose* 
      (format t "~%Copying control file to ~a...~%"
	      (string-upcase (config-machine-name *current-config*))))
    (remote-copy-and-execute vcodefile outfile))))

(defun execute-vcode-file-background (configuration vcodefile outfile errfile
						    mem-size max-time qname)
  (when (not (config-background-function configuration))
    (nesl-error "Configuration does not support background processing."))
  (let* ((backfile (temp-file "background"))
	 (mem-size (if (eql mem-size 'default)
		       (config-memsize configuration)
		     mem-size))
	 (time (round (* 60 max-time)))
	 (qsub-command (funcall (config-background-function configuration)
				backfile errfile mem-size time qname)))
    (when (probe-file backfile) (delete-file backfile))
    (if *verbose* 
	(format t "~%Submitting VCODE background process to ~a~%    ~
                   memory size = ~a,     maximum time = ~a minute~:P~%"
		(string-upcase (config-machine-name *current-config*)) 
		mem-size max-time)
      (format t "~%"))
    (with-open-file (jstr backfile :direction :output)
      (write-line (vinterp-command-line vcodefile outfile mem-size) jstr))
    (remote-execute qsub-command)))

(defun parse-line (stream)
  (let ((list (read stream nil 0)))
    (when (not (listp list))
      (nesl-error "Error while running VCODE."))
    list))

(defun parse-segdes-line (stream)
  (let ((left-bracket (read stream nil 0)))
    (when (not (eql left-bracket 'nesl::[))
      (nesl-error "Error while running VCODE."))
    (read-delimited-list #\] stream)))

(defun convert-anything (thing)
  (cond ((typep thing 'integer) thing)
	((typep thing 'float) thing)
	((member thing '(t nil f)) (if (eql thing t) "T" "F"))
	((typep thing 'character) (char-code thing))
	(t (error "Don't know how to print out ~s." thing))))

(defparameter *verbose* t)

(defun write-vcode-file (topsym codefile definitions)
  (when *verbose* (format t "~%Writing VCODE file...."))
  (when (probe-file codefile) (delete-file codefile))
  (write-func topsym '() definitions codefile))

(defun read-vcode-result (outfile result-types)
  (when *verbose* (format t "~%Reading VCODE output...."))
  (let ((result 
	 (with-open-file (s outfile :direction :input)
	   (let ((result nil))
	     (dotimes (i (config-munch-lines *current-config*)) 
		      (read-line s nil)) ; throw away CM output
	     (dolist (type result-types)
		     (push
		      (cond
		       ((member type '(int bool nesl::float char)
				:test #'equal)
			(parse-line s))
		       ((eql type 'nesl::segdes)
			(parse-segdes-line s))
		       (t (nesl-error "Can't handle the result type ~s." 
				      type)))
		      result))
	     (nreverse result)))))
    (when (not *debug*) (delete-file outfile))
    result))

(defun run-vcode (topsym result-types definitions)
  "Run-vcode makes it possible to run vcode programs from lisp.  It
invokes the vcode interpreter for you.

Topsym is a vcode function of no arguments.  Run-vcode generates the code
for topsym to a temporary file and runs the vcode interpreter on the
temporary file.  

Each of the input-lines is converted into one line of text which is
sent as the standard input to the vcode interpreter.  The vcode function
named topsym should use vcode's input primitives to read this
information.

The vcode function named topsym should use vcode's output primitives
to write its results.  Result-types should be a list of the types of
the written results.  If the written results are indeed of the
appropriate types, they will be converted into lisp data
structures, and the list of written results will be the value returned
by run-vcode."

  (let* ((codefile (temp-file *codefile*))
	 (outfile (temp-file *outfile*)))
    (write-vcode-file topsym codefile definitions)
    (when (probe-file outfile) (delete-file outfile))
    (execute-vcode-file codefile outfile)
    (when (not (probe-file outfile))
      (nesl-error "Error while executing VCODE."))
    (read-vcode-result outfile result-types)))

(defun run-vcode-background (topsym definitions mem-time-qname-machine)
  (let* ((codefile (temp-file-num *codefile* (incf *runcount*)))
	 (outfile (temp-file-num *outfile* *runcount*))
	 (errfile (temp-file-num *errfile* *runcount*))
	 (configuration 
	  (if (eql (fourth mem-time-qname-machine) 'default)
	      *current-config* 
	    (get-configuration (fourth mem-time-qname-machine)))))
    (write-vcode-file topsym codefile definitions)
    (when (probe-file outfile) (delete-file outfile))
    (when (probe-file errfile) (delete-file errfile))
    (execute-vcode-file-background configuration codefile outfile errfile 
				   (first mem-time-qname-machine) 
				   (second mem-time-qname-machine)
				   (third mem-time-qname-machine))
    (cons outfile errfile)))

(defun read-vcode-err-file (errfile)
  (with-open-file (errstr errfile :direction :input)
    (loop 
     (multiple-value-bind (line eof) 
       (read-line errstr nil)
       (if line (write-line line))
       (if (or (not line) eof) (return)))))
  (delete-file errfile))

(defun get-background-result (out-err-files result-types)
  (when (probe-file (cdr out-err-files))
    (when *verbose* (format t "~%Getting background result...."))
    (read-vcode-err-file (cdr out-err-files))
    (read-vcode-result (car out-err-files) result-types)))
