(in-package 'nesl-lisp)

(defparameter *binops*
  '((nesl::or 2) (nesl::nor 2) (nesl::xor 2)
    (nesl::and 3) (nesl::nand 3)
    (nesl::= 4 " == ") (nesl::/= 4) (nesl::< 4) (nesl::> 4) 
    (nesl::<= 4) (nesl::>= 4)
    (nesl::+ 5) (nesl::- 5) (nesl::++ 5) (nesl::<- 5)
    (nesl::* 6) (nesl::/ 6) (nesl::-> 6)))

(defparameter *unaryops*
  '((nesl::length |#|) (nesl::negate -) (nesl::string @)))

(defun get-print-name (function)
  (or (third (assoc function *binops*))
      (second (assoc function *unaryops*))
      function))

;;;;;;;;;;;;;;;;;;;
;;;  PRETTY PRINT UTILITIES
;;;;;;;;;;;;;;;;;;;

(defun inc-lines (list inc)
  (if (null list) nil
    (cons (cons (+ inc (car (car list))) (cdr (car list)))
	  (inc-lines (cdr list) inc))))

(defun new-line (str)
  (list (cons 0 str)))

(defun append-lines-2 (lines1 lines2 &optional (max-length 120))
  (if (= (length lines1) 1)
      (let* ((line1 (car lines1))
	    (length (+ (car line1) (length (cdr line1)))))
	(if (> (+ length (length (cdr (car lines2)))) max-length)
	    (cons line1 (inc-lines lines2 (car line1)))
	  (cons (cons (car line1)
		      (format nil "~a~a" (cdr line1) (cdr (car lines2))))
		(inc-lines (cdr lines2) length))))
    (cons (car lines1) (append-lines-2 (cdr lines1) lines2 max-length))))

(defun append-lines-loop (lines-list)
  (if (= (length lines-list) 1) 
      (car lines-list)
    (append-lines-2 (car lines-list) (append-lines-loop (cdr lines-list)))))

(defun append-lines (lines1 &rest lines-list)
  (append-lines-loop (cons lines1 lines-list)))

(defun cond-append (str lines indent)
  (if (= (length lines) 1)
      (append-lines (new-line str) lines)
    (append (new-line str) (inc-lines lines indent))))

(defun cond-wrap-paren (flag body)
  (if flag
      (append-lines (new-line "(") body (new-line ")"))
    body))

;;;;;;;;;;;;;;;;;;;
;;;  PRETTY PRINT ROUTINES
;;;;;;;;;;;;;;;;;;;

(defun cnesl-with-binding (a)
  (let ((pattern-lines (cnesl-exp (first a) 1))
	(exp-lines (cnesl-exp (second a))))
    (append-lines pattern-lines (new-line " = ") exp-lines)))

(defun cnesl-with-bindings (a)
  (let ((bind1 (cnesl-with-binding (car a))))
    (if (= (length a) 1) bind1
      (append (append-lines bind1 (new-line "; "))
	      (cnesl-with-bindings (cdr a))))))

(defun cnesl-with (bindings exp prec)
  (let ((binding-lines (cnesl-with-bindings bindings))
	(exp-lines (cnesl-exp exp)))
    (cond-wrap-paren 
     (> prec 0)
     (append
      (cond-append "let " binding-lines 2)
      (cond-append "in " exp-lines 2)))))

(defun cnesl-over-binding (a)
  (if (and (symbolp (first a)) (eql (first a) (second a)))
      (cnesl-exp (first a))
    (let ((pattern-lines (cnesl-exp (first a) 1))
	  (exp-lines (cnesl-exp (second a))))
      (append-lines pattern-lines (new-line " in ") exp-lines))))

(defun cnesl-over-bindings (previous-binds a)
  (if (null a) previous-binds
    (cnesl-over-bindings
     (append-lines-2 (append-lines previous-binds (new-line "; "))
		     (cnesl-over-binding (car a))
		     50)
     (cdr a))))

(defun cnesl-over (bindings exp &optional sieve)
  (let* ((binding-lines 
	  (cnesl-over-bindings (cnesl-over-binding (car bindings))
			       (cdr bindings)))
	(head 
	 (if (eql (first (car bindings)) exp)
	     binding-lines
	   (append-lines-2 (append-lines (cnesl-exp exp 1) (new-line ": "))
			   binding-lines 50)))
	(body
	 (if sieve 
	     (append-lines-2 (append-lines head (new-line " | "))
			     (cnesl-exp sieve) 50)
	   head)))
    (append-lines (new-line "{") body (new-line "}"))))

(defun cnesl-if (cond thenp elsep prec)
  (let ((if-lines (cnesl-exp cond 10))
	(then-lines (cnesl-exp thenp))
	(else-lines (cnesl-exp elsep)))
    (cond-wrap-paren 
     (> prec 0)
     (append 
      (cond-append "if " if-lines 2)
      (cond-append "then " then-lines 2)
      (cond-append "else " else-lines 2)))))

(defun cnesl-infix (op p1 p2 currentprec lprec rprec)
  (let ((p1-lines (cnesl-exp p1 lprec))
	(p2-lines (cnesl-exp p2 rprec))
	(op-line (new-line op)))
    (cond-wrap-paren 
     (> currentprec (min lprec rprec))
     (append-lines-2 (append-lines p1-lines op-line) p2-lines 50))))

(defun cnesl-pair (p1 p2 precedence)
  (cnesl-infix ", " p1 p2 precedence 1 0))

(defun cnesl-binary-op (name binop left right precedence)
  (cnesl-infix (or (third binop) (string-downcase (format nil " ~a " name)))
	       left right precedence 
	       (1- (second binop)) (second binop)))

(defun cnesl-unary-op (name op exp precedence)
  (cond-wrap-paren (>= precedence 8)
		   (append-lines (new-line (string (or (second op) name)))
				 (cnesl-exp exp 8))))

(defun cnesl-func (name exp precedence)
  (let ((binary-op? (assoc name *binops*))
	(unary-op? (assoc name *unaryops*)))
    (cond
     (binary-op? 
      (cnesl-binary-op name binary-op? (second exp) (third exp) precedence))
     (unary-op? (cnesl-unary-op name unary-op? exp precedence))
     (t (let ((fullname (string-downcase
			 (if (eql (find-package 'neslp) (symbol-package name))
			     (format nil "|~a|(" name)
			   (format nil "~a(" name)))))
	  (append-lines (new-line fullname) (cnesl-exp exp 0)
			(new-line ")")))))))

(defun cnesl-get-seq-exps (exp)
  (if (eql (car exp) 'nesl::seq_dist)
      (list (second (second exp)))
    (append (cnesl-get-seq-exps (second (second exp)))
	    (list (third (second exp))))))

(defun cnesl-seq-elts (exp-list)
  (let ((exp-lines (cnesl-exp (car exp-list) 1)))
    (if (= (length exp-list) 1) exp-lines
      (append-lines exp-lines (new-line ",") 
		    (cnesl-seq-elts (cdr exp-list))))))

(defun cnesl-seq (exp)
  (let ((exps (cnesl-get-seq-exps exp)))
    (append-lines (new-line "[") (cnesl-seq-elts exps) (new-line "]"))))

(defun cnesl-elt (exp)
  (let ((left-lines (cnesl-exp (second (second exp))))
	(right-lines (cnesl-exp (third (second exp)))))
    (append-lines left-lines (new-line "[") right-lines (new-line "]"))))

(defun cnesl-iseq (args)
  (let ((start-lines (cnesl-exp (first-pair args)))
	(stride-lines (cnesl-exp (second-pair args)))
	(end-lines (cnesl-exp (third-pair args)))
	(cline (new-line ":")))
    (append-lines (new-line "[") start-lines cline end-lines 
		  cline stride-lines (new-line "]"))))

(defun cnesl-vector (exp)
  (if (vcode-vector-p (first-pair (second exp)))
      (new-line (cnesl-print-data (nest-constant exp)))
    (cnesl-func (first exp) (second exp) 0)))

(defun cnesl-print-data (self &optional (paren-p nil))
  (cond ((nested-sequence-p self)
	 (cnesl-print-nested-sequence self))
	((nesl-struct-p self)
	 (cnesl-print-nesl-struct self paren-p))
	((vcode-vector-p self)
	 (cnesl-print-vcode-vector self))
	((characterp self) (format nil "`~a" self))
	((atom self) (format nil "~a" self))
	(t (nesl-error "Internal error: trying to print invalid type"))))
	
(defun cnesl-print-vcode-vector (self)
  (let ((data (vcode-vector-data self))
	(type (vcode-vector-type self)))
    (if (eql (length data) 1)
	(if (eql type 'char)
	    (format nil "`~a" (first data))
	  (format nil "~a" (first data)))
      (if (eql (length data) 0)
	  (format nil "**PRIM_VECT[]**")
	(format nil "**PRIM_VECT[~a~{, ~a~}]**"
		(car data) (cdr data))))))

(defun cnesl-print-nested-sequence (self)
  (let ((data (nested-sequence-data self))
	(type (nested-sequence-type self)))
    (if (eql type 'char)
	(format nil "~s" (coerce data 'string))
      (let ((ndata (mapcar #'(lambda (a) (cnesl-print-data a t))
			   (if (<= (length data) *max-print-length*)
			       data
			       (subseq data 0 *max-print-length*)))))
	(if (= (length data) 0)
	    (format nil "[]")
	  (format nil (if (<= (length data) *max-print-length*)
			  "[~a~{, ~a~}]" "[~a~{, ~a~},...]")
		  (car ndata) (cdr ndata)))))))

(defun cnesl-print-nesl-struct (self &optional (paren-p nil))
  (let ((data (nesl-struct-data self))
	(type (nesl-struct-type self)))
    (if (eql type 'nesl::pair)
	(format nil (if paren-p "(~a, ~a)" "~a, ~a")
		(cnesl-print-data (first data) t)
		(cnesl-print-data (second data)))
      (format nil "~a(~a)" type (cnesl-print-data (car data))))))

(defun cnesl-exp (a &optional (precedence 0))
  (cond ((atom a) 
	 (new-line (cond ((characterp a) (format nil "`~a" a))
			 ((symbolp a) (string-downcase (format nil "~a" a)))
			 ((stringp a) (format nil "~s" a))
			 (t (string-downcase (format nil "~s" a))))))
	((listp a) 
	 (cond ((eql (car a) 'nesl::with)
		(cnesl-with (second a) (third a) precedence))
	       ((eql (car a) 'nesl::over)
		(cnesl-over (second a) (third a)))
	       ((and (eql (car a) 'nesl::pack)
		     (listp (second a))
		     (eql (car (second a)) 'nesl::over))
		(cnesl-over (second (second a))
			    (second (third (second a)))
			    (third (third (second a)))))
	       ((eql (car a) 'nesl::vector)
		(cnesl-vector a))
	       ((eql (car a) 'nesl::if)
		(cnesl-if (second a) (third a) (fourth a) precedence))
	       ((eql (car a) 'nesl::pair)
		(cnesl-pair (second a) (third a) precedence))
	       ((or (eql (car a) 'nesl::make_sequence)
		    (eql (car a) 'nesl::seq_dist))
		(cnesl-seq a))
	       ((and (eql (car a) 'nesl::elt)
		     (eql (car (second a)) 'nesl::pair))
		(cnesl-elt a))
	       ((eql (car a) 'nesl::iseq)
		(cnesl-iseq (second a)))
	       (t
		(cnesl-func (first a) (second a) precedence))))))

(defun cnesl-type (type parenth?)
  (cond ((atom type)
	 (new-line (string-downcase (format nil "~s" type))))
	((listp type)
	 (cond ((eql (first type) 'nesl::vector)
		(append-lines 
		 (new-line "[") (cnesl-type (second type) t) (new-line "]")))
	       ((eql (first type) 'function)
		(append-lines (cnesl-type (third type) nil)
			      (new-line " -> ")
			      (cnesl-type (second type) nil)))
	       ((eql (first type) 'nesl::pair)
		(cond-wrap-paren 
		 parenth?
		 (append-lines (cnesl-type (second type) t)
			       (new-line ", ")
			       (cnesl-type (third type) nil))))
	       (t 
		(let ((funlines (new-line (string-downcase 
					   (format nil "~s" (car type))))))
		  (if (eql (length type) 1)
		      funlines
		    (append-lines funlines (new-line "(")
				  (cnesl-type (second type) nil)
				  (new-line ")")))))))
	(t (nesl-error "Invalid type ~a" type))))

(defun cnesl-typebinds (bindlist)
  (let* ((bind (car bindlist))
	 (bindlines (new-line 
		     (string-downcase 
		      (format nil "~a in ~a"(first bind) (second bind))))))
    (if (= (length bindlist) 1) bindlines
      (append-lines bindlines (cnesl-typebinds bindlist)))))

(defun cnesl-full-type (type)
  (let ((type-lines (cnesl-type (car type) nil))
	(context-lines 
	 (if (cdr type)
	     (append-lines (new-line " :: ") (cnesl-typebinds (cdr type)))
	   (new-line ""))))
    (append-lines type-lines context-lines)))

(defun cnesl-type-list (typelist)
  (if (= (length typelist) 1) 
      (cnesl-type (car typelist) t)
    (append-lines (cnesl-type (car typelist) t)
		  (cnesl-type-list (cdr typelist)))))

(defun cnesl-function (arguments type body)
  (let ((arg-lines (cnesl-exp arguments))
	(body-lines (cnesl-exp body))
	(type-lines (if type 
			(append-lines (new-line " : ") 
				      (cnesl-full-type 
				       (cons (cons 'function (car type))
					     (cdr type))))
		      (new-line ""))))
    (append (append-lines (new-line "function ") arg-lines 
			  type-lines (new-line " ="))
	    (inc-lines body-lines 2))))

(defun cnesl-datatype (arguments context)
  (let ((type-lines (cnesl-type arguments nil))
	(context-lines (if context 
			   (append-lines (new-line " :: ")
					 (cnesl-typebinds context))
			 (new-line ""))))
    (append-lines (new-line "datatype ") type-lines context-lines)))

(defun cnesl-toplevel (exp)
  (cond ((listp exp)
	 (cond ((eql (car exp) 'nesl::defop)
		(cnesl-function (second exp) (third exp) (fourth exp)))
	       ((eql (car exp) 'nesl::defrec)
		(cnesl-datatype (second exp) (cddr exp)))
	       (t (cnesl-exp exp))))
	(t (cnesl-exp exp))))

(defun pprint-nesl (a &optional (stream t))
  (let ((exp-str (cnesl-toplevel a)))
    (dolist (line exp-str)
      (write-char #\newline stream)
      (dotimes (i (car line)) (write-char #\space stream))
      (write-string (cdr line) stream)
      )))

(defun pprint-nesl-string-rec (lines indent)
  (if (null lines) ""
    (concatenate 
     'string 
     (let* ((line (car lines))
	    (nindent (+ (car line) indent)))
       (format nil "~%~a~a" 
	       (make-sequence 'string nindent :initial-element #\space)
	       (cdr line)))
     (pprint-nesl-string-rec (cdr lines) indent))))

(defun pprint-nesl-string (exp &optional (indent 0))
  (pprint-nesl-string-rec (cnesl-exp exp) indent))
