(in-package 'nesl-lisp)

(defun speaknesl nil
  (setq nudl (list 'neslnud))
  (setq ledl (list 'neslled))
  (setq lbpl (list 'nesllbp))
  nil) 

(defun forgetnesl nil
  (setq nudl (list 'nud))
  (setq ledl (list 'led))
  (setq lbpl (list 'lbp))
  nil) 

(defun check-token (symbol)
  (if (eql token symbol) 
      (progn (advance) t)
    nil))

(defun check (symbol)
  (if (equal token symbol)
      (advance)
      (cgolerr (cat '|missing | symbol '| before | token '|.|) 0 nil)))

(defun get-token ()
  (prog1 token (advance)))

(defun force-token (symbol error-message)
  (if (eql token symbol) 
      (progn (advance) t)
    (cgolerr (format nil error-message token))))

(defmacro defconstruct (name body)
  `(setf (get ',name 'neslnud)
      #'(lambda () ,body)))

(defun add-infix (op precedence &optional trans)
  (let ((newop (or trans op)))
    (setf (get op 'neslled) 
	  #'(lambda (left) (list newop (list 'nesl::pair 
					     left (parse precedence)))))
    (setf (get op 'nesllbp) precedence)))

(defmacro add-infix-fun (op lprec rprec code)
  `(progn
    (setf (get ,op 'neslled) 
	  #'(lambda (left) (let ((right (parse ,rprec))) ,code)))
    (setf (get ,op 'nesllbp) ,lprec)))

(defun add-prefix (op precedence &optional trans)
  (let ((newop (or trans op)))
    (setf (get op 'neslnud) 
	  #'(lambda () (list newop (parse precedence))))))

(defun parsecontext ()
  (let* ((typevar (get-token))
	(c (force-token 
	    'nesl::in 
	    "Expected an IN instead of ~a when defining a type context."))
	(typeclass (get-token))
	(context (list typevar typeclass)))
    (declare (ignore c))
    (if (check-token 'nesl::|;|)
	(cons context (parsecontext))
      (list context))))

(defparameter *currently-parsing-type* nil)

(defun parsetype ()
  (let ((*currently-parsing-type* t))
    (declare (special *currently-parsing-type*))
    (let ((type (parse 2))
	  (context (if (check-token 'nesl::|::|)
		       (parsecontext)
		     nil)))
      (if (eql (car type) 'nesl::>>)
	  (cons (reverse (cdr type)) context)
	(cgolerr "Not a function type")))))

;;(learn 'nesl) 

(setf (get ' 'nesllbp) (- 1)) 

(setf (get 'nesl::|;| 'nesllbp) (- 1)) 

(defconstruct if
  (let ((condition (parse 3))
	(c1 (force-token 
	    'nesl::then "Expected a THEN instead of ~a in an IF construct."))
	(then-part (parse 3))
	(c2 (force-token 
	    'nesl::else "Expected an ELSE instead of ~a in an IF construct."))
	(else-part (parse 3)))
    (declare (ignore c1 c2))
    `(if ,condition ,then-part ,else-part)))

(defun parse-let-bindings ()
  (let* ((pattern (parse 3))
	 (c (force-token 
	     'nesl::=  "Expected an = instead of ~a after the pattern ~
                        in a LET binding."))
	 (expression (parse 3)))
    (declare (ignore c))
    (if (and (check-token 'nesl::|;|) (not (eql token 'nesl::in)))
	(cons (list pattern expression) (parse-let-bindings))
      (list (list pattern expression)))))

(defconstruct nesl::let
  (let ((bindings (parse-let-bindings)))
    (force-token 
     'nesl::in 
     "Expected a ; or IN instead of ~a after a binding of a LET construct.")
    (list 'nesl::with bindings (parse 3))))

(defconstruct nesl::function
  (let* ((funname (get-token))
	 (*current-fundef* funname)
	 (argument (list (parse 3)))
	 (type (if (check-token 'nesl::|:|) (parsetype) nil))
	 (doc (if (stringp token) (get-token))))
    (declare (special *current-fundef*))
    (force-token 
     'nesl::= 
     (if type 
        "Expected an = instead of ~a after the type declaration of a function."
       "Expected an = or : instead of ~a after the parameters of a function."))
    (list 'defop (cons funname argument) type (parse 3)
	  :documentation doc)))

(defconstruct nesl::datatype
  (let* ((*currently-parsing-type* t)
	 (typename (get-token))
	 (type (list (parse 3)))
	 (context (if (check-token 'nesl::|::|)
		      (parsecontext)
		    nil)))
    (declare (special *currently-parsing-type*))
    (cons 'defrec (cons (cons typename type) context))))

(defconstruct nesl::describe
  (let ((dval (progn (check 'nesl::|(|) token)))
    (advance)
    (check 'nesl::|)|)
    (list 'nesl::describe dval))) 

(defun parse-empty-sequence ()
  (let ((*currently-parsing-type* t))
    (declare (special *currently-parsing-type*))
    (flatten-exp (list (make-nested-sequence :data nil))
		 (expand-type (list 'nesl::vector (parse 29))
			      *definitions*))))

(defun parse-integer-sequence (val1)
  (let ((val2 (parse 3))
	(val3 (if (check-token 'nesl::|:|) (parse 3) 1))
	(c (force-token 
	    'nesl::] 
	    "Expected a ] instead of ~a to terminate an integer sequence.")))
    (declare (ignore c))
    `(nesl::iseq (nesl::pair ,val1 (nesl::pair ,val3 ,val2)))))

(defun parse-sequence-list (depth current-list)
  (if (> depth 100)
      (cgolerr "The maximum length for a sequence in the [a,b,c,...] format is 100.
For larger sequences use one of read_object_from_file, read_int_seq_from_file,
or read_float_seq_from_file.")
    (if (check-token 'nesl::|,|)
	(parse-sequence-list 
	 (+ depth 1)
	 `(nesl::make_sequence (nesl::pair ,current-list ,(parse 4))))
      (progn 
	(force-token 
	 'nesl::] "Expected a ] instead of ~a to terminate a sequence.")
	current-list))))

(defconstruct nesl::[ 
  (if *currently-parsing-type*
      (prog1 (list 'nesl::vector (parse 29)) (check 'nesl::]))
    (if (check-token 'nesl::])
	(parse-empty-sequence)
      (let ((val1 (parse 4)))
	(if (check-token 'nesl::|:|)
	    (parse-integer-sequence val1)
	  (parse-sequence-list 1
	   `(nesl::seq_dist (nesl::pair ,val1 1))))))))

(defun parse-apply-to-each-binds ()
  (if (check-token 'nesl::|;|)
      (let* ((bind-pattern (parse 3))
	     (bind-exp 
	      (if (check-token 'nesl::in) (parse 3) 
		(if (symbolp bind-pattern)
		    bind-pattern
		  (cgolerr "Invalid binding in an APPLY-TO-EACH construct")))))
	(cons (list bind-pattern bind-exp) (parse-apply-to-each-binds)))))
	    
(defconstruct nesl::{
  (let* ((body (parse 3))
	 (body? (check-token 'nesl::|:|))
	 (first-bind-pattern (if body? (parse 3) body))
	 (first-bind-exp 
	  (if (check-token 'nesl::in) (parse 3) 
	    (if (symbolp first-bind-pattern)
		first-bind-pattern
	      (cgolerr "Invalid binding in an APPLY-TO-EACH construct."))))
	 (bindings (cons (list first-bind-pattern first-bind-exp)
			 (parse-apply-to-each-binds)))
	 (sieve (if (check-token 'nesl::|\||) (parse 3) nil))
	 (c (force-token 
	     'nesl::} 
	     "Expected a } instead of ~a to terminate an APPLY-TO-EACH."))
	 (main `(over ,bindings 
		  ,(if sieve `(nesl::pair ,body ,sieve) body))))
    (declare (ignore c))
    (if sieve
	`(nesl::pack ,main)
      main)))

(progn
  (setf
   (get 'nesl::[ 'neslled)
   #'(lambda (left)
       (prog1
	   `(nesl::elt (nesl::pair ,left ,(parse 3)))
	 (force-token 
	  'nesl::]
	  "Expected ] instead of ~a to terminate a sequence reference."))))
  (setf (get 'nesl::[ 'nesllbp) '30)) 

(defconstruct nesl::|(|
  (let ((exp (parse 3)))
    (force-token 
     'nesl::|)|
     "Expected ) instead of ~a to terminate a parenthesized expression.")
    exp))

(progn
  (setf (get 'nesl::|(| 'neslled)
        #'(lambda (left)
            (prog2
              nil
              (cons left
                    (if (not (equal token 'nesl::|)|)) (list (parse 3))))
              (check 'nesl::|)|))))
  (setf (get 'nesl::|(| 'nesllbp) '30)) 

(defconstruct nesl::&
  (let ((r (parse 35)))
    (list 'nesl::print_debug (format nil "~a: " r) r)))

(deftok ->)
(deftok >>)
(deftok <-)
(deftok |::|)
(deftok ++)
;;(deftok |:+|)
;;(deftok |+:|)
(deftok /=)
(deftok ==)
(deftok <=)
(deftok >=)
(deftok &=)
(deftok |:=|)
(deftok |\|\||)

(add-infix-fun 'nesl::|,| 4 3 (list 'nesl::pair left right))
(add-infix-fun 'nesl::>> 3 3 (list 'nesl::>> left right))
(add-infix-fun 'nesl::= 2 2 (list 'nesl::set left right))
(add-infix-fun 'nesl::|:=| 5 5 (list '|:=| left right))
(add-infix-fun 'nesl::&= 2 2
  (list 'background left 
	(match-args 
	 right (list-pair 'exp '(|:=| nesl::mem default) 
			  '(|:=| nesl::maxtime 1)
			  '(|:=| nesl::qname default)  
			  '(|:=| nesl::machine default)))))

(add-infix 'nesl::<- 23)
(add-infix 'nesl::-> 24)
(add-infix 'nesl::++ 20)

(add-prefix 'nesl::|#| 25 'nesl::length)
(add-prefix 'nesl::@ 25 'nesl::string)
(add-prefix 'nesl::- 25 'nesl::negate)

(add-infix 'nesl::nand 8)
(add-infix 'nesl::and 8)
(add-infix 'nesl::or 7)
(add-infix 'nesl::nor 7)
(add-infix 'nesl::xor 7)
(add-infix 'nesl::== 10 'nesl::=)
(add-infix 'nesl::/= 10)
(add-infix 'nesl::< 10)
(add-infix 'nesl::> 10)
(add-infix 'nesl::<= 10)
(add-infix 'nesl::>= 10)
(add-infix 'nesl::+ 20)
(add-infix 'nesl::- 20)
(add-infix 'nesl::* 21)
(add-infix 'nesl::/ 21)
(add-infix 'nesl::|\|\|| 21 'nesl::pad_string)

(add-infix-fun 'nesl::^ 22 22
	       (if (eq right 2) 
		   `(nesl::power2 ,left)
		 (if (eq right 3) 
		     `(nesl::power3 ,left) 
		   `(nesl::power (nesl::pair ,left ,right)))))

(defconstruct nesl::ctrans
  (let ((code (parse 0)))
    (pprint code)
    `(NOOP)))

(defun pairp (a) (and (listp a) (eql (first a) 'nesl::pair)))
(defun defaultp (a) (and (listp a) (eql (first a) '|:=|)))
(defun car-pair (a) (if (pairp a) (second a) a))
(defun cdr-pair (a) (if (pairp a) (third a) nil))
(defun cddr-pair (a) (cdr-pair (cdr-pair a)))
(defun first-pair (a) (car-pair a))
(defun second-pair (a) (car-pair (cdr-pair a)))
(defun third-pair (a) (car-pair (cddr-pair a)))
(defun cons-pair (a b) (if (null b) a (list 'nesl::pair a b)))
(defun list-pair (&rest args) (list-pair-r args))
(defun list-pair-r (args)  
  (if args (cons-pair (car args) (list-pair-r (cdr args))) nil))
(defun flatten-pair (a)
  (if (null a) a
    (cons (car-pair a) (flatten-pair (cdr-pair a)))))

(defun default-key (a)
  (if (not (defaultp a))
      (nesl-error "Bad format in an argument pattern.")
    (second a)))

(defun default-val (a)
  (if (not (defaultp a))
      (nesl-error "Bad format in an argument pattern.")
    (third a)))

(defun match-args (args defaults)
  (if (or (null defaults) (defaultp (car-pair defaults)))
      (progn 
	(check-defaults args defaults)
	(fill-defaults args defaults))
    (cons-pair (car-pair args) 
	       (match-args (cdr-pair args) (cdr-pair defaults)))))

(defun find-default (key tree)
  (if (null tree) nil
    (if (eql key (default-key (car-pair tree)))
	(default-val (car-pair tree))
      (find-default key (cdr-pair tree)))))

(defun fill-defaults (args defaults)
  (if (null defaults) nil
    (let ((val (or (find-default (default-key (car-pair defaults)) args)
	           (default-val (car-pair defaults)))))
      (cons-pair val (fill-defaults args (cdr-pair defaults))))))

(defun check-defaults (args defaults)
  (if (null args) nil
    (if (find-default (default-key (car-pair args)) defaults)
	(check-defaults (cdr-pair args) defaults)
      (nesl-error "Invalid optional argument ~a" 
		  (default-key (car-pair args))))))

