/*
* Copyright (c) 1992 Carnegie Mellon University 
*                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
*                                  Jonathan Hardwick, Jay Sipelstein,
*                                  Marco Zagha
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
* CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
* ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The SCAL project requests users of this software to return to 
*
*  Guy Blelloch				guy.blelloch@cs.cmu.edu
*  School of Computer Science
*  Carnegie Mellon University
*  5000 Forbes Ave.
*  Pittsburgh PA 15213-3890
*
* any improvements or extensions that they make and grant Carnegie Mellon
* the rights to redistribute these changes.
*/

/* Module for dealing with stack:
 * 1. the stack represents the actual runtime behavior of the VCODE program.
 * 2. pointers from stack entries indicate the entry in the vblock list 
 *    that is responsible for this vector.  vectors are NOT duplicated 
 *    by copy and push operations, only new pointers are created. see
 *    vstack.c for more details.
 *
 * POP and COPY operations (implicit or explicit) are carried on in the stack
 * and in the vstack.  It is the responsibility of the vstack to manage
 * vector memory.
 */

#include <stdio.h>
#include <ctype.h>		/* for pseudo parser in do_read */
#include "vcode.h"
#include "vstack.h"
#include "program.h"
#include "constant.h"
#include "stack.h"

int *read_array;		/* input buffer for do_read() */
int input_size_max = 1000;	/* size of buffer */

/* The stack is an array of stack_entry's (which are pointers
 * to the vb of the associated vector).  stack_size is an index
 * into the stack such that stack[stack_index] is the element on the
 * top of the stack.
 */
vb_t **stack;
int stack_index;

/* ------------------------ stack_entry allocator -----------------*/
#define STACK_MAX_INITIAL 5	/* initial size of stack */
int stack_max;			/* number of stack_entry's allocated */

/* if there is room on stack, return pointer to next element,
 * otherwise allocate more space.
 * define this as a macro to optimize the standard case.
 */
#define new_se()					\
    (++stack_index < stack_max ? stack_index : new_se_real())

static stack_entry_t new_se_real ()
{
    stack_max *= 2;			/* double size */
    stack = (vb_t **) 
	    realloc(stack, stack_max * sizeof(stack_entry_t));
    if (stack == NULL) {
	fprintf(stderr, "vinterp: error allocating internal structures (stack_entry).\n");
	vinterp_exit (1);
    }
    return stack_index;
}
    
/* decriment stack_size to free a stack_entry_t */
#define free_se(_se) 					\
    do {						\
	stack_index--;					\
    } while (0)

static void se_alloc_init()
{
    stack = (vb_t **)
	    malloc (STACK_MAX_INITIAL * sizeof (stack_entry_t));
    if (stack == NULL) {
        fprintf(stderr, "vinterp: error allocating stack (se_alloc_init).\n");
        vinterp_exit (1);
    }
    stack_index = -1;
    stack_max = STACK_MAX_INITIAL;
}

/* ---------------------------stack manipulation code-------------------*/
void stack_init()
{
    se_alloc_init();

    /* allocate input buffer */
    read_array = (int *)malloc (input_size_max*sizeof(int));
    if (read_array == NULL) {
	fprintf(stderr, "vinterp: can't malloc read buffer.\n\tRerun with smaller -r value.\n");
	vinterp_exit(1);
    }
}
    
/* Pop an item from the stack: first, pop out vstack entry, then 
 * remove se from stack.
 */
void stack_pop(se)
stack_entry_t se;
{
    stack_entry_t to_se;		/* we push up elements under pop */

    assert ((se <= stack_index) && (se >= 0));

    vstack_pop(stack[se]);
    /* copy each element into the previous one */
    for (to_se = se; to_se < stack_index; to_se ++)
	stack[to_se] = stack[to_se + 1];
    
    assert(to_se == stack_index);		/* copy up to TOS */

    free_se(to_se);				/* free item at end */
}

/* push a new vector on the stack, given length and type, and seglen */
void stack_push(len, type, seglen)
int len;
TYPE type;
int seglen;
{
    stack_entry_t se = new_se();
    vb_t *new_vb;

    new_vb = new_vector(len, type, seglen);
    stack[se] = new_vb;
}

/* handle VCODE conditionals: grab bool scalar from top of stack and return
 * its truth value */
int do_cond(pe)
prog_entry_t *pe;
{
    vb_t *flag_vb = se_vb(TOS);
    int result;

    /* check type and length */
    if (check_args && (flag_vb->type != Bool || flag_vb->len != 1)) {
	fprintf(stderr, "vinterp: line %d: IF: top of stack not bool scalar.\n", pe->lineno);
	vinterp_exit (1);
    }

    /* get value from vector */
    result = (int) ext_vub(flag_vb->vector, 0, 1, SCRATCH);

    /* remove flag from stack */
    stack_pop(TOS);
    return result;
}

/* scalar constants get put on the stack using replace.
 * vector constants are already in vector mem, so just create stack entry
 *   for it.
 */
void do_const(pe)
prog_entry_t *pe;
{
    vec_p dest;

    if (pe->is_in_const_pool) {
	/* vector constant */
	stack_entry_t se = new_se();
	int index = pe->misc.const_pool_index;
	stack[se] = const_vb(index);
    } else {
	/* scalar constant */
	stack_push(1, pe->type, -1);	/* make a 1 elt vector */
	dest = se_vb(TOS)->vector;

	switch (pe->type) {		/* put in the value */
	    case Int: 
		assert_mem_size(rep_vuz_scratch(1));
		rep_vuz(dest, 0, pe->misc.int_const, 1, SCRATCH);
		break;
	    case Float: 
		assert_mem_size(rep_vud_scratch(1));
		rep_vud(dest, 0, pe->misc.float_const, 1, SCRATCH);
		break;
	    case Bool: 
		assert_mem_size(rep_vub_scratch(1));
		rep_vub(dest, 0, pe->misc.bool_const, 1, SCRATCH);
		break;
	    default:
		fprintf(stderr, "vinterp: illegal type for CONST.\n");
		vinterp_exit (1);
		break;
	}
    }
}

/* copy count elements of the stack, starting from start, moving up:
 * COPY 2 3 when stack looks like (0 1 2 3 4 5) results in (0 1 2 3 4 5 1 2)
 */
void do_copy(count, start)
int start,	/* 0 is top of stack */
    count;
{
    int from, last;
    stack_entry_t se;

    if (check_args && (start + count > stack_size)) {
	fprintf(stderr, "vinterp: can't do a COPY %d %d; only %d items on the stack.\n",
		count, start, stack_size);
	vinterp_exit (1);
    }
	
    /* copy each se until done */
    for (from = stack_index - start - count + 1, last = stack_index - start;
	 from <= last;
	 from ++) {
	se = new_se();
	stack[se] = stack[from];
	vstack_copy(se_vb(se));
    }
}

/* Pop out values from middle of stack:
 * POP 2 3 when stack is (0 1 2 3 4 5) results in (0 3 4 5). */
void do_pop(count, start)
int count, 
    start;				/* 0 is top of stack */
{
    stack_entry_t bad, num, from;

    /* error checking */
    if (check_args) {
	if (start + count > stack_size) {
	    fprintf(stderr, "vinterp: can't do a POP %d %d; only %d items on the stack.\n",
		    count, start, stack_size);
	    vinterp_exit (1);
	} else if (count < 0 || start < 0) {
	    fprintf(stderr, "vinterp: negative argument: POP %d %d\n", count, start);
	    vinterp_exit (1); 
	}
    }
	
    /* vstack_pop the bad stack elements */
    for (bad = stack_index - start, num = 0; num < count; num++, bad --) {
	vstack_pop(se_vb(bad));
    }

    /* copy the remaining elements up */
    for (from = stack_index - start + 1; from <= stack_index; from ++) {
	stack[from-count] = stack[from];	/* move things count back */
    }

    stack_index -= count;
}

/* get_args: return pointers to top elements on the stack */
void get_args(pe, arg_array)
prog_entry_t *pe;
stack_entry_t arg_array[];
{
    int numargs = pe->vopdes->arg_num;		/* # args needed */
    int i;

    if (check_args && numargs > stack_size) {
	fprintf(stderr, "vinterp: line %d: Only %d values on stack; %s reqires %d.\n",
			pe->lineno, stack_size, pe->vopdes->vopname, numargs);
	vinterp_exit (1);
    }

    for (i = 0; i < numargs; i++) {
	arg_array[numargs - i - 1] = stack_index - i;
    }
}

/* show the current state of the stack.  for debugging purposes */
void show_stack()
{
    stack_entry_t se;

    for (se = 0; se < stack_size; se++) {
	fprintf(stderr, "%d(%s) ", se_vb(se)->len, 
		type_string(se_vb(se)->type));
    }
    fprintf(stderr, "\n");
}

/* write a vector onto a stream, makes no change to the stack */
static void write_value(vb, stream)
vb_t *vb;
FILE *stream;
{
    int len = vb->len;
    int i;
    vb_t *vb_segdes = NULL;		/* if we're printing a segdes */
    vb_t *vb_print = NULL;

    if (vb->type == Segdes) {
	/* need to convert to a lengths representation and then 
	 * print the results of that.
	 */
	vb_segdes = new_vector(vb->len, Int, -1);

	assert_mem_size(len_fos_scratch(vb->seg_len, vb->len));
	len_fos(vb_segdes->vector, vb->vector, vb->seg_len, vb->len, SCRATCH);

	vb_print = vb_segdes;
	fprintf(stream, "[");

    } else {
	vb_print = vb;
	fprintf(stream, "(");
    }

    for (i = 0; i < len; i++) {
	switch (vb_print->type) {
	    case Int:
	    case Segdes: {
		int val;
		assert_mem_size(ext_vuz_scratch(len));
		val = ext_vuz(vb_print->vector, i, len, SCRATCH);
		fprintf(stream, " %d", val);
		break;
	    }
	    case Bool: {
		cvl_bool val;
		assert_mem_size(ext_vub_scratch(len));
		val = ext_vub(vb_print->vector, i , len, SCRATCH);
		fprintf(stream," %c", ((int) val) ? 'T' : 'F');
		break;
	    }
	    case Float: {
		double val;
		assert_mem_size(ext_vub_scratch(len));
		val = ext_vud(vb_print->vector, i, len, SCRATCH);
		fprintf(stream, " %.14e", val);
		break;
	    }
	    default:
		fprintf(stderr, "vinterp: internal error: Illegally typed vector.\n");
		vinterp_exit (1);
	}
    }
    if (vb->type == Segdes) {
	fprintf(stream, " ]");
	vstack_pop(vb_segdes);
    } else fprintf(stream, " )");

    fprintf(stream, "\n");
    fflush(stream);
}

/* write a string to stream; do not modify the stack */
static void write_string(vb, stream)
vb_t *vb;
FILE *stream;
{
    int len = vb->len;
    int i;

    if (vb->type != Int) {
	fprintf(stderr, "vinterp: attempt to print non-integer vector as a string\n");
	vinterp_exit(1);
    }

    assert_mem_size(ext_vuz_scratch(len));
    for (i = 0; i< len; i++) {
	fprintf(stream, "%c", (char) ext_vuz(vb->vector, i, len, SCRATCH));
    }
    fflush(stream);
}

void show_stack_values(stream)
FILE *stream;
{
    stack_entry_t se;

    for (se = 0; se < stack_size; se++) {
	write_value(se_vb(se), stream);
    }
}


/* get a value of given type from stream, and assign to buffer.
 * returns pointer to location to put next result, or NULL if
 * data wasn't found or if the end of the vector was encountered.
 */
static int *get_value(buffer, value_type, stream)
int *buffer;
TYPE value_type;
FILE *stream;
{
    int c;

    /* switch on value_type */
    switch (value_type) {
        case Float:
	    if (fscanf(stream, " %lf", (double *)buffer) == 1)
		return (int*)(1 + (double*)buffer);
	    break;
	case Bool:
	    while ((c = getc(stream)) != EOF) {
		if (!isspace(c)) {
		    if (c == 'T') 	{*buffer = 1; return ++buffer;}
		    else if (c == 'F')  {*buffer = 0; return ++buffer;}
		    else if (c == ')')  {
			(void) ungetc(c, stream);
			return NULL;
		    } else {
			fprintf(stderr, "vinterp: input error: booleans are specified by T and F.\n");
			vinterp_exit (1);
		    }
		}
	    }
	    break;
	case String:
	    /* For now, just get a char.  Eventually, we would like '\' 
	     * to do something useful.
	     */
	    c = getc(stream);
	    if ((c != (int) '\"') && (c != EOF)) {
		*buffer = c;
		return ++buffer;
	    } else {
		(void) ungetc(c, stream);
		return NULL;
	    }
	    break;
	case Int:
	    if (fscanf(stream, " %d", buffer) == 1) 
		return ++buffer;
	    break;
	default:
	    fprintf(stderr, "vinterp: internal error: illegal type in get_value()\n");
	    vinterp_exit(1);
	}
    return NULL;
}

/* perform the WRITE function; write_value() and write_string() do the work.
 * This puts two values on the stack: 
 *    boolean - T if the write was successful, F otherwise
 *    error-string - null if no error, otherwise an error message
 */
void do_write(pe, stream)
prog_entry_t *pe;
FILE *stream;
{

    if (pe->type == String)
	write_string(se_vb(TOS), stream);
    else 
	write_value(se_vb(TOS), stream);

    stack_pop(TOS);			/* remove vector from stack */

    stack_push(0, Int, -1);		/* error message */
    stack_push(1, Bool, -1);		/* error flag */
    assert_mem_size(rep_vub_scratch(1));
    rep_vub(se_vb(TOS)->vector, 0, (cvl_bool) 1, 1, SCRATCH);
}

/* read from stream.  This a portable interface to CVL.  Ideally, CVL should
 * have some I/O instructions.
 * We read values of the correct type from stream, storing them in an array.
 * When we're done, we then allocate a vector and replace each element of the
 * vector with the corresponding element of the input.
 * This puts three values on the stack: 
 *    boolean - T if the read was successful, F otherwise
 *    error-string - null if no error, otherwise an error message
 *    value vector - value actually read in
 */
void do_read(pe, stream)
prog_entry_t *pe;
FILE *stream;
{
    int count = 0;			/* total number elements read */
    int i;
    vec_p vector;			/* where to copy input */
    TYPE type = pe->type;
    TYPE read_type;			/* type of data being read: for type
					 * Segdes -> Int  */
    TYPE stack_type;			/* type of vector element:
					 * String -> Int */
    int c;				/* char read in */
    int *new_value_buffer;		/* for reading in data */

    char error_msg[256];
    int error_flag= 0;

    int start_vector;			/* must appear as first char */
    int end_vector;			/* must appear as last char */

    /* print out a nice message if working interactively */
    if (isatty(fileno(stream))) {
	char *type_string = NULL, *form_string = NULL;
	switch(type) {
	    case Int: 
		type_string = "integer";
		form_string = "(1 2 3)";
		break;
	    case Float:
		type_string = "float";
		form_string = "(1.0 2.0 3.0)";
		break;
	    case Bool:
		type_string = "boolean";
		form_string = "(T F)";
		break;
	    case String:
		type_string = "string";
		form_string = "\"string\"";
		break;
	    case Segdes:
		type_string = "segment descriptor";
		form_string = "[1 2 3]";
		break;
	    default:
		fprintf(stderr, "vinterp: internal error: improper type %d in do_read().\n", type);
		vinterp_exit(1);
		break;
	}
        fprintf(stderr, "vinterp: type %s vector, in form %s: ", type_string, form_string);
    }

    /* figure out vector delimiters, and types to be read and stored */
    switch (type) {
	case Int: case Float: case Bool:
	    start_vector = '('; end_vector = ')';
	    stack_type = read_type = type;
	    break;
	case String: 
	    end_vector = start_vector = '"';
	    read_type = String;
	    stack_type = Int;
	    break;
	case Segdes:
	    start_vector = '['; end_vector = ']';
	    read_type = Int;
	    stack_type = Segdes;
	    break;
	default:
	    fprintf(stderr, "vinterp: internal error: improper type %d in do_read().\n", type);
	    vinterp_exit(1);
	    break;
    }

    /* skip whitespace */
    while ((c = getc(stream)) != EOF && isspace(c))
	;

    /* get start of vector */
    if (c != start_vector) {
	error_flag = 1;
	(void) sprintf(error_msg, "input error: vector must start with %c", start_vector);
	goto ERROR;
    }

    new_value_buffer = read_array;
    count = 0;

    /* read values until we're done */
    while (1) {
	/* check to see if room in buffer, reallocate if not */
	if (new_value_buffer >= read_array + input_size_max) {
	    input_size_max *= 2;
	    read_array = (int *)realloc((char*)read_array, 
		     	                input_size_max * sizeof(int));
	    if (read_array == NULL) {
	        fprintf(stderr, "vinterp: could not reallocate read buffer.\n");
	        vinterp_exit(1);
	    } else {
		switch (read_type) {
		    case Int: case Bool: case String:
			new_value_buffer = read_array + count;
			break;
		    case Float:
			new_value_buffer = (int*)(count + (double*)read_array);
			break;
		    default:
			fprintf(stderr, "vinterp: internal error: bad type in do_read()\n");
			vinterp_exit(1);
		}
	    }
	}

	/* do the input */
	new_value_buffer = get_value(new_value_buffer, read_type, stream);
	if (new_value_buffer == NULL)
	    break;
	count ++;
    }

    /* skip whitespace */
    while ((c = getc(stream)) != EOF && isspace(c))
	;

    /* get end of vector */
    if (c != end_vector) {
	error_flag = 1;
	(void) sprintf(error_msg, "input error: vector must end with %c", end_vector);
	/* things failed, put out error message and return */
ERROR:
	{
	    int i;
	    int len = strlen(error_msg);

	    stack_push(0, type, -1);	/* null result */

	    stack_push(len, Int, -1);	/* error message */
	    assert_mem_size(rep_vuz_scratch(1));
	    for (i = 0; i < len; i++) {		/* put string on stack */
		rep_vuz(se_vb(TOS)->vector, i, (int) error_msg[i], len, SCRATCH);
	    }

	    stack_push(1, Bool, -1);	/* error flag */
	    assert_mem_size(rep_vub_scratch(1));
	    rep_vub(se_vb(TOS)->vector, 0, (cvl_bool) 0, 1, SCRATCH);
	    return;
	}
    }

    stack_push(count, stack_type, -1);	/* allocate stack space */
    vector = se_vb(TOS)->vector;
    for (i=0; i < count; i++) {		/* copy values in, one by one ...  */
	switch (stack_type) {
	    case Int:
	    case Segdes: 	/* create the lengths vector for a segdes */
		assert_mem_size(rep_vuz_scratch(count));
		rep_vuz(vector, i, read_array[i], count, SCRATCH);
		break;
	    case Bool:
		assert_mem_size(rep_vub_scratch(count));
		rep_vub(vector, i, read_array[i], count, SCRATCH);
		break;
	    case Float:
		assert_mem_size(rep_vud_scratch(count));
		rep_vud(vector, i, *((double *)read_array +i), count, SCRATCH);
		break;
	    default: 
		fprintf(stderr, "vinterp: illegal type in READ.\n");
		vinterp_exit (1);
	}
    }
    if (type == Segdes) {
	/* do a MAKE_SEGDES from the length vector */
	int vector_len;
	stack_entry_t len_se = TOS;

	assert_mem_size(add_ruz_scratch(count));
	vector_len = add_ruz(se_vb(len_se)->vector, count, SCRATCH);

	stack_push(count, Segdes, vector_len);

	assert_mem_size(mke_fov_scratch(count, vector_len));
	mke_fov(se_vb(TOS)->vector, se_vb(len_se)->vector, 
		vector_len, count,  SCRATCH);

	stack_pop(len_se);		/* remove the length vector */
    }

    /* put results of operation on stack */
    stack_push(0, Int, -1);		/* error msg is null */
    stack_push(1, Bool, -1);		/* no errors */
    assert_mem_size(rep_vub_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, (cvl_bool) 1, 1, SCRATCH);
}

/* ----------------------- File I/O ------------------------*/

/* Some of these cray things might be needed for System V in general */

#ifdef cray
#include <sys/types.h>
#define MAXPATHLEN PATHSIZE
#endif

#ifdef CM2
#include <sys/types.h>
#endif

#include <sys/param.h>
#include <sys/file.h>
#include <errno.h>

extern char *sys_errlist[];

/* file_list: List of open files and their file descriptors.
 *	      The index of a FILE in file_list will be used by vcode
 *	      to refer to the file.
 *	      We initialize file_list so that stdin, stdout, and stderr
 *	      are already present.
 * NOFILE is max number open files for a process.
 */

static FILE *file_list[NOFILE] = {stdin, stdout, stderr, };   

/* Check that these were defined correctly in vcode.h */
/* Removed #error "*_FD defined incorrectly in vcode.h" -- Guy */
#if (STDIN_FD != 0 || STDOUT_FD != 1 || STDERR_FD !=2)
#if _STDC_
*_FD defined incorrectly in vcode.h
#else
*_FD defined incorrectly in vcode.h
#endif
#endif

/* add a file to file_list.  return index into list, -1 if error */
static int fd_insert(file)
FILE *file;
{
    int fd;

    for (fd = 0; (fd < NOFILE) && (file_list[fd] != (FILE *)NULL); fd ++)
	;

    if (fd < NOFILE) {
	/* everything ok */
	assert(file_list[fd] == (FILE *)NULL);
	file_list[fd] = file;
    } else {
	/* no room in file table */
	fd = -1;
    }

    return fd;
}

/* open a file.  Take two arguments off the stack: 
 *    filename - string (must convert CVL string to C string)
 *    mode -     integer describing how to open file: 
 *	1 - open for reading 
 *	2 - create for writing
 *	3 - append on write (write to end of file or create for writing)
 * Puts three values on the stack: 
 *    boolean -          T if open was successful, F is error
 *    error string -     character vector containing error message, empty if
 *                       none
 *    file descriptor -  an integer (index into file_list)
 *
 * We create the opened file so that it is readable and writable by all.
 */

void do_fopen(pe)
prog_entry_t *pe;
{
    vb_t *mode_vb = se_vb(TOS);			/* mode for opening file */
    int cvl_mode;
    char *open_mode;				/* type arg to fopen(2) */

    vb_t *filename_vb;				/* name of file to open */
    char filename[MAXPATHLEN];
    int filename_len;				/* length of filename vector */
    int len;					/* number chars read so far */

    int fd;					/* file descriptor */
    FILE *file;					/* stream returned by fopen */
    int error_flag = 0;				/* has error occurred? */
    char *error_msg;				/* error msg to return */
    int i;

    /* check type and length of mode */
    if (check_args && (mode_vb->type != Int || mode_vb->len != 1)) {
	fprintf(stderr, "vinterp: line %d: FOPEN: top of stack not integer scalar. \n", pe->lineno);
	vinterp_exit(1);
    }

    /* get CVL mode off of stack */
    assert_mem_size(ext_vuz_scratch(1));
    cvl_mode = (int) ext_vuz(mode_vb->vector, 0, 1, SCRATCH);

    /* convert to open_type */
    if (cvl_mode == 1) open_mode = "r";
    else if (cvl_mode == 2) open_mode = "w";
    else if (cvl_mode == 3) open_mode = "a";
    else {
	fprintf(stderr, "vinterp: line %d: FOPEN: top of stack (%d) not legal mode.\n", pe->lineno, cvl_mode);
	vinterp_exit(1);
    }

    /* remove mode from stack */
    stack_pop(TOS);

    /* now work with filename */
    filename_vb = se_vb(TOS);

    /* check type of filename (strings are stored as ints) */
    if (check_args && (mode_vb->type != Int)) {
	fprintf(stderr, "vinterp: line %d: FOPEN: second argument not string. \n", pe->lineno);
	vinterp_exit(1);
    }

    /* convert filename vector to C string */
    filename_len = filename_vb->len;
    assert_mem_size(ext_vuz_scratch(filename_len));
    for (len = 0; len < filename_len; len++) {
	filename[len] = (char) ext_vuz(filename_vb->vector, len, 
				       filename_len, SCRATCH);
    }
    filename[len] = '\0';		/* terminate string */

    stack_pop(TOS);			/* remove file name from stack */

    /* open file */
    file = fopen(filename, open_mode);

    if (file != (FILE *)NULL) {		/* open was successful */
	/* find unused slot in array */
	for (fd = 0; (fd < NOFILE) && (file_list[fd] != (FILE *)NULL); fd ++)
	    ;

	if (fd < NOFILE) {
	    /* everything ok */
	    assert(file_list[fd] == (FILE *)NULL);
	    file_list[fd] = file;
	    error_msg = "";
	} else {
	    /* no room in file table */
	    error_flag = 1;
	    error_msg = "no room in internal file table";
	    fd = -1;
	}
    } else { 		/* error opening file */
	error_msg = sys_errlist[errno];
	error_flag = 1;
	fd = -1;
    }

    /* put results on stack */
    stack_push(1, Int, -1);		/* file descriptor */
    assert_mem_size(rep_vuz_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, fd, 1, SCRATCH);

    len = strlen(error_msg);		/* error message */
    stack_push(len, Int, -1);
    assert_mem_size(rep_vuz_scratch(1));
    for (i = 0; i < len; i++) {		/* put string on stack */
	rep_vuz(se_vb(TOS)->vector, i, (int) error_msg[i], len, SCRATCH);
    }

    stack_push(1, Bool, -1);		/* negation of error_flag */
    assert_mem_size(rep_vub_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, (cvl_bool) !error_flag, 1, SCRATCH);
}

/* close a file: Takes a single argument, an integer file descriptor.
 * returns two values: 
 *    boolean - indicating success or failure
 *    string  - empty if successful, error message if failure
 */
void do_fclose(pe)
prog_entry_t *pe;
{
    vb_t *fd_vb = se_vb(TOS);		/* get fd from stack */
    int fd;
    FILE *file;

    int error_flag = 0;	
    char *error_msg = "";

    int i, len;

    /* check for integer scalar value */
    if (check_args && (fd_vb->type != Int || fd_vb->len != 1)) {
	fprintf(stderr, "vinterp: line %d: FCLOSE: top of stack not integer scalar. \n", pe->lineno);
	vinterp_exit(1);
    }

    /* get fd off of stack */
    assert_mem_size(ext_vuz_scratch(1));
    fd = (int) ext_vuz(fd_vb->vector, 0, 1, SCRATCH);

    /* remove fd from stack */
    stack_pop(TOS);			/* remove file name from stack */

    file = file_list[fd];		/* get stream from list */

    if (file != (FILE *)NULL) {		/* make sure fd is valid */
	/* call fclose */
	if (fclose(file) != EOF) {
	    /* success */
	    error_flag = 0;
	    error_msg = "";
	    file_list[fd] = (FILE *)NULL;	/* open up slot */
	} else {
	    /* error */
	    error_flag = 1;
	    error_msg = "could not close file";
	}
    } else {			/* invalid fd */
	error_flag = 1;
	error_msg = "illegal VCODE file descriptor in FCLOSE";
    }

    /* put results of operation on stack */
    len = strlen(error_msg);		/* error message */
    stack_push(len, Int, -1);
    assert_mem_size(rep_vuz_scratch(1));
    for (i = 0; i < len; i++) {		/* put string on stack */
	rep_vuz(se_vb(TOS)->vector, i, (int) error_msg[i], len, SCRATCH);
    }

    stack_push(1, Bool, -1);		/* negation of error_flag */
    assert_mem_size(rep_vub_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, (cvl_bool) !error_flag, 1, SCRATCH);
}


/* Read from a file. File descriptor is on top of stack.
 * Returns three values on the stack:
 *   bool - T if read is successful
 *   string - error message if failure in read, null otherwise
 *   vector - result of the read
 * Call do_read() to do the actual read.
 */   
void do_fread(pe)
prog_entry_t *pe;
{
    
    vb_t *fd_vb = se_vb(TOS);		/* get fd from stack */
    int fd;
    FILE *file;				/* from file_list */

    /* check for integer scalar value */
    if (check_args && (fd_vb->type != Int || fd_vb->len != 1)) {
	fprintf(stderr, "vinterp: line %d: FREAD: top of stack not integer scalar. \n", pe->lineno);
	vinterp_exit(1);
    }

    /* get fd off of stack */
    assert_mem_size(ext_vuz_scratch(1));
    fd = (int) ext_vuz(fd_vb->vector, 0, 1, SCRATCH);

    stack_pop(TOS);			/* remove fd from stack */

    file = file_list[fd];		/* get actual file handler */

    if (file == NULL) {			/* check for validity */
	/* invalid: write results to stack */
	char *fread_error = "illegal VCODE file descriptor in FREAD";
	int i;
	int len = strlen(fread_error);

	stack_push(0, pe->type, -1);	/* null result */

	stack_push(len, Int, -1);	/* error message */
	assert_mem_size(rep_vuz_scratch(1));
	for (i = 0; i < len; i++) {		/* put string on stack */
	    rep_vuz(se_vb(TOS)->vector, i, (int) fread_error[i], len, SCRATCH);
	}

	stack_push(1, Bool, -1);	/* error flag */
	assert_mem_size(rep_vub_scratch(1));
	rep_vub(se_vb(TOS)->vector, 0, (cvl_bool) 0, 1, SCRATCH);
    } else {
	/* do_read does all the work */
        do_read(pe, file);
    }
}


/* Write to a file. File descriptor is on top of stack.
 * Returns two values on the stack:
 *   bool - T if read is successful
 *   string - error message if failure in read, null otherwise
 * Call do_write() to do the actual write
 */   
void do_fwrite(pe)
prog_entry_t *pe;
{
    
    vb_t *fd_vb = se_vb(TOS);		/* get fd from stack */
    int fd;
    FILE *file;				/* from file_list */

    /* check for integer scalar value */
    if (check_args && (fd_vb->type != Int || fd_vb->len != 1)) {
	fprintf(stderr, "vinterp: line %d: FREAD: top of stack not integer scalar. \n", pe->lineno);
	vinterp_exit(1);
    }

    /* get fd off of stack */
    assert_mem_size(ext_vuz_scratch(1));
    fd = (int) ext_vuz(fd_vb->vector, 0, 1, SCRATCH);

    stack_pop(TOS);			/* remove fd from stack */

    file = file_list[fd];		/* get actual file handler */

    if (file == NULL) {			/* check for validity */
	/* invalid: write results to stack */
	char *fwrite_error = "illegal VCODE file descriptor in FWRITE";
	int i;
	int len = strlen(fwrite_error);
	
	/* remove write vector from stack */
	stack_pop(TOS);

	stack_push(len, Int, -1);	/* error message */
	assert_mem_size(rep_vuz_scratch(1));
	for (i = 0; i < len; i++) {		/* put string on stack */
	    rep_vuz(se_vb(TOS)->vector, i, (int) fwrite_error[i], len, SCRATCH);
	}

	stack_push(1, Bool, -1);	/* error flag */
	assert_mem_size(rep_vub_scratch(1));
	rep_vub(se_vb(TOS)->vector, 0, (cvl_bool) 0, 1, SCRATCH);
    } else {
	/* should do some sort of check to see whether or not the file 
	 * was opened to write */

	/* do_write does all the work */
	do_write(pe, file);
    }
}

/* get the top of stack and turn it into an fd.
 * Return an error message if problem, NULL if ok.
 */
char *fd_from_stack(pe, pfd)
prog_entry_t *pe;
int *pfd;
{
    vb_t *fd_vb;				/* file descriptor vector */
    int fd;					/* int form of fd */
    char *error_msg = NULL;			/* error msg to return */

    fd_vb = se_vb(TOS);

    /* check type and length */
    if (check_args && (fd_vb->type != Int) && (fd_vb->len != 1) ) {
	fprintf(stderr, "vinterp: line %d: %s: top of stack not legal file descriptor.\n",
	       pe->lineno, pe->vopdes->vopname);
	vinterp_exit(1);
    }

    /* get value */
    assert_mem_size(ext_vuz_scratch(1));
    fd = (int) ext_vuz(fd_vb->vector, 0, 1, SCRATCH);

    /* check for validity */
    if ((fd != NULL_STREAM_FD) &&
	(fd < 0 || fd >= NOFILE || file_list[fd] == NULL)) {
	/* invalid: write results to stack */
	error_msg = "illegal VCODE file descriptor";
    }

    stack_pop(TOS);		/* remove from stack */

    *pfd = fd;
    return error_msg;
}

/* do_spawn() : This allows VCODE to communicate with other C processes.
 * The SPAWN command takes 4 args from the stack:
 *    execution string - a string that will be passed to execvp
 *    in_fd -  a file descriptor -  stdin of new process
 *    out_fd - a file descriptor -  stdout of new process
 *    err_fd - a file descriptor -  stderr of new process
 * Any of these arguments may be the NULL_STREAM_FD, in which case,
 * new streams are created.
 * The command returns three file descriptors, a boolean status flag 
 * and an error message.
 */
void do_spawn(pe)
prog_entry_t *pe;
{
    vb_t *command_vb;				/* command to run (vector) */
    char *command;				/* command to run (string) */
    int command_len;				/* length of command vector */
    int len, i;					/* temporary */

    int in_fd, out_fd, err_fd;			/* int form of fds */

    int error_flag = 0;				/* has error occurred? */
    char *error_msg = "";			/* error msg to return */
    char *err_msg, *out_msg, *in_msg;		/* returns of fd_from_stack */

    int child_in_fd[2], child_out_fd[2], child_err_fd[2];  /* for pipe */

    int isParent;				/* return of fork() */

    /* get the various fd from the stack */
    err_msg = fd_from_stack(pe, &err_fd);  error_flag |= (err_msg != NULL);
    out_msg = fd_from_stack(pe, &out_fd);  error_flag |= (out_msg != NULL);
    in_msg = fd_from_stack(pe, &in_fd);	   error_flag |= (in_msg != NULL);

    if (error_flag) {
	error_msg = "illegal Vcode file descriptor";
	goto ERROR;
    }

    /* now get the command */
    command_vb = se_vb(TOS);
    if (check_args && (command_vb->type != Int)) {
	fprintf(stderr, "vinterp: line %d: SPAWN: first arg not string.\n", pe->lineno);
	vinterp_exit(1);
    }

    /* convert command vector to C string */
    command_len = command_vb->len;
    command = (char *)malloc((command_len+1) * sizeof(char));
    if (command == NULL) {
	fprintf(stderr, "vinterp: line %d: internal error: couldn't allocate command string (len = %d) for SPAWN\n", pe->lineno, command_len);
	vinterp_exit(1);
    }
    assert_mem_size(ext_vuz_scratch(command_len));
    for (len = 0; len < command_len; len++) {
	command[len] = (char) ext_vuz(command_vb->vector, len, 
				       command_len, SCRATCH);
    }
    command[len] = '\0';		/* terminate string */

    stack_pop(TOS);			/* remove command name from stack */

    /* Now we can do the spawning */
    
    /* For each NULL_STREAM argument, we must set up a pipe
     * so that the two processes can communicate.
     */
    if (in_fd == NULL_STREAM_FD)  pipe(child_in_fd);
    if (out_fd == NULL_STREAM_FD) pipe(child_out_fd);
    if (err_fd == NULL_STREAM_FD) pipe(child_err_fd);
    
    /* Second, fork off a subprocess */
    isParent = fork();

    if (isParent) { 	/* parent process will be the interpreter */
	/* if NULL_STREAM_FD was used, get the corect end of the pipe,
	 * convert the fd into a FILE *, put it in the file_list,
	 * and close other end.
	 */
	FILE * file = stdin;		/* initialized to non-NULL */

	if (in_fd == NULL_STREAM_FD) {
	    file = fdopen(child_in_fd[1], "w");  /* should check for error */
	    in_fd = fd_insert(file);
	    close(child_in_fd[0]);
	}
	if (out_fd == NULL_STREAM_FD) {
	    file = fdopen(child_out_fd[0], "r");  /* should check for error */
	    out_fd = fd_insert(file);
	    close(child_out_fd[1]);
	}
	if (err_fd == NULL_STREAM_FD) {
	    file = fdopen(child_err_fd[0], "r");  /* should check for error */
	    err_fd = fd_insert(file);
	    close(child_err_fd[1]);
	}

	if (in_fd == -1 || out_fd == -1 || err_fd == -1) {
	    error_msg = "ran out of VCODE descriptors";
	    error_flag = 1;
	} else if (file == NULL) {
	    error_msg = "could not perform fdopen() during spawn";
	    error_flag = 1;
	}
    } else {		/* child will exec the command */
	int argc;	/* arg count for subprocess */
	char **argv;
	int i;
	int inspace;	/* for command parser */

	/* must break the command string up into pieces */
	argv = (char **) malloc (sizeof (char *) * command_len);
	argc = 0;
	inspace = 1;	/* start on space to catch beginning */
	for (i = 0; i < command_len; i++) {
	    if (isspace(command[i])) {
		command[i] = NULL;
		inspace = 1;
	    } else if (inspace) {
		/* found new word: went from space to non-space */
		argv[argc++] = command + i;
		inspace = 0;
	    }
	}
	argv[argc] = NULL;

	/* Set up IO strams for child.  If NULL_STREAM was specified, 
	 * make correct end of pipe into appropriate stream, and close other
	 * end.  Otherwise, use supplied stream. 
	 */
	if (in_fd == NULL_STREAM_FD) {
	    dup2(child_in_fd[0], fileno(stdin));
	    close(child_in_fd[1]);
	} else 
	    dup2(fileno(file_list[in_fd]), fileno(stdin));

	if (out_fd == NULL_STREAM_FD) {
	    dup2(child_out_fd[1], fileno(stdout));
	    close(child_out_fd[2]);
	} else 
	    dup2(fileno(file_list[out_fd]), fileno(stdout));

	if (err_fd == NULL_STREAM_FD) {
	    dup2(child_err_fd[1], fileno(stderr));
	    close(child_err_fd[2]);
	} else 
	    dup2(fileno(file_list[err_fd]), fileno(stderr));


	/* use execvp (search through PATH) to start co-process */
	execvp(*argv, argv);

	/* reach here only on error */
	fprintf(stderr, "vinterp: line %d internal error: execvp failed:\n",
		 pe->lineno);
	fprintf(stderr, "\t%s\n", sys_errlist[errno]);
	fprintf(stderr, "\tcommand was %s\n", command);

	vinterp_exit(1);
    }

ERROR:
    /* put results on stack */
    stack_push(1, Int, -1);		/* err_fd */
    assert_mem_size(rep_vuz_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, err_fd, 1, SCRATCH);

    stack_push(1, Int, -1);		/* out_fd */
    assert_mem_size(rep_vuz_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, out_fd, 1, SCRATCH);

    stack_push(1, Int, -1);		/* in_fd */
    assert_mem_size(rep_vuz_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, in_fd, 1, SCRATCH);

    len = strlen(error_msg);		/* error message */
    stack_push(len, Int, -1);
    assert_mem_size(rep_vuz_scratch(1));
    for (i = 0; i < len; i++) {		/* put string on stack */
	rep_vuz(se_vb(TOS)->vector, i, (int) error_msg[i], len, SCRATCH);
    }

    stack_push(1, Bool, -1);		/* negation of error_flag */
    assert_mem_size(rep_vub_scratch(1));
    rep_vuz(se_vb(TOS)->vector, 0, (cvl_bool) !error_flag, 1, SCRATCH);
}

