;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; MUTABLE TYPE VARIABLES FOR USE IN TYPE CHECKING
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct var pointer)

(defun make-variable (type)
  (make-var :pointer type))

(defun is-variable? (v) (var-p v))

(defun is-bound-variable? (v definitions)
  (and (var-p v) (not (variable-type? (var-pointer v) definitions))))

(defun is-unbound-variable? (v definitions)
  (and (var-p v) (variable-type? (var-pointer v) definitions)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SETTING MUTABLE TYPE VARIABLES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun check-inclusion-list (var typelist definitions)
  (if typelist
      (or (check-inclusion var (car typelist) definitions)
	  (check-inclusion-list var (cdr typelist) definitions))
    nil))

(defun check-inclusion (var type definitions)
  (cond ((is-bound-variable? type definitions)
	 (check-inclusion var (var-pointer type) definitions))
	((is-variable? type)
	 (eq var type))
	((atom type) nil)
	(t (check-inclusion-list var (cdr type) definitions))))

(defun set-variable (val newval definitions)
  ;;(format t "val: ~a~%newval: ~a~%~%" val newval)
  (let ((valtype (var-pointer val))
	(newvaltype (if (is-variable? newval)
			(var-pointer newval)
		      newval)))
    (or (check-inclusion val newval definitions)
	(if (type-< newvaltype valtype definitions)
	    (progn
	      (setf (var-pointer val) newval)
	      nil)
	  (if (type-< valtype newvaltype definitions)
	      (progn
		(setf (var-pointer val) newval)
		(setf (var-pointer newval) valtype)
		nil)
	    :error1)))))

;;;;;;;;;;;;;;;;;;;;;
;;; STRIPPING OFF THE MUTABLE TYPE VARIABLES FROM A TYPE EXPRESSION
;;;;;;;;;;;;;;;;;;;;;

;;; Used after the completion of type checking.

;;(defparameter *typevarlist* 
;;  '(nesl::alpha nesl::beta nesl::gamma nesl::delta
;;    nesl::epsilon nesl::rho nesl::nu nesl::omega))

(defparameter *typevarlist* 
  '(nesl::a nesl::b nesl::c nesl::d nesl::e))

(defun typevarnames (n)
  (if (> n (length *typevarlist*))
      (do* ((i 0 (+ i 1))
	   (name (intern (format nil "A~d" i) 'nesl) 
		 (intern (format nil "A~d" i) 'nesl))
	   (l (list name) (cons name l)))
	  ((= i n) (reverse l)))
    (subseq *typevarlist* 0 n)))

(defun clean-type-list (type definitions)
  (if type
      (let ((thead (clean-type-r (car type) definitions))
	    (ttail (clean-type-list (cdr type) definitions)))
	(cons (cons (car thead) (car ttail))
	      (union (cdr thead) (cdr ttail))))
    nil))

(defun clean-type-r (type definitions)
  (cond ((is-unbound-variable? type definitions)
	 (cons type (list type)))
	((is-variable? type)
	 (clean-type-r (var-pointer type) definitions))
	((atom type)
	 (cons type nil))
	(t
	 (clean-type-list type definitions))))

(defun clean-type (type definitions)
  (let* ((ctype (clean-type-r type definitions))
	 (class (mapcar #'var-pointer (cdr ctype)))
	 (new-vars (typevarnames (length (cdr ctype)))))
    (do* ((ftype (car ctype) (subst (car nvars) (car vars) ftype))
	  (vars (cdr ctype) (cdr vars))
	  (nvars new-vars (cdr nvars)))
	 ((not vars) (cons ftype (mapcar #'list new-vars class))))))

(defun print-type (type definitions)
  (let ((clean-type (clean-type type definitions)))
    (pretty-type-full clean-type)))

(defun print-type-list (type definitions)
  (let ((clean-type (clean-type type definitions)))
    (pretty-type-list-full clean-type)))

;;;;;;;;;;;;;;;;;;;;;
;;; UNIFICATION
;;;;;;;;;;;;;;;;;;;;;

(defun unify-list (type1 type2 definitions)
  (if type1
      (if type2
	  (or (unify (car type1) (car type2) definitions)
	      (unify-list (cdr type1) (cdr type2) definitions))
	:error2)
    (if type2 :error2 nil)))

(defun unify (type1 type2 definitions)
  ;;(format t "type1: ~a~%type2: ~a~%~%" type1 type2)
  (cond ((eql type1 type2) nil)
	((is-bound-variable? type1 definitions)
	 (unify (var-pointer type1) type2 definitions))
	((is-bound-variable? type2 definitions)
	 (unify type1 (var-pointer type2) definitions))
	((is-variable? type1)
	 (set-variable type1 type2 definitions))
	((is-variable? type2)
	 (set-variable type2 type1 definitions))
	((or (atom type1) (atom type2))
	 :error3)
	;; In this final case both types are lists (records)
	;; and the first element must be atomic and equal
	(t (if (and (atom (car type1)) (atom (car type2))
		    (eql (car type1) (car type2)))
	       (unify-list (cdr type1) (cdr type2) definitions)
	     :error4))))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK A FUNCTION CALL
;;;;;;;;;;;;;;;;;;;;;;;;;

(defun instantiate-fun-type (type)
  (do ((vars (cdr type) (cdr vars))
       (spec (car type)))
      ((not vars) spec)
      (setq spec (subst (make-variable (second (car vars)))
			       (first (car vars))
			       spec))))

(defun typecheck-list (list type-env)
  (if (null list) nil
    (cons (typecheck-exp (car list) type-env)
	  (typecheck-list (cdr list) type-env))))

(defun typecheck-function (expression type-env)
  (let* ((definitions (second type-env))
	 (arg-types (typecheck-list (cdr expression) type-env))
	 (fundef (get-fundef (car expression) definitions))
	 (function-types
	  (cond ((eql (car (third type-env)) (car expression))
		 (cdr (third type-env)))
		(fundef
		 (instantiate-fun-type (fundef-type fundef)))
		(t (nesl-error "Function ~a is undefined." 
			       (car expression))))))
    (when (unify-list arg-types (cdr function-types) definitions)
      (let ((function-types
	     (if (eql (car (third type-env)) (car expression))
		 (cdr (third type-env))
	       (instantiate-fun-type (fundef-type fundef)))))
	(nesl-error "For function call ~a,~%~
               inferred argument types don't match function specification.~%  ~
               Argument types: ~a~%  ~
               Function types: ~a~%"
		    expression 
		    (print-type-list arg-types definitions)
		    (print-type-list (cdr function-types) definitions))))
    (car function-types)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; FIND SPECIFIC TYPES
;;;;;;;;;;;;;;;;;;;;;;;;;

;; These functions are used by trans.lisp in the final generation of code.

(defun get-instantiated-function-type (fundef arg-types definitions) 
  (let* ((function-types (instantiate-fun-type (fundef-type fundef))))
    (when (unify-list arg-types (cdr function-types) definitions)
	  (nesl-error "INTERNAL ERROR in function call ~a,~%~
               inferred argument types don't match function specification.~%  ~
               Argument types: ~a~%  ~
               Function spec:  ~a~%"
		      (car (fundef-names fundef))
		      (print-type-list arg-types definitions)
		      (print-type-list (cdr function-types) definitions)))
    (car (clean-type-r function-types definitions))))

;;; used by base-typecase to determine the type of a typevar
(defun foo-type (type var argtypes definitions)
  (let ((inst (instantiate-fun-type (cons (cons var (car type)) (cdr type)))))
    (unify-list argtypes (cddr inst) definitions)
    (car (clean-type (car inst) definitions))))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK PATTERNS 
;;;;;;;;;;;;;;;;;;;;;;;

;;; Used in pattern matching with a WITH, OVER or DEFOP.

(defun check-valid-varname (name definitions)
  (when (nondefinable-constant? name definitions)
    (nesl-error "The symbol ~a is a constant that cannot be redefined."
		name))
  (when (not (symbolp name))
    (nesl-error "The value ~s is not a valid variable name." name)))

(defun bind-typecheck-list (list environment)
  (if list
      (let ((first-match (bind-typecheck-exp (car list) environment))
	    (rest-match (bind-typecheck-list (cdr list) environment)))
	(cons (append (car first-match) (car rest-match))
	      (cons (cdr first-match) (cdr rest-match))))
    (cons nil nil)))

(defun bind-typecheck-rec (expression environment)
  (let* ((checked-list (bind-typecheck-list (cdr expression) environment))
	 (arg-types (cdr checked-list))
	 (typedef (get-typedef (car expression) environment))
	 (function-types
	  (if typedef (instantiate-fun-type (typedef-type typedef))
	    (nesl-error "Record-type ~a undefined." (car expression)))))
    ;;(format t "~%Arg types for ~a~%  ~a~%" expression (print-type arg-types))
    ;;(format t "~%Record types before:~%  ~a~%" (print-type function-types))
    (when (unify-list arg-types (cdr function-types) environment)
      (nesl-error "In pattern match ~a." expression))
    (cons (car checked-list) (car function-types))))

(defun bind-typecheck-exp (expression environment)
  (if (atom expression) 
      (let ((variable (make-variable 'nesl::any)))
	(check-valid-varname expression environment)
	(cons (list (cons expression variable))
	      variable))
    (bind-typecheck-rec expression environment)))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK A "with" FORM
;;;;;;;;;;;;;;;;;;;;;;;

(defun typecheck-with (expression type-env)
  (when (or (not (= (length expression) 3))
	    (not (listp (second expression))))
    (nesl-error 
     "In the WITH expression ~%  ~a.~%~
      The syntax for WITH is:  (WITH (expbind*) exp)."
     expression))
  (typecheck-exp (third expression) 
		 (cons (typecheck-bindings nil (second expression) type-env)
		       (cdr type-env))))

(defun typecheck-binding (over? binding type-env)
  (when (or (not (listp binding))
	    (not (= (length binding) 2)))
    (nesl-error 
     "In the binding ~a of a ~a form.~%~
            The syntax for a binding is:  (pattern exp)." 
     binding (if over? "V. or OVER" "WITH")))
  (let* ((exp-type (typecheck-exp (second binding) type-env))
	 (var-type (bind-typecheck-exp (car binding) (second type-env)))
	 (match-type (if over? (list 'nesl::vector (cdr var-type))
		       (cdr var-type)))) 
    (when (unify match-type exp-type (second type-env))
      (nesl-error 
       "In the binding ~a of a ~a form.~%~
            The inferred type on the right hand side~%  ~a~%~
            does not match the pattern on the left."
       binding (if over? "V. or OVER" "WITH") 
       (print-type exp-type (second type-env))))
    (car var-type)))

(defun typecheck-bindings (over? bindings type-env)
  (if bindings
      (let ((bind-vars (typecheck-binding over? (car bindings) type-env)))
	(typecheck-bindings over? (cdr bindings) 
			    (cons (append bind-vars (car type-env))
				  (cdr type-env))))
    (car type-env)))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK AN "over" FORM
;;;;;;;;;;;;;;;;;;;;;;;

(defun typecheck-over (expression type-env)
  (when (or (not (= (length expression) 3))
	    (not (listp (second expression)))
	    (not (> (length (second expression)) 0)))
    (nesl-error 
     "In the OVER expression ~%  ~a.~%~
      The syntax for OVER is:  (OVER (expbind+) exp)."
     expression))
  (list 'nesl::vector
	(typecheck-exp 
	 (third expression) 
	 (cons (typecheck-bindings t (second expression) type-env)
	       (cdr type-env)))))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK AN "if" FORM
;;;;;;;;;;;;;;;;;;;;;;;

(defun typecheck-if (expression type-env)
  (let* ((arg-types (typecheck-list (cdr expression) type-env)))
    (when (not (= (length expression) 4))
      (nesl-error 
       "Error in the IF expression ~%  ~a.~%~
        The syntax for IF is:  (IF exp exp exp)."
       expression))
    (when (unify (car arg-types) 'bool (second type-env))
      (nesl-error 
       "In the IF expression~%  ~a~%~
        the first argument must be of type BOOL.  The inferred type is ~a."
       expression (car arg-types)))
    (when (unify (second arg-types) (third arg-types) (second type-env))
      (nesl-error 
       "In the IF expression~%  ~a~%~
        the two branches must be of the same type.  The inferred types are~%  ~
        IF type:   ~a~%  ELSE type: ~a."
       expression 
       (print-type (second arg-types) (second type-env)) 
       (print-type (third arg-types) (second type-env))))
    (second arg-types)))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK AN EXPRESSION
;;;;;;;;;;;;;;;;;;;;;;;

(defun typecheck-exp (expression type-env)
  (cond ((nesl-constant-p expression))
	((symbolp expression)
	 (or (cdr (assoc expression (car type-env)))
	     (let ((val-type (get-variable expression (second type-env))))
	       (if val-type (cdr val-type) nil))
	     (nesl-error "Variable ~a is undefined." expression)))
	((listp expression)
	 (let ((expression (convert-vexp expression)))
	   (cond ((eql (car expression) 'with)
		  (typecheck-with expression type-env))
		 ((eql (car expression) 'over)
		  (typecheck-over expression type-env))
		 ((eql (car expression) 'if)
		  (typecheck-if expression type-env))
		 (t
		  (typecheck-function expression type-env)))))
	(t (nesl-error "Invalid expression, ~s." expression))))

;;;;;;;;;;;;;;;;;;;;;;;
;;; TYPE CHECK A FUNCTION (POTENTIALLY RECURSIVE)
;;;;;;;;;;;;;;;;;;;;;;;

(defun typecheck-op (names types code definitions)
  (let* ((nvars (bind-typecheck-list (cdr names) definitions))
	 (source-types (cdr nvars))
	 (full-type (cons (make-variable 'nesl::any) source-types))
	 (type-check (when (and types (unify-list full-type 
						  (instantiate-fun-type types)
						  definitions))
		       (nesl-error "Supplied type does not match interface.")))
	 (var-bindings (car nvars))		
	 (result-type 
	  (typecheck-exp code (list var-bindings
				    definitions
				    (cons (car names) full-type)))))
    (when (unify result-type (car full-type) definitions)
      (nesl-error "Inferred result type~% ~a~%~
                   does not match either a user supplied result type,~%~
                   or the result type in a recursive call:~% ~a."
		  (print-type result-type definitions)
		  (print-type (car full-type) definitions)))
    (clean-type full-type definitions)))

