;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;; Returns all the variables in a binding expression.
;; For example (bindvars-exp '(pair (pair x y) z))
;; will return (x y z).
(defun bindvars-exp (varlist)
  (if (atom varlist)
      (list varlist)
    (mapcan 'bindvars-exp (cdr varlist))))

(defun ext-bindings (bindings env)
  (if (null bindings) nil
    (append (ext-exp (second (car bindings)) env)
	    (ext-bindings (cdr bindings) 
			  (append (bindvars-exp (first (car bindings))) 
				  env)))))

(defun ext-with (exp env)
  (append (ext-bindings (second exp) env)
	  (ext-exp (third exp) 
		   (append (mapcan 
			    #'(lambda (x) (bindvars-exp (first x))) 
			    (second exp)) env))))

(defun ext-func (exp env)
  (ext-list (cdr exp) env))

(defun ext-symbol (exp env)
  (if (not (find exp env)) (list exp) nil))

(defun ext-over (exp env)
  (ext-with exp env))

(defun ext-if (exp env)
  (ext-list (cdr exp) env))

(defun ext-list (list env)
  (if (null list) nil
    (append (ext-exp (car list) env)
	    (ext-list (cdr list) env))))

(defun ext-exp (exp env)
  (cond ((nesl-constant-p exp) nil)
	((symbolp exp)
	 (ext-symbol exp env))
	((listp exp)
	 (let ((nexp (convert-vexp exp)))
	   (cond ((eql (car nexp) 'if)
		  (ext-if nexp env))
		 ((eql (car nexp) 'with)
		  (ext-with nexp env))
		 ((eql (car nexp) 'over)
		  (ext-with nexp env))
		 (t (ext-func nexp env)))))
	(t (nesl-error "Invalid expression, ~s." exp))))

(defun vprefixp (symbol)
  (let ((sstring (string symbol)))
    (and (>= (length sstring) 2)
	 (string-equal "v." symbol :end2 2))))

(defun add-vprefix (symbol)
  (intern (format nil "V.~a" symbol) (symbol-package symbol)))

(defun vpostfix (symbol)
  (let ((value (read-from-string (subseq (string symbol) 2))))
    (cond ((numberp value) value)
          ((symbolp value) (intern (string value) (symbol-package symbol)))
          (t (nesl-error "~s is an invalid form" symbol)))))

(defun convert-vexpt-list (list)
  (cond ((null list) (cons nil nil))
	((eql (car list) 'v.)
	 (let ((rest (convert-vexpt-list (cddr list))))
	   (cons (car rest) (cons (cadr list) (cdr rest)))))
	(t
	 (let ((rest (convert-vexpt-list (cdr list)))
	       (this (if (and (symbolp (car list)) (vprefixp (car list)))
			 (cons nil (vpostfix (car list)))
		       (let ((sym (gensym "ARG-")))
			 (cons (list (list sym (car list))) sym)))))	
	   (cons (append (car this) (car rest))
		 (cons (cdr this) (cdr rest)))))))

(defun convert-vexp (exp)
  (if (and (symbolp (car exp)) (vprefixp (car exp)))
      (let ((conv (convert-vexpt-list (cdr exp)))
	    (op (vpostfix (car exp))))
	(when (not (car conv))
	  (nesl-error "In expression ~s,~%~
             a v. expression must have at least one argument which is not~%~
             preceeded with a v." exp))
	`(over ,(car conv)
	       ,(cons op (cdr conv))))
    exp))

(defun conv-over (exp pflag env)
  (let* ((bindings (second exp))
	 (body (conv-exp (third exp) t env))
	 (free-body (remove-duplicates
		     (ext-exp (third exp) 
			      (mapcan #'(lambda (binding)
					  (bindvars-exp (first binding)))
				      bindings)))))
    (when (zerop (length bindings))
      (nesl-error 
       "In an over expression, there must be at least one binding.~%~s"
       exp))
     `(with (((nesl::vector segdes ,(first (first bindings)))
	      ,(conv-exp (second (first bindings)) pflag env))
	     ,@(mapcar #'(lambda (a)
			   `((nesl::vector segdes ,(first a))
			     ,(conv-exp (second a) pflag env)))
		       (cdr bindings))
	     ,@(mapcar (if pflag
			   #'(lambda (a) 
			       `(,a (neslp::prim-dist ,a segdes seg_len)))
			 #'(lambda (a) `(,a (nesl::prim-dist ,a segdes))))
		       free-body)
	     (seg_len 
	      ,(if pflag 
		   '(nesl::make-segdes 
		     (nesl::prim-+-reduce (nesl::len segdes) seg_len))
		 'segdes)))
	 (nesl::vector segdes ,body))))

(defun conv-bindings (bindings pflag env)
  (cond ((null bindings) nil)
	(t (cons (list (first (car bindings))
		       (conv-exp (second (car bindings)) pflag env))
		 (conv-bindings (cdr bindings) pflag env)))))

(defun conv-with (expression pflag env)
  `(with ,(conv-bindings (second expression) pflag env)
     ,(conv-exp (third expression) pflag env)))

;; If we have a constant expression (no free variables), and pflag is true,
;; then we have to distribute the expression across all parallel 
;; occurences.
(defun conv-constant (constant pflag env)
  (declare (ignore pflag))
  `(nesl::prim-dist ,(conv-exp constant nil env) seg_len))
  
(defun conv-list (list pflag env)
  (cond ((null list) nil)
	(t (cons (conv-exp (car list) pflag env)
		 (conv-list (cdr list) pflag env)))))

(defun pname (name)
  (if (not (symbolp name))
      (nesl-error "~a is an invalid function name." name)
    (intern (string name) 'neslp)))

(defun add-parg (exp pflag)
  (if pflag (append exp (list 'seg_len)) exp))

(defun conv-func (expression pflag env)
  (let ((funname (if pflag (pname (car expression)) (car expression)))
	(args (conv-list (cdr expression) pflag env)))
    (add-parg (cons funname args) pflag)))

(defun do-thing (a)
  `((nesl::vector junk-segdes ,a) 
    (nesl::pack (nesl::vector seg_len ,a) vcond)))

(defun conv-if (expression pflag env)
  (if (not pflag)
      `(if ,@(conv-list (cdr expression) pflag env))
    (let ((cond (conv-exp (second expression) pflag env))
	  (then (conv-exp (third expression) pflag env))
	  (else (conv-exp (fourth expression) pflag env))
	  (then-vars (remove-duplicates (ext-exp (third expression) nil)))
	  (else-vars (remove-duplicates (ext-exp (fourth expression) nil))))
      `(with ((cond ,cond)
	      (vcond (nesl::vector seg_len cond))
	      (total-count (nesl::len seg_len))
	      (true-count (nesl::count vcond))
	      (false-count (nesl::- total-count true-count)))
         (if (nesl::= true-count total-count)
	     ,then
	   (if (nesl::= false-count total-count)
	       ,else
	     (with (((nesl::vector if-segdes if-values)
		     (nesl::flag-merge 
		      vcond
		      (with ((vcond (nesl::vector seg_len (nesl::not cond)))
			     ,@(mapcar #'do-thing else-vars)
			     (seg_len (nesl::make-segdes false-count)))
			(nesl::vector seg_len ,else))
		      (with (,@(mapcar #'do-thing then-vars)
			       (seg_len (nesl::make-segdes true-count)))
			(nesl::vector seg_len ,then)))))
	       if-values)))))))

(defun conv-exp (expression pflag env)
  (cond (;; this is true if there are no free variables in the body
	 (and pflag (not (ext-exp expression nil)))
	 (conv-constant expression pflag env))
	((nesl-constant-p expression) 
	 (coerce-nesl-constant expression))
	((symbolp expression) expression)
	((listp expression)
	 (let ((nexp (convert-vexp expression)))
	   (cond ((eql (car nexp) 'if)
		  (conv-if nexp pflag env))
		 ((eql (car nexp) 'nesl::nif)
		  (nesl-error "NIF is outdated, please use IF."))
		 ((eql (car nexp) 'nesl::sif)
		  `(if ,@(conv-list (cdr nexp) pflag env)))
		 ((eql (car nexp) 'with)
		  (conv-with nexp pflag env))
		 ((eql (car nexp) 'over)
		  (conv-over nexp pflag env))
		 ((eql (car nexp) 'nesl::time)
		  `(nesl::time ,(conv-exp (second nexp) pflag env)))
		 (t (conv-func nexp pflag env)))))
	(t (nesl-error "Invalid expression, ~s." expression))))

(defun conv-body (variables body pflag)
  (if (not pflag)
      (conv-exp body nil nil)
    ;; If the body is parallel, then we have to make sure that 
    ;; the free variables get distributed over the current context.
    (let* ((flat-variables (mapcan #'bindvars-exp variables))
	   (free-variables (remove-duplicates 
			    (ext-exp body flat-variables)))
	   (converted-body (conv-exp body t nil)))
      `(with (,@(mapcar #'(lambda (a) `(,a (nesl::prim-dist ,a seg_len)))
			free-variables))
	 ,converted-body))))

(defun conv-type (type pflag)
  (if pflag
      (append type (list 'nesl::segdes))
    type))

(defun conv-types (types pflag)
  (mapcar #'(lambda (a) (conv-type a pflag)) types))

(defun conv-names (names pflag)
  (if pflag
      (append (cons (pname (car names)) (cdr names)) (list 'seg_len))
    names))
