;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;; Store character strings for the first *max-push-pop* integers
(defparameter *max-push-pop* 400)
(defparameter *intlist* nil)

(defun make-int-array (size)
  (let ((result (make-array size)))
    (dotimes (i size)
      (setf (aref result i) (format nil "~d" i)))
    result))

(eval-when (load eval)
  (setq *intlist* (make-int-array *max-push-pop*)))

(defun time-stamp ()
  (multiple-value-bind 
   (second-ignore-me minute hour date month year) (get-decoded-time)
   (declare (ignore second-ignore-me))
   (format nil "~D/~D/~D ~2,'0D:~2,'0D" month date year hour minute)))

(defun get-calls-recursive (func current-list definitions)
  (rplaca current-list (cons func (car current-list)))
  ;; BACK CALL FIX (get-fundef (car func) definitions)
  (let* ((code (generate-code (car func) (cdr func) nil definitions))
	 (tcode (if (and (listp code) (eql (car code) :inline))
		    (cdr code) code)))
    (dolist (line tcode)
      (when (and (eql (first line) 'CALL)
		 (not (find (second line) (car current-list) :test #'equal)))
	(get-calls-recursive (second line) current-list definitions)))
    (car current-list)))

(defun get-code-recursive (name source-type definitions)
  (let ((calls (get-calls-recursive (cons name source-type)
				    (cons nil nil) definitions))
	(result nil))
    (dolist (func calls)
      (setq result (append (cons (list 'FUNC func)
				 (get-cached-code 
				  ;; BACK CALL FIX
				  (car func)
				  (cdr func) definitions))
			   result)))
    result))

(defun write-vect (vector stream)
  (format stream "( ")
  (dotimes (j (length vector))
     (format stream "~A " (convert-anything (elt vector j))))
  (format stream ")~%"))

(defun write-const-line (type value stream)
  (cond ((eql type 'nesl::segdes)
	 (format stream "CONST INT ")
	 (write-vect value stream)
	 (format stream "MAKE_SEGDES~%"))
	((and (eql type 'char) 
	      (not (or (> (length value) 100)
		       (find #\  value) (find #\" value) 
		       (find #\\ value))))
	 (format stream "CONST CHAR ")
	 (format stream "~s~%" (coerce value 'string)))
	(t
	 (format stream "CONST ~a " (if (eql type 'char) 'int type))
	 (write-vect value stream))))

(defun write-const (types values stream)
  (mapcar #'(lambda (x y) (write-const-line x y stream)) types values))

(defun write-code (code definitions stream)
  (declare (special *nesl-version*))
  (format stream "{ Generated by the NESL Compiler, Version ~a, on ~a. }~%"
	  *nesl-version* (time-stamp))
  (dolist (line code)
    (let ((pline (cond ((or (eql (car line) 'FUNC) (eql (car line) 'CALL))
			(list (car line)
			      (get-full-name 
			       (car (second line))
			       (cdr (second line))
			       definitions)))
		       (t line))))
      (when (eql (car pline) 'FUNC)
	    (format stream "~%")
	    )
      (cond ((eql (car pline) 'VCONST)
	     (write-const-line (second pline) 
			       (vcode-vector-data (third pline))
			       stream))
	    ((eql (car pline) 'COPY)	
	     (format stream "COPY ~a ~a~%" 
		     (aref *intlist* (second pline))
		     (aref *intlist* (third pline)))
	     )
	    ((eql (car pline) 'POP)
	     (format stream "POP ~a ~a~%" 
		     (aref *intlist* (second pline))
		     (aref *intlist* (third pline)))
	     )
	    (t 
	     (format stream "~{~a ~}~%" pline)
	     ))
      (when (eql (car pline) 'FUNC)
	(let ((source-type nil) ;; ((substitute 'nesl::segdes 'nesl::vector
			        ;; (flatten-type (cdr (second line)))))
	      (result-type nil) ;; ((substitute 'nesl::segdes 'nesl::vector
			   ;;(flatten-type (get-return-type (car (second line))
				;;  (cdr (second line))
				;;  definitions))))
	      )
	  (format stream "{I~{ ~S~}}~%{O~{ ~S~}}~%" source-type result-type)
	  )))))

(defun write-func (name source-type definitions filename)
  (let ((code  (get-code-recursive (get-fundef name definitions) 
				   source-type definitions)))
    (with-open-file (ofile (merge-pathnames filename) :direction :output)
      (write-code code
		  definitions
		  ofile))))

