(in-package 'nesl-lisp)

(defparameter *header*
"
% This file generated automatically from the function specifications.
\\section{List of Functions}
\\applabel{functions}

\\newcommand{\\tdefop}[4]{
  \\vspace{.15in} \\noindent 
  {\\tt #1} \\hfill {\\em \\{#3 $\\rightarrow$ #2 $:$ #4\\} }\\\\[.05in] }
\\newcommand{\\ldefop}[3]{
  \\vspace{.15in} \\noindent 
  {\\tt #1} \\hfill {\\em \\{#3 $\\rightarrow$ #2\\} }\\\\[.05in] }
\\newcommand{\\fun}[1]{{\\tt #1}}
\\newcommand{\\farg}[1]{{\\tt #1}}

This section lists the functions available in \\nesl.
Each function is listed in the following way:

\\vspace{.05in}
\\tdefop{{\em function interface}}{result-type}{source-types}{type-bindings}
Definition of function.
\\vspace{.2in}

\\noindent The hierarchy of the type classes is shown in \\figref{typeclasses}.
")

(defparameter *tlist* 
  '((nesl::a0 "$a_0$") (nesl::a1 "$a_1$") (nesl::a2 "$a_2$") (nesl::a3 "$a_3$")
    (nesl::a4 "$a_4$") (nesl::a5 "$a_5$") (nesl::a6 "$a_6$") (nesl::a7 "$a_7$")
    (nesl::b0 "$b_0$") (nesl::b1 "$b_1$") (nesl::b2 "$b_2$") (nesl::b3 "$b_3$")
    (nesl::b4 "$b_4$") (nesl::b5 "$b_5$") (nesl::b6 "$b_6$") (nesl::b7 "$b_7$")
    (nesl::inf "$\\infty$") (nesl::minf "$-\\infty$")
    (nesl::t "T") (nesl::f "F")))

(defparameter alpha "$\\alpha$")

(defun gen-latex-element (element stream)
  (cond ((atom element) 
	 (format stream "~a" 	  
		 (if (and (constantp element) (not (eql element t)))
		     element
		   (second (assoc element *tlist*)))))
	((listp element)
	 (format stream "[")
	 (gen-latex-element (car element) stream)
	 (dolist (element (cdr element))
	   (format stream ", ")
	   (gen-latex-element element stream))
	 (format stream "]"))))
      
(defun gen-latex-sequence (name elements stream)
  (format stream "  \\farg{~(~a~)} & = & " name)
  (gen-latex-element elements stream)
  (format stream "\\\\~%"))

(defun gen-latex-example (args interface example stream)
  (format stream "{\\tt \\tabcolsep 4pt \\defexample{lcl}{~%")
  (do ((arg-values (cddr example) (cdr arg-values))
       (arg-names (cdr args) (cdr arg-names)))
      ((null arg-names))
      (gen-latex-sequence (first arg-names) (first arg-values) stream))
  (format stream "  [.07in]~%")
  (gen-latex-sequence interface (first example) stream)
  (format stream "}}~%"))

(defun list-replace-specials (list)
  (if list 
      (let ((rest (list-replace-specials (cdr list))))
	(cond ((member (car list) '(#\_ #\#))
	       (cons #\\ (cons (car list) rest)))
	      (t (cons (car list) rest))))
    nil))

(defun replace-specials (string)
  (coerce (list-replace-specials (coerce string 'list)) 'string))

(defun latex-string (a)
  (replace-specials (string-downcase (string a))))

(defun gen-latex-type (interface types stream)
  (let* ((type (subst alpha 'nesl::alpha types))
	 (typeval (first type))
	 (desttype (latex-string (pretty-type (car typeval))))
	 (sourcetypes (mapcar #'latex-string
			      (mapcar #'pretty-type (cdr typeval))))
	 (typebind  (second type)))
    (if typebind 
	(format stream "~(\\tdefop{~a}{~a}{~a~{,~a~}}{~{ ~a~}}~)~%"
		interface desttype (car sourcetypes) (cdr sourcetypes)
		(list (first typebind) "{\\rm in}" (second typebind)))
      (format stream "~(\\ldefop{~a}{~a}{~a~{,~a~}}~)~%"
	      interface desttype (car sourcetypes) (cdr sourcetypes)))))

(defun gen-latex-description (function stream definitions)
  (cond ((stringp function)
	 (write-string (replace-specials function) stream)
	 (format stream "~%~%"))	
	((symbolp function)
	 (let ((fundef (get-fundef function definitions)))
	   (when (not fundef)
	     (error "While generating documentation, no such function ~a."
		    function))
	   (let* ((args  (fundef-names fundef))
		  (infix (fundef-infix fundef))
		  (interface (latex-string
			      (pretty-function-interface args infix)))
		  (types (fundef-type fundef))
		  (documentation (fundef-documentation fundef))
		  (example (fundef-example fundef)))
	     (gen-latex-type interface types stream)
	     (format stream "\\index{~a}~%" 
		     (latex-string (or infix (car args))))
	     (when documentation
	       (format stream "~a~%" (replace-specials documentation)))
	     (when example
	       (format stream "For example:~%~%")
	       (gen-latex-example args interface example stream))
	     (format stream "~%"))))))

(defun write-all (doclist file definitions)
  (with-open-file (ofile file 
			 :direction :output
			 :if-exists :supersede)
    (format ofile "~a~%" *header*)
    (dolist (op doclist)
      (gen-latex-description op ofile definitions))))

(defun make-doc ()
  (write-all *doclist* 
	     "/afs/cs/project/scandal/papers/nesl/manual.2.6/funlist.tex"
	     *definitions*))

(defparameter *nesl-bugs* `(
"Certain errors will drop you out of the read/eval/print loop into
the lisp error handler.   You can restart by typing (nesl)."
))

(in-package 'nesl)

(lisp::defparameter nesl-lisp::*doclist* '(

  "\\subsection{Scalar Functions} \\applabel{scalar-ops}"
  
  "\\subsubsection{Logical Functions}
  All the logical functions work on either integers or booleans.
  In the case of integers, they work bitwise over
  the bit representation of the integer."
  
  not or and xor nor nand

  "\\subsubsection{Comparison Functions}
  All comparison functions work on integers, floats and characters."

  = /= < > <= >= 

  "\\subsubsection{Predicates}"

  plusp minusp zerop oddp evenp

  "\\subsubsection{Arithmetic Functions}"
  
  + - negate abs diff max min * / rem lshift rshift sqrt isqrt 
  ln log exp expt sin cos tan asin acos atan sinh cosh tanh

  "\\subsubsection{Conversion Functions}"

  btoi code_char char_code float ceil floor trunc round

  "\\subsubsection{Other Scalar Functions}"

  rand

  "\\subsection{Sequence Functions} \\applabel{sequence-ops}"

  "\\subsubsection{Simple Sequence Functions}"

  dist elt rep length index 

  "\\subsubsection{Scans and Reduces}"

  plus_scan max_scan min_scan or_scan and_scan
  iseq 
  sum max_val min_val any all
  count max_index min_index

  "\\subsubsection{Sequence Reordering Functions}"

  -> permute <- rotate reverse

  "\\subsubsection{Simple Sequence Manipulation}"

  pack ++ cons snoc vsep subseq drop take 
  ;; replace_subseq

  "\\subsubsection{Other Sequence Functions}
These are more complex sequence functions.  The step complexities
of these functions are not $O(1)$."

  rank sort collect kth_smallest 
  search_for_subseqs remove_duplicates union intersection

  "\\subsubsection{Nesting Sequences}
  The two functions \\fun{partition} and
  \\fun{flatten} are the primitives for moving between levels of
  nesting.  All other functions for moving between levels of nesting
  can be built out of these.  The functions \\fun{split} and \\fun{bottop}
  are often useful for divide-and-conquer routines."

  partition flatten split bottop head_rest rest_tail

  "\\subsection{Functions on Any Type}"

  eql string hash select

  "\\subsection{Functions with Side Effects}

The functions in this section are not purely functional.  Unless
otherwise noted, none of them can be called in parallel---they cannot
be called within an apply-to-each construct.  The routines in this
section are not part of the core language, they are meant for
debugging, I/O, timing and display."

  "\\subsubsection{Input and Output Routines}
Of the functions in this section, only \\fun{print_char}, \\fun{
print_string}, \\fun{printf_char}, \\fun{printf_string}, and
\\fun{print_debug} can be called in parallel."

  print_char print_string print_debug
  write_object_to_file read_object_from_file write_string_to_file
  open_out_file close_file printf_char printf_string

  "\\subsubsection{Plotting Functions}"

  make_window bounding_box draw_points draw_lines draw_segments
  plot_function

  "\\subsubsection{Other Side Effecting Functions}"

  time

  
))
