;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package :nesl-lisp) 

(defparameter *current-fundef* nil)
(defparameter *current-typedef* nil)

(defun trans-phase1 (variables body pflag)
  (if (and (listp body) (eql (car body) :primitive))
      (if pflag 
	  (cons :primitive (cons (list 'POP 1 0) (cdr body)))
	body)
    (free-body (conv-body variables body pflag))))

(defun phase1-trans-body (vars body pflag)      
  (cond ((and (listp body) (eql (car body) 'nesl::base-typecase))
	 `(,(first body) ,(second body)
	   ,@(mapcar #'(lambda (a) 
			 (list (first a) 
			       (trans-phase1 vars (second a) pflag)))
		     (cddr body))))
	((and (listp body) (eql (car body) 'nesl::poly-typecase))
	 `(,(first body) ,(second body)
	   ,(trans-phase1 vars (third body) pflag)
	   ,(trans-phase1 vars (fourth body) pflag)
	   ,pflag
	   ,@(cddddr body)))
	(t (trans-phase1 vars body pflag))))

(defun special-body? (body)
  (and (listp body)
       (or (eql (car body) 'nesl::base-typecase)
	   (eql (car body) 'nesl::poly-typecase)
	   (eql (car body) :primitive))))

(defun parse-defop (arguments definitions)
  (let* ((names (first arguments))
	 (args (if (eql (second arguments) 'nesl::!)
		   (cddr arguments)
		 (cons nil (cdr arguments))))
	 (keys (cddr args)))
    (declare (special *redefine-default*))
    (when (or (not (listp names))
	      (not (plusp (length names)))
	      (not (symbolp (first names)))
	      (< (length args) 2)
	      (not (zerop (mod (length keys) 2))))
      (nesl-error "Bad function definition: (DEFOP ~a ...).~%~
                   The syntax for DEFOP is:~%~%  ~
         (DEFOP (Ident Ident*) [! typespec]~%    exp)~%"
		  (car arguments)))
    (make-defop names (first args) (second args)
		(get-keyword :type-check keys t)
		(get-keyword :documentation keys nil)
		(get-keyword :example keys nil)
		(get-keyword :redefine keys *redefine-default*)
		(get-keyword :shortdoc keys nil)
		(get-keyword :infix keys nil)
		definitions)))

(defun add-function (names type body pflag definitions
			   &key documentation shortdoc example redefine infix)
  (let* ((ntype (cons (conv-type (car type) pflag) (cdr type)))
	 (nnames (conv-names names pflag))
	 (olddef (get-fundef (car nnames) definitions)))
    (declare (special *redefine-default*))
    (when (and (not *redefine-default*) olddef (fundef-redefine olddef))
      (nesl-error "~a is a built in function, it cannot be redefined."
		  (car names)))
    (make-function-def nnames definitions
		       :documentation documentation
		       :shortdoc shortdoc
		       :type ntype
		       :example example
		       :redefine redefine
		       :infix infix)
    (finalize-function-def (phase1-trans-body (cdr names) body pflag)
			   definitions)))

(defun make-defop (names type body 
			 type-check documentation example redefine shortdoc
			 infix definitions)
  (let* ((*current-fundef* (car names))
	 (simplified-type (if type (simplify-type type) nil))
	 (check-valid-user-type 
	  (check-type-list simplified-type definitions))
	 (type-check (and type-check (not (special-body? body))))
	 (flattype (if type-check
		       (typecheck-op names simplified-type body definitions)
		     simplified-type)))
    (declare (special *current-fundef*) (ignore check-valid-user-type))
    (when (vprefixp (car names))
      (nesl-error 
       "You cannot define a function with a name that starts with v."))
    ;; Add parallel version
    (add-function names flattype body t definitions)
    ;; Add serial version
    (add-function names flattype body nil definitions
		  :documentation documentation
		  :shortdoc shortdoc
		  :example example
		  :redefine redefine
		  :infix infix)
    (values names (cons 'function flattype))))

(defun get-defrec-typebindings (list)
  (if (or (null list) (keywordp (car list)))
      nil
    (cons (car list) (get-defrec-typebindings (cdr list)))))

(defun parse-defrec (arguments definitions)
  (let ((names (first arguments)))	       
    (when (or (not (listp names))
	      (not (plusp (length names)))
	      (not (symbolp (first names)))
	      (< (length arguments) 1))
      (nesl-error "Bad type definition: (DEFREC ~a ...).~%~
                   The syntax for DEFREC is:  ~
         (DEFREC (Ident typeexp*) typebind*)."
		  (car arguments)))
    (let* ((typebindings (get-defrec-typebindings (cdr arguments)))
	   (keys (nthcdr (length typebindings) (cdr arguments))))
      (declare (special *redefine-default*))
      (make-defrec names typebindings
		   (get-keyword :documentation keys nil)
		   (get-keyword :redefine keys *redefine-default*)
		   definitions))))

(defun add-type (names type pflag definitions
		       &key documentation redefine)
  (add-function names type '(:primitive) pflag definitions
		:documentation documentation
		:redefine redefine)
  (add-type-def (car (conv-names names pflag) )
		(cons (conv-type (car type) pflag) (cdr type))
		definitions))

(defun make-args (count)
  (do ((i count (- i 1))
       (l nil (cons (intern (format nil "A~a" i)) l)))
      ((zerop i) l)))
	  
(defun make-defrec (names typebindings
			  documentation redefine
			  definitions)
  (let* ((name (car names))
	 (*current-typedef* name)
	 (simp-types (intern-type-list (cdr names)))
	 (check-valid-type (check-type-list (cons simp-types typebindings)
					    definitions))
	 (full-names (cons name (make-args (length (cdr names)))))
	 (return-type (cons name (mapcar #'first typebindings)))
	 (type (cons (cons return-type simp-types) typebindings))
	 (doc (or documentation
		  (format nil "Constructor for the record type ~a." name))))
    (declare (special *current-typedef*) (ignore check-valid-type))
    (add-type full-names type t definitions)
    (add-type full-names type nil definitions
	      :documentation doc
	      :redefine redefine)
    (values full-names (cons 'function type))))
