;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

;;;
;;; Code for running VCODE process. 
;;; (Original code written by Timothy Freeman.  Major modifications
;;;  by Guy Blelloch.)
;;;
;;; Tested under:
;;; Lucid/Sun common lisp (with :lcl4.1 and :sun in the features list)
;;; Allegoro common lisp (version 3.1, with :allegro and :allegro-v3.1 
;;;                        in the features list)

(in-package :nesl-lisp) 

;; Definition of some utilities for accessing the system.
;; These are compiled conditionally on the Type of lisp.

;;; This takes a string and executes it as if it was a line of text
;;; to the shell.  Standard in, out and error go to the terminal.
(defun run-shell-line (command-string)
  #+lucid
  (lcl::shell command-string)
  #+(or allegro excl)
  (system::run-shell-command command-string)
  #+cmu
  (extensions:run-program "/bin/csh" (list "-fc" command-string)
			  :output *standard-output* :error *standard-output*)
  #-(or lucid allegro excl cmu)
  (error "The NESL/VCODE interface has not been implemented ~
          for the Lisp you are using."))

(defun getenv (varname)
  "Gets an environment variable.  
Returns a string if defined or nil if undefined."
  #+cmu (cdr (assoc (intern varname "KEYWORD") extensions::*environment-list*))
  #+(and lucid lcl3.0) (lucid-common-lisp:environment-variable varname)
  #+(and lucid (not lcl3.0))  (system:environment-variable varname)
  #+(or allegro excl) (system:getenv varname)
  #-(or allegro lucid cmu excl) nil)

;; DEFINITION OF CONFIGURATION STRUCTURE.

(defstruct config
  memsize
  (checkargs t)
  (interp-file "vinterp")
  (munch-lines 0)
  (afs t)
  user-name
  machine-name
  rsh-command
  rcp-source-machine)

(defparameter *current-config* nil)

(defparameter *config-list* nil)

(defmacro add-configuration (name config)
  `(push (cons ',name ,config) *config-list*))

(defun nesl::configuration ()
  (let ((machine-name (config-machine-name *current-config*)))
    (format t "Machine Name:      ~s~%~
               Interpreter File:  ~s~%~
               Memory Size:        ~s~%~
               Argument Checking:  ~s~%"
	    machine-name
	    (config-interp-file *current-config*)
	    (config-memsize *current-config*)
	    (config-checkargs *current-config*))
    (when (not (equal machine-name "local"))
	  (format t "rsh Command:       ~s~%"
		  (config-rsh-command *current-config*))
	  (format t "Remote User Name:  ~s~%"
		  (config-user-name *current-config*)))
    (when (not (config-afs *current-config*))
	  (format t "No access to AFS, will be using rcp for copying.~%"))))

(defun nesl::list-machines ()
  (format t "The current machine configurations are:~%")
  (mapcar #'(lambda (a) (format t "~a~%" (car a))) (reverse *config-list*))
  (format t "To use one type: (use-machine NAME).~%")
  nil)

(defun nesl::listmachines () (nesl::list-machines))

(defun nesl::set-memsize (size)
  (when (not (integerp size))
    (nesl-error "The size argument to SET-MEMSIZE must be an integer."))
  (setf (config-memsize *current-config*) size))

(defun nesl::setmemsize (size) (nesl::set-memsize size))

(defun nesl::set-interpreter-file (filename)
  (setf (config-interp-file *current-config*) filename))

(defun nesl::set-username (user)
  (setf (config-user-name *current-config*) user))

(defun nesl::set-argcheck ()
  (setf (config-checkargs *current-config*)
	(not (config-checkargs *current-config*))))

(defmacro nesl::use-machine (&optional config-name)
  (if config-name 
      `(let ((config (cdr (assoc ',config-name *config-list*))))
	 (cond (config
		(setq *current-config* config)
		(nesl::configuration))
	       (t
		(format t "Machine ~a is not a valid configuration.~%" 
			',config-name)
		(nesl::list-machines))))
    `(nesl::list-machines)))

(defmacro nesl::usemachine (&optional config-name)
  `(nesl::use-machine ,config-name))

(defun nesl::cm-finger ()
  (run-shell-line
   (format nil "~a -l ~a ~a /usr/local/bin/cmfinger"
	   (config-rsh-command *current-config*)
	   (config-user-name *current-config*)
	   (config-machine-name *current-config*))))

(defun nesl::cmfinger ()
  (nesl::cm-finger))

(defun init-machine-configuration ()
  (let ((config (or (getenv "NESL_CONFIG") "local")))
    (format t "; Using machine configuration ~s.~%" config)
    (setq *current-config* 
	  (cdr (assoc (let ((*package* (find-package 'nesl)))
			(declare (special *package*))
			(lisp::read-from-string config))
		      *config-list*)))))

(defvar *username* nil)

(defun user-name ()
  (let ((getenv (getenv "USER")))
    (if getenv getenv
	(if *username* *username*
	    (progn
	      (format t "What is your user name? ")
	      (setq *username* (read-line))
	      *username*)))))		      

(defun uniquify-file-name (file) 
  (concatenate 'string (namestring file) (user-name)))

(defparameter *codefile* (uniquify-file-name "vcode_code"))
;;(defparameter *infile*   (uniquify-file-name "vcode_in"))
(defparameter *outfile*  (uniquify-file-name "vcode_out"))

(defun execute-vcode-file (vcodefile outfile)
  (let* ((machine-name (config-machine-name *current-config*))
	 (user-name (config-user-name *current-config*))
	 (afs (config-afs *current-config*))
	 (interp-file (concatenate 'string *vcode-interpreter-dir* 
				   (config-interp-file *current-config*)))
	 (mem-size (config-memsize *current-config*))
	 (temp-dir *vcode-temp-directory*)
	 (source-machine (config-rcp-source-machine *current-config*))
	 (rsh-command (config-rsh-command *current-config*))
	 (copy-file (concatenate 'string *nesl-path* "src/runNESL"))
	 (vcodefile (concatenate 'string temp-dir vcodefile))
	 (outfile (concatenate 'string temp-dir outfile))
	 (flags (if (config-checkargs *current-config*)
		    (list "-c" "-m" mem-size)
		  (list "-m" mem-size))))
    (cond 
     ;; FOR RUNNING VCODE LOCALLY
     ((equal machine-name "local")
      (when *verbose* (format t "Starting VCODE process locally...~%"))
      (run-shell-line
       (format nil "~a~{ ~a~} ~a > ~a"
	       interp-file flags vcodefile outfile)))

     ;; RUNNING VCODE REMOTELY ON MACHINE WITHOUT AFS ACCESS
     ((not afs)
      (when *verbose* 
	(format t "Copying control file to ~a...~%"
		(string-upcase machine-name)))
      (run-shell-line
       (format nil "~a -l ~a ~a ~
                       \"rcp -p ~a:~a runNESL;./runNESL ~a ~a ~a ~a ~a\""
	       rsh-command
	       user-name machine-name
	       source-machine copy-file
	       source-machine interp-file vcodefile mem-size outfile)))

     ;; RUNNING VCODE REMOTELY ON MACHINE WITH AFS ACCESS
     (t
      (when *verbose* 
	(format t "Starting VCODE process on remote machine ~a...~%"
		(string-upcase machine-name)))
      (run-shell-line
       (format nil "~a -l ~a ~a ~
                           \"~a~{ ~a~} ~a > ~a\""
	       rsh-command
	       user-name machine-name
	       interp-file flags vcodefile outfile))))))

(defun parse-line (line)
  (let ((list (read-from-string line)))
    (when (not (listp list))
      (nesl-error "Error while running VCODE."))
    (substitute nil 'f list)))

(defun parse-segdes-line (line)
  (substitute nil 'f (read-from-string 
		      (substitute #\( #\[
				  (substitute #\) #\] line)))))

(defun convert-anything (thing)
  (cond ((typep thing 'integer) thing)
	((typep thing 'float) thing)
	((member thing '(t nil f)) (if (eql thing t) "T" "F"))
	((typep thing 'character) (char-code thing))
	(t (error "Don't know how to print out ~s." thing))))

(defparameter *verbose* t)

(defun run-vcode (topsym result-types definitions)
  "Run-vcode makes it possible to run vcode programs from lisp.  It
invokes the vcode interpreter for you.

Topsym is a vcode function of no arguments.  Run-vcode generates the code
for topsym to a temporary file and runs the vcode interpreter on the
temporary file.  

Each of the input-lines is converted into one line of text which is
sent as the standard input to the vcode interpreter.  The vcode function
named topsym should use vcode's input primitives to read this
information.

The vcode function named topsym should use vcode's output primitives
to write its results.  Result-types should be a list of the types of
the written results.  If the written results are indeed of the
appropriate types, they will be converted into lisp data
structures, and the list of written results will be the value returned
by run-vcode."

  (let* ((*default-pathname-defaults* (pathname *vcode-temp-directory*))
	 (codefile (merge-pathnames *codefile*))
	 (outfile (merge-pathnames *outfile*)))
    (declare (special *default-pathname-defaults*))
    (when *verbose* (format t "Writing VCODE file....~%"))
    (when (probe-file codefile) (delete-file codefile))
    (write-func topsym '() definitions codefile)
    (when (probe-file outfile) (delete-file outfile))
    (execute-vcode-file *codefile* *outfile*)
    (when (not (probe-file outfile))
      (nesl-error "Error while executing VCODE."))
    (when *verbose* (format t "Reading VCODE output....~%"))
    (with-open-stream (s (open outfile :direction :input))
      (let ((result nil))
	(dotimes (i (config-munch-lines *current-config*)) 
		 (read-line s nil))	; throw away CM output
	(dolist (type result-types)
		(let ((line (read-line s nil)))
		  (when (or (null line))
		    (nesl-error "Error while executing VCODE."))
		  (push
		   (cond
		    ((member type '(int bool nesl::float char)
			     :test #'equal)
		     (parse-line line))
		    ((eql type 'nesl::segdes)
		     (parse-segdes-line line))
		    (t (error "Can't handle the result type ~s yet." type)))
		   result)))
	(nreverse result)))))
