;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

(defstruct fundef
  names
  type
  code
  documentation
  shortdoc
  example
  interface
  fun-cache
  bugs
  redefine
  number)

(defstruct typedef
  name
  type
  documentation
  redefine)

(defstruct definition-table
  function-defs
  current-def
  type-defs
  type-hash
  type-count
  variables
  prim-types
  constants)

;;;;;;;;;;;;

(defun make-defs-table ()
  (make-definition-table 
   :type-hash (make-hash-table :test #'equal)
   :type-count 0))

;;;;;;;;;;;;;;;;;;;;;

(defun make-function-def (names definitions 
				&key type documentation shortdoc 
				example redefine)
  (let* ((old-fundef (get-fundef (car names) definitions))
	 (funstruct (make-fundef :names names
				 :type type 
				 :documentation documentation
				 :shortdoc shortdoc
				 :example example
				 :redefine redefine
				 :number (if old-fundef 
					     (1+ (fundef-number old-fundef))
					   0))))
    (setf (definition-table-current-def definitions)
	  (cons (car names) funstruct))))

(defun finalize-function-def (code definitions)
  (let ((current-def (definition-table-current-def definitions)))
    (setf (fundef-code (cdr current-def)) code)
    (push current-def (definition-table-function-defs definitions))))

(defun get-fundef (name definitions)
  (let* ((cfundef (definition-table-current-def definitions))
	 (val (or (and (eql name (car cfundef)) (cdr cfundef))
		  (cdr (assoc name (definition-table-function-defs 
				     definitions))))))
    (if (fundef-p val) val nil)))

(defun get-typenum (type definitions)
  (let* ((type-hash (definition-table-type-hash definitions))
	 (val (gethash type type-hash)))
    (if val val
      (setf (gethash type type-hash)
	    (incf (definition-table-type-count definitions))))))

(defun add-cached-code (fundef argtypes code definitions)
  (push (cons (get-typenum argtypes definitions) code)
	(fundef-fun-cache fundef))
  code)

(defun get-cached-code (fundef type definitions)
  (let ((typenum (get-typenum type definitions)))
    (cdr (assoc typenum (fundef-fun-cache fundef)))))

(defun get-full-name (fundef type definitions)
  ;; BACK CALL FIX
  (let ((name (car (fundef-names fundef)))
	(number (fundef-number fundef)))
    (intern
     (if (= number 0)
	 (format nil "~A_~d" name (get-typenum type definitions))
       (format nil "~A~d_~d" name number (get-typenum type definitions))))))

;;;;;;;;;;;;;;;;;;;;;

(defun add-type-def (name type definitions)
  (let* ((typestruct (make-typedef :name name :type type)))
    (push (cons name typestruct)
	  (definition-table-type-defs definitions))))

(defun get-typedef (name definitions)
  (let ((val (cdr (assoc name (definition-table-type-defs definitions)))))
    (if (typedef-p val) val nil)))

;;;;;;;;;;;;;;;;

(defun add-prim-type (typename definitions)
  (push (cons typename nil) 
	(definition-table-prim-types definitions)))

(defun add-type-class (typename subtypes definitions)
  (labels ((get-subtypes (subtype)
	     (let* ((typelist (definition-table-prim-types definitions))
		    (subsubtypes (cdr (assoc subtype typelist))))
	       (if subsubtypes 
		   (mapcan #'get-subtypes subsubtypes)
		 (list subtype)))))
    (push (cons typename subtypes)
	  (definition-table-prim-types definitions))))

(defun subtypes (type definitions)
  (assoc type (definition-table-prim-types definitions)))

(defun primitive-type? (type definitions)
  (let ((foo (assoc type (definition-table-prim-types definitions))))
    (and foo (not (cdr foo)))))

;;;;;;;;;;;;;;;

(defun add-variable (var-symbol value type definitions)
  (push (cons var-symbol (cons value type))
	(definition-table-variables definitions)))

(defun get-variable (var-symbol definitions)
  (cdr (assoc var-symbol (definition-table-variables definitions))))

;;;;;;;;;;;;;;;

(defun add-nondefinable-constant (name definitions)
  (push name (definition-table-constants definitions)))

(defun nondefinable-constant? (name definitions)
  (member name (definition-table-constants definitions)))
