(in-package 'nesl-lisp)

(defparameter *header*
"
% This file generated automatically from the function specifications.
\\section{List of Functions}
\\applabel{functions}

\\newcommand{\\tdefop}[4]{
  \\vspace{.15in} \\noindent 
  {\\tt #1} \\hfill {\\em \\{#2 $\\leftarrow$ #3 $:$ #4\\} }\\\\[.05in] }
\\newcommand{\\ldefop}[3]{
  \\vspace{.15in} \\noindent 
  {\\tt #1} \\hfill {\\em \\{#2 $\\leftarrow$ #3\\} }\\\\[.05in] }
\\newcommand{\\fun}[1]{{\\tt #1}}
\\newcommand{\\farg}[1]{{\\tt #1}}

This section lists the functions available in \\nesl.
Each function is listed in the following way:

\\vspace{.05in}
\\tdefop{(function-name arguments)}{result-type}{source-types}{type-bindings}
Definition of function.
\\vspace{.2in}

\\noindent The hierarchy of the type classes is shown in \\figref{typeclasses}.
")

(defparameter *tlist* 
  '((nesl::a0 "$a_0$") (nesl::a1 "$a_1$") (nesl::a2 "$a_2$") (nesl::a3 "$a_3$")
    (nesl::a4 "$a_4$") (nesl::a5 "$a_5$") (nesl::a6 "$a_6$") (nesl::a7 "$a_7$")
    (nesl::b0 "$b_0$") (nesl::b1 "$b_1$") (nesl::b2 "$b_2$") (nesl::b3 "$b_3$")
    (nesl::b4 "$b_4$") (nesl::b5 "$b_5$") (nesl::b6 "$b_6$") (nesl::b7 "$b_7$")
    (nesl::inf "$\\infty$") (nesl::minf "$-\\infty$")
    (nesl::t "T") (nesl::f "F")))

(defparameter alpha "$alpha$")

(defun gen-latex-element (element stream)
  (cond ((atom element) 
	 (format stream "~a" 	  
		 (if (and (constantp element) (not (eql element t)))
		     element
		   (second (assoc element *tlist*)))))
	((listp element)
	 (format stream "\\#v(")
	 (gen-latex-element (car element) stream)
	 (dolist (element (cdr element))
	   (format stream " ")
	   (gen-latex-element element stream))
	 (format stream ")"))))
      
(defun gen-latex-vector (name elements stream)
  (format stream "  \\farg{~(~a~)} & = & " name)
  (gen-latex-element elements stream)
  (format stream "\\\\~%"))

(defun gen-latex-example (args example stream)
  (format stream "{\\tt \\tabcolsep 4pt \\defexample{lcl}{~%")
  (do ((arg-values (cddr example) (cdr arg-values))
       (arg-names (cdr args) (cdr arg-names)))
      ((null arg-names))
      (gen-latex-vector (first arg-names) (first arg-values) stream))
  (format stream "  [.07in]~%")
  (gen-latex-vector args (first example) stream)
  (format stream "}}~%"))

(defun gen-latex-type (name types stream)
  (let* ((type (subst alpha 'nesl::alpha types))
	 (typeval (first type))
	 (desttype (pretty-type (car typeval)))
	 (sourcetypes (mapcar #'pretty-type (cdr typeval)))
	 (typebind  (second type)))
    (if typebind 
	(format stream "~(\\tdefop{~a}{~a}{~{ ~a~}}{~{ ~a~}}~)~%"
		name desttype sourcetypes
		(list (first typebind) "{\\rm in}" (second typebind)))
      (format stream "~(\\ldefop{~a}{~a}{~{ ~a~}}~)~%"
		name desttype sourcetypes))))	      
  
(defun gen-latex-description (function stream definitions)
  (cond ((stringp function)
	 (write-string function stream)
	 (format stream "~%~%"))	
	((symbolp function)
	 (let ((fundef (get-fundef function definitions)))
	   (when (not fundef)
	     (error "While generating documentation, no such function ~a."
		    function))
	   (let ((args (fundef-names fundef))
		 (types (fundef-type fundef))
		 (documentation (fundef-documentation fundef))
		 (example (fundef-example fundef)))
	     (gen-latex-type args types stream)
	     (when documentation
	       (format stream "~a~%" documentation))
	     (when example
	       (format stream "For example:~%~%")
	       (gen-latex-example args example stream))
	     (format stream "~%"))))))

(defun write-all (doclist file definitions)
  (with-open-file (ofile file 
			 :direction :output
			 :if-exists :supersede)
    (format ofile "~a~%" *header*)
    (dolist (op doclist)
      (gen-latex-description op ofile definitions))))

(defun make-doc ()
  (write-all *doclist* 
	     "/afs/cs/project/scandal/papers/nesl/manual.2.4/funlist.tex"
	     *definitions*))

(defparameter *nesl-bugs* `(
"The function PUT does not work on nested vectors if there
are collisions.  For example
  (put #v(\"cow\" \"cat\" \"horse\") #v(1 2 1) #v(\"barn\" \"barn\" \"house\"))
will not work since the \"cow\" and \"horse\" are heading for 
the same \"barn\".  The statement
  (put #v(\"cow\" \"cat\" \"horse\") #v(1 2 0) #v(\"barn\" \"barn\" \"house\"))
works fine."

"The function REP does not work on nested vectors."

"Certain errors will drop you out of the read/eval/print loop into
the lisp error handler.   You can restart by typing (nesl)."

"Returning NaN or Infinity floating point values will cause
a NESL error."
))

(in-package 'nesl)

(lisp::defparameter nesl-lisp::*doclist* '(

  "\\subsection{Scalar Functions} \\applabel{scalar-ops}"
  
  "\\subsubsection{Logical Functions}
  All the logical functions work on either integers or booleans.
  In the case of integers, they work bitwise over
  the bit representation of the integer."
  
  not or and xor nor nand

  "\\subsubsection{Comparison Functions}
  All comparison functions work on integers, floats and characters."

  = /= < > <= >= 

  "\\subsubsection{Predicates}"

  plusp minusp zerop oddp evenp

  "\\subsubsection{Arithmetic Functions}"
  
  + - negate abs diff max min * / rem lshift rshift sqrt isqrt 
  ln log exp expt sin cos tan asin acos atan sinh cosh tanh

  "\\subsubsection{Conversion Functions}"

  btoi code-char char-code float ceil floor trunc round

  "\\subsubsection{Other Scalar Functions}"

  rand

  "\\subsection{Vector Functions} \\applabel{vector-ops}"

  "\\subsubsection{Simple Vector Functions}"

  dist elt rep length index 

  "\\subsubsection{Scans and Reduces}"

  +-scan max-scan min-scan or-scan and-scan
  iseq 
  +-reduce max-reduce min-reduce or-reduce and-reduce
  count max-index min-index

  "\\subsubsection{Vector Reordering Functions}"

  get permute put const-put cond-put rotate

  "\\subsubsection{Vector Manipulation}"

  pack pack-index append cons snoc vpair vsep subseq drop take

  "\\subsubsection{Nesting Vectors}
  The two functions \\fun{partition} and
  \\fun{flatten} are the primitives for moving between levels of
  nesting.  All other functions for moving between levels of nesting
  can be built out of these.  The functions \\fun{split} and \\fun{bottop}
  are often useful for divide-and-conquer routines."

  partition flatten split bottop head-rest rest-tail

  "\\subsection{Other}"

  "\\subsubsection{Functions on Any Types}"

  eql select string time

  "\\subsubsection{Printing Routines}"

  print-char print-string print printl print-debug

  
))
