/*
* Copyright (c) 1992 Carnegie Mellon University 
*                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
*                                  Jonathan Hardwick, Jay Sipelstein,
*                                  Marco Zagha
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
* CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
* ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The SCAL project requests users of this software to return to 
*
*  Guy Blelloch				guy.blelloch@cs.cmu.edu
*  School of Computer Science
*  Carnegie Mellon University
*  5000 Forbes Ave.
*  Pittsburgh PA 15213-3890
*
* any improvements or extensions that they make and grant Carnegie Mellon
* the rights to redistribute these changes.
*/

/* These functions are the actions for the YACC parser. */

#include <stdio.h>
#include <strings.h>

#include "vcode.h"
#include "y.tab.h"
#include "symbol_table.h"
#include "link_list.h"
#include "program.h"
#include "constant.h"
#include "parse.h"

/* ------------------------utilities--------------------------------------*/

/* inc_prog_num: increment next_prog_num counter and check to see if we
 * overflown PROG_SIZE_MAX
 */
static void inc_prog_count()
{
	if (++next_prog_num >= PROG_SIZE_MAX) {
	    fprintf(stderr, "vinterp: program too large, more than %d instructions.\n", PROG_SIZE_MAX);
	    vinterp_exit(1);
	}
}

/* map a lex/yacc token to a TYPE */
TYPE type(n)
int     n;
{
    switch(n) {
        case INT:       return(Int);
        case BOOL:      return(Bool);
        case FLOAT:     return(Float);
	case SEGDES:	return(Segdes);
	case CHAR:	return(String);
        default:        fprintf(stderr, "vinterp: parser: bad type %d, at line %d\n", type, yylineno);
                        return(Illegal);
    }
}

/* -------------------------- basic actions ------------------------------*/

/* parse_stmt: add a new statement (elt-op, vec-op, seg-op) to the program.
 * The grammar assures us that the type arg for the operation is correct,
 * so don't need to error check this.
 */
void parse_stmt(op, type)
VOPCODE op;
TYPE type;
{
    prog_entry_t *prog = program + next_prog_num;
    
    prog->vop = op;
    prog->vopdes = vop_table_look(op);
    prog->lineno = yylineno;
    prog->type = type;
    inc_prog_count();
}

/* Found a new label:  FUNC IDENTIFIER
 * Must enter it into the symbol table, along with its location 
 * in the program so that we can branch to it.
 * FUNC is a nop, so don't generate any code for it.
 */
void parse_label(label)
char *label;
{
    if (strlen(label) >= FN_NAME_MAX) {
	fprintf(stderr, "vinterp: warning: label %s greater than %d characters, will truncate.\n",
			label, FN_NAME_MAX);
	label[FN_NAME_MAX - 1] = '\0';
    }
    if (hash_table_lookup(label) != NULL) {
	/* print warning and continue */
	fprintf(stderr, "vinterp: warning: label %s being redefined on line %d",
	    label, yylineno);
    }
    hash_table_enter(label, next_prog_num);
}

/* Function Call: enter location of call into link_list for linking later.
 * Generate a CALL op in program with aux field to be filled in at link
 * time.
 */
void parse_call(label)
char *label;
{
    prog_entry_t *prog;

    link_list_enter(label, next_prog_num);

    prog = program + next_prog_num;
    prog->vop = CALL;
    prog->vopdes = vop_table_look(CALL);
    prog->lineno = yylineno;
    prog->misc.branch = UNKNOWN_BRANCH;

    inc_prog_count();
}

/* generate a COPY or POP op code. */
void parse_stack(op, arg1, arg2)
VOPCODE op;
int arg1, arg2;
{
    prog_entry_t *prog = program + next_prog_num;

    if (op != COPY && op != POP) {
	fprintf(stderr, "vinterp: bad token value in parse_stack: %d, at line %d", op, yylineno);
	parse_error = 1;
	return;
    }
    
    prog->vop = op;
    prog->vopdes = vop_table_look(op);
    prog->lineno = yylineno;
    prog->misc.stack_args.arg1 = arg1;
    prog->misc.stack_args.arg2 = arg2;

    inc_prog_count();
}

/* ----------------------------- constants --------------------------- */
/* There are two types of VCODE constants: vectors and scalars:
 * A scalar syntactically corresponds to 
		CONST INT 5
 * For this case, the parser calls parse_const(), which puts the
 * constant directly into the corresponding prog_entry.
 * type of constant is second argument.  Switch on this
 * value and grab actual constant from constant global.
 */
void parse_const(op, type)
VOPCODE op;
TYPE type;
{
    prog_entry_t *prog = program + next_prog_num;
    int new_parse_error = 0;

    if (op != CONST) {
	fprintf(stderr, "vinterp: bad token value in parse_const: %d, at line %d\n", op, yylineno);
	new_parse_error = 1;
    }
	
    prog->vop = op;
    prog->vopdes = vop_table_look(op);
    prog->lineno = yylineno;
    prog->type = type;
    prog->is_in_const_pool = 0;

    switch (type) {
	case Int:
	    prog->misc.int_const = constant.ival;
	    break;
	case Float:
	    prog->misc.float_const = constant.dval;
	    break;
	case Bool:
	    prog->misc.bool_const = constant.bval;
	    break;
	default:
	    fprintf(stderr, "bad type value in parse_const: %d, at line %d\n", type, yylineno);
	    new_parse_error = 1;
    }

    if (new_parse_error == 1)
	parse_error = 1;

    inc_prog_count();
}

/* A vector syntactically corresponds to 
 *              CONST INT (1 2 3 4 5)
 * or 		CONST CHAR "constant"
 * In the case, we parse in three parts: parse_begin_vec() allocates
 * storage and the prog_entry; parse_val_vec() adds the next element
 * to the constant vector; parse_end_vec() adds the vector to the 
 * constant pool.
 */

static TYPE current_type = Illegal;		/* type of current vector */

void parse_begin_vec(type)
TYPE type;
{
    current_type = type;

    assert(type == Int || type == Float || type == Bool || type == Segdes);

    if (! const_begin(type))
	parse_error = 1;
    return;
}

/* get a new element of a constant vector */
void parse_val_vec(type)
TYPE type;
{
    if (current_type != type) {
	fprintf(stderr, "vinterp: parser error: illegal type mixing in constant, line %d\n", yylineno);
	parse_error = 1;
	return;
    }

    assert(type == Int || type == Float || type == Bool || type == Segdes);
    const_new_val(type, constant);
}

void parse_end_vec()
{
    prog_entry_t *prog = program + next_prog_num;
    int index;

    prog->vop = CONST;
    prog->vopdes = vop_table_look(CONST);
    prog->lineno = yylineno;
    prog->type = current_type;
    prog->is_in_const_pool = 1;

    index = const_end(current_type);
    if (index == -1) {
	parse_error = 1;
    }
    prog->misc.const_pool_index = index;
    inc_prog_count();
}

void parse_string(string, len)
char *string;
int len;
{
    current_type = String;
    const_string(string, len);

    parse_end_vec();
}


/*------------------ conditionals --------------------------*/
/* Conditionals get treated here.  Since we're generating an array
 * of prog_entry's and not a program graph, its easiest to maintain
 * our own stack of pending IFs in order to resolve the branches.
 * So: parse_if: put an IF entry in the program, and push the program number
 *            onto the if_stack.
 *     parse_else: put an ELSE node in the program.  This is just a branch to 
 *            the corresponding ENDIF.  pop of the IF entry from the
 *	      stack and put in the branch to here.  push an ELSE node
 *            onto the stack.
 *     parse_endif: pop an entry (either IF or ELSE) from the if_stack.
 *            Put the current program number into branch entry of node.
 *	      No need to make new program entry for the ENDIF.
 */

/* for now, make the if_stack an array of ints.  this should eventually be
 * changed to a real stack that can grow dynamically.
 */
#define IF_STACK_MAX	50
int if_stack[IF_STACK_MAX];
int if_stack_num = 0;

static void if_push(node)
int node;
{
    if (if_stack_num >= IF_STACK_MAX) {
	fprintf(stderr, "vinterp: fatal error: IF nesting too deep (> %d), line %d\n", IF_STACK_MAX, yylineno);
	vinterp_exit(1);
    } else {
	if_stack[if_stack_num++] = node;
    }
    return;
}

static int if_pop()
{
    if (if_stack_num <= 0) { 
	fprintf(stderr, "vinterp: fatal error: improper IF nesting, line %d\n", yylineno);
	vinterp_exit(1);
    }
    return if_stack[--if_stack_num];
}

/* indicate if stack is currently empty */
static int if_is_empty()
{
    return (if_stack_num == 0);
}

static void if_clear()
{
    if_stack_num = 0;
}
	

void parse_if()
{
    prog_entry_t *prog = program + next_prog_num;
    
    prog->vop = IF;
    prog->vopdes = vop_table_look(IF);
    prog->lineno = yylineno;

    prog->misc.branch = UNKNOWN_BRANCH;

    if_push(next_prog_num);
    inc_prog_count();
}

void parse_else()
{
    int if_prog_num = if_pop();
    prog_entry_t *prog = program + next_prog_num;
    prog_entry_t *if_prog = program + if_prog_num;
    
    /* fix up if node */
    if_prog->misc.branch = next_prog_num + 1; /* branch to stmt after ELSE */

    prog->vop = ELSE;
    prog->vopdes = vop_table_look(ELSE);
    prog->lineno = yylineno;

    prog->misc.branch = UNKNOWN_BRANCH;

    if_push(next_prog_num);
    inc_prog_count();
}

void parse_endif()
{
    int if_prog_num = if_pop();
    prog_entry_t *if_prog = program + if_prog_num;

    if_prog->misc.branch = next_prog_num;
}

/* ---------------------- return ------------------------------------- */

/* returns:  current language grammar only allows one return per
 *           per function.  We'll enforce this here by making sure
 *           the if_stack is empty at this point.  Grammar
 *           should enforce everything else.
 */
void parse_return()
{
    prog_entry_t *prog = program + next_prog_num;
    
    prog->vop = RET;
    prog->vopdes = vop_table_look(RET);
    prog->lineno = yylineno;

    if (!if_is_empty()) {
	fprintf(stderr, "vinterp: RETURN found in middle of IF...ENDIF, line %d. Only one return per procedure.\n", yylineno);
	if_clear();
	parse_error = 1;
    }

    inc_prog_count();
}
