/*
* Copyright (c) 1992 Carnegie Mellon University 
*                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
*                                  Jonathan Hardwick, Jay Sipelstein,
*                                  Marco Zagha
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
* CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
* ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The SCAL project requests users of this software to return to 
*
*  Guy Blelloch				guy.blelloch@cs.cmu.edu
*  School of Computer Science
*  Carnegie Mellon University
*  5000 Forbes Ave.
*  Pittsburgh PA 15213-3890
*
* any improvements or extensions that they make and grant Carnegie Mellon
* the rights to redistribute these changes.
*/

/* This module contains the facility CVL fucntions */

#ifndef cray
#include <sys/time.h>
#include <sys/resource.h>
#else
#endif

#include <cvl.h>
#include "defins.h"

/* -----------------------Timing functions-----------------------------*/
/* returns number of seconds and microseconds in argument
 * (number of clock ticks on cray)
 */
void tgt_fos(cvl_timer_p)	
cvl_timer_t *cvl_timer_p;
{
#ifndef cray
 	struct rusage rusage;
 	getrusage(0, &rusage);          /* user time of process */
	cvl_timer_p->sec = rusage.ru_utime.tv_sec;
	cvl_timer_p->usec = rusage.ru_utime.tv_usec;
#else
	*cvl_timer_p = rtclock();
#endif
}

/* double precision difference in seconds between two cvl_timer_t values:
 *   t1 occurs after t2
 */
double tdf_fos(t1, t2)
cvl_timer_t *t1, *t2;
{
#ifdef cray
  /* .006 is the clock period (in usecs) of the Y-MP */
  return (double) (1.0e-6)*(.006)*(double)(*t1 - *t2);
#else
  return (double)(t1->sec - t2->sec) +
	 (1.0e-6)*(double)(t1->usec - t2->usec);
#endif /* cray */
}

/* --------------------Size Functions---------------------------------*/

/* siz_ (length): returns number of units of memory of vector.  Units are
   expressed in terms of the quantity CVL_SIZE_UNIT */

#define CVL_SIZE_UNIT sizeof(int)

#define make_siz_func(_name, _type) 				\
   int _name (length)						\
   int length;							\
   {								\
	return length * (sizeof(_type) / CVL_SIZE_UNIT);	\
   }		

make_siz_func(siz_fob, cvl_bool)		/* size of cvl_boolean vector */
make_siz_func(siz_foz, int)			/* size of integer vector */
make_siz_func(siz_fod, double)			/* size of double vector */

/* --------------------Segment Descriptors---------------------------*/

/* n = number of elements in segmented vector */
/* m = number of segments in vector */
/* The segment descriptor is an integer vector of m segment lengths */

int siz_fos(n, m)   /* size of segment descriptor */
int n, m;
{
  return m * (sizeof(int) / CVL_SIZE_UNIT);
}

void mke_fov(sd, l, n, m, scratch)	/* create segment descriptor sd */
vec_p sd, l, scratch;			/* l is a vector of segment lengths */
int n, m;
{
  if (sd != l) cpy_wuz(sd, l, m, scratch);
}

make_inplace(mke_fov,INPLACE_TRUE)

int mke_fov_scratch(n , m)		/* Must take two args */
int n, m;
	{return 0;}

void len_fos(l, sd, n, m, scratch)	/* inverse of mke_fov */
vec_p l, sd, scratch;
int n, m;
{
  if (sd != l) cpy_wuz(l, sd, m, scratch);
}

int len_fos_scratch(n, m)
int n,m;
	{
	return 0;
	}

make_inplace(len_fos,INPLACE_TRUE)


/* -------------------Memory functions------------------------------*/

typedef union maxaligned { int i; double d; float f;} MAXALIGN;

vec_p alo_foz(size)			/* allocate vector memory */
int size;				/* returns NULL if unsuccessful */
{
  return (vec_p) (MAXALIGN*) malloc(size * CVL_SIZE_UNIT);
}

void fre_fov(pointer)			/* free vector memory */
vec_p pointer;
{
  free((char *)pointer);
}

/* ----------------Memory Arithmetic Functions---------------------*/

/* NB: These functions assume that CVL_SIZE_UNIT is sizeof(int)  */

/* Also, we need to make sure these pointers are maximally aligned.
 * Therefore vec_p should be a pointer to something large enough to hold
 * any CVL data type. 
 */

/* add_fov returns a vec_p V, such that a vector at v, of size a, does
 * not overlap into V.
 */
vec_p add_fov(v, a)
vec_p v;
int a;
{
	return ( (vec_p) (MAXALIGN*) ((int *)v + a));
}

/* subtract two pointers and return size of interval in CVL_SIZE_UNITs */
int sub_fov(v1, v2)
vec_p v1, v2;
{
	if ((int *)v1 > (int *)v2)
		return ((int *) v1 - (int *)v2);
	else
		return ((int *) v2 - (int *)v1);
}

/* compare two vec_p's.  We need this because vectors maybe more complicated
 * in some implementations.
 */
cvl_bool eql_fov(v1, v2)
vec_p v1, v2;
{
	return ((int*) v1 == (int*)v2);
}

/* compare two vec_p's for inequality.  Return is same as strcmp. */
int cmp_fov(v1, v2)
vec_p v1, v2;
{
    return ((int *)v1 - (int *)v2);
}
