;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl)

(odefop (prim-+-scan a) ((int <- int vector)
			(float <- float vector))
  ((+_SCAN INT) (+_SCAN FLOAT))
  :primitive t :pversion nil)

(odefop (prim-max-scan a) ((int <- int vector)
			  (float <- float vector)
			  (char <- char vector))
  ((MAX_SCAN INT) (MAX_SCAN FLOAT) (MAX_SCAN INT))
  :primitive t :pversion nil)


(odefop (prim-min-scan a) ((int <- int vector)
			  (float <- float vector)
			  (char <- char vector))
  ((MIN_SCAN INT) (MIN_SCAN FLOAT) (MIN_SCAN INT))
  :primitive t :pversion nil)

(odefop (prim-or-scan a) ((bool <- bool vector))
  ((OR_SCAN BOOL))
  :primitive t :pversion nil)

(odefop (prim-and-scan a) ((bool <- bool vector))
  ((AND_SCAN BOOL))
  :primitive t :pversion nil)

(odefop (prim-+-reduce a) ((int <- int vector)
			  (float <- float vector))
  ((+_REDUCE INT) (+_REDUCE FLOAT))
  :primitive t :pversion nil)

(odefop (prim-max-reduce a) ((int <- int vector)
			    (float <- float vector)
			    (char <- char vector))
  ((MAX_REDUCE INT) (MAX_REDUCE FLOAT) (MAX_REDUCE INT))
  :primitive t :pversion nil)

(odefop (prim-or-reduce a) ((bool <- bool vector))
  ((OR_REDUCE BOOL))
  :primitive t :pversion nil)

(odefop (prim-and-reduce a) ((bool <- bool vector))
  ((AND_REDUCE BOOL))
  :primitive t :pversion nil)

(odefop (prim-min-reduce a) ((int <- int vector)
			    (float <- float vector)
			    (char <- char vector))
  ((MIN_REDUCE INT) (MIN_REDUCE FLOAT) (MIN_REDUCE INT))
  :primitive t :pversion nil)

(odefop (prim-extract V I S) ((int <- int int vector)
			     (float <- float int vector)
			     (char <- char int vector)
			     (bool <- bool int vector))
  ((EXTRACT INT) (EXTRACT FLOAT) (EXTRACT INT) (EXTRACT BOOL))
  :primitive t :pversion nil)

(odefop (prim-replace D I V S) ((int <- int int int vector)
			     (float <- float int float vector)
			     (char <- char int char vector)
			     (bool <- bool int bool vector))
  ((REPLACE INT) (REPLACE FLOAT) (REPLACE INT) (REPLACE BOOL))
  :primitive t :pversion nil)

(odefop (prim-pack V Flags S) (((int vector) <- int bool vector)
			      ((float vector) <- float bool vector)
			      ((char vector) <- char bool vector)
			      ((bool vector) <- bool bool vector))
  ((PACK INT) (PACK FLOAT) (PACK INT) (PACK BOOL))
  :primitive t)

(odefop (fpermute v i f vs ds) ((int <- int int bool vector vector)
			       (float <- float int bool vector vector)
			       (char <- char int bool vector vector)
			       (bool <- bool int bool vector vector))
  ((FPERMUTE INT) (FPERMUTE FLOAT) (FPERMUTE INT) (FPERMUTE BOOL))
  :primitive t :pversion nil)

(odefop (dpermute v i d vs ds) ((int <- int int int vector vector)
			       (float <- float int float vector vector)
			       (char <- char int char vector vector)
			       (bool <- bool int bool vector vector))
  ((DPERMUTE INT) (DPERMUTE FLOAT) (DPERMUTE INT) (DPERMUTE BOOL))
  :primitive t :pversion nil)

(odefop (prim-permute v i vs) ((int <- int int vector)
			 (float <- float int vector)
			 (char <- char int vector)
			 (bool <- bool int vector))
  ((PERMUTE INT) (PERMUTE FLOAT) (PERMUTE INT) (PERMUTE BOOL))
  :primitive t :pversion nil)

(odefop (bfpermute v i f ss ds) ((int <- int int bool vector  vector)
				(float <- float int bool vector  vector)
				(char <- char int bool vector vector)
				(bool <- bool int bool vector vector))
  ((BFPERMUTE INT) (BFPERMUTE FLOAT) (BFPERMUTE INT) (BFPERMUTE BOOL))
  :primitive t :pversion nil)

(odefop (bpermute v i ss ds) ((int <- int int vector vector)
			     (float <- float int vector vector)
			     (char <- char int vector vector)
			     (bool <- bool int vector vector))
  ((BPERMUTE INT) (BPERMUTE FLOAT) (BPERMUTE INT) (BPERMUTE BOOL))
  :primitive t :pversion nil)

(odefop (read-int) ((int <-))
  ((READ INT))
  :primitive t :pversion nil)

(odefop (read-float) ((float <-))
  ((READ FLOAT))
  :primitive t :pversion nil)

(odefop (read-char) ((char <-))
  ((READ INT))
  :primitive t :pversion nil)

(odefop (read-bool) ((bool <-))
  ((READ BOOL))
  :primitive t :pversion nil)

(odefop (read-segdes) ((vector <-))
  ((READ SEGDES))
  :primitive t :pversion nil)

;; this is temporary
;;(odefop (prim-dist A L) ((vector <- vector vector)) A)

(opsection "Vector Functions" "\\seclabel{vector-ops}")

(opsubsection "Simple Vector Functions" "")

(defop (eql a b) (bool <- bool bool)
  (not (xor a b)))

(odefop (eql a b) ((bool <- float float)
		  (bool <- int int)
		  (bool <- char char))
  (= a b))

(ndefop (eql a b) ((bool <- alpha alpha) (alpha any))
  (typecase a
     (= a b)
     (if (= (length a) (length b))
	 (and-reduce (v.eql a b))
       f)
     (and (eql (first a) (first b)) 
	  (eql (second a) (second b)))))

(ndefop (select Flag v1 v2) ((alpha <- bool alpha alpha) (alpha any))
  (if flag v1 v2))

;;;;;;;;;;;;;;;;;;;;;;;

(ndefop (dist-index len count) ((v.int <- int int))
   (flatten (over ((ind (index len)))
	      (over ((in (index count)))
		(+ ind (* in len))))))

(ndefop (prim-dist a segdes) ((alpha <- alpha vector) (alpha any))
  (typecase a
    (prim-dist a segdes)
    (with ((i (len segdes))
	   (vals (flatten (v.dist a v.i)))
	   (l (length a)))
      (second (partition (permute vals (dist-index l i)) (dist l i))))
    (tup (prim-dist (first a) segdes) 
	 (prim-dist (second a) segdes))))

(ndefop (dist a l) ((v.alpha <- alpha int) (alpha any))
  (with ((segdes (vector l)))
    (tup segdes (prim-dist a segdes)))
  :documentation
  "Generates a vector of length \\farg{l} with the value \\farg{a}
in each element."
  :example
  ((a0 a0 a0 a0 a0) <- a0 5)
  :document yes)


(ndefop (dist-l v (s x)) ((v.alpha <- alpha v.beta) ((alpha any) (beta any)))
  (tup s (prim-dist v s)))

(ndefop (elt-indices seg-lengths i) ((v.int <- v.int int))
  (with ((offset (elt (+-scan seg-lengths) i))
	 (length (elt seg-lengths i)))
    (v.+ (index length) v.offset)))

(ndefop (elt a i) ((alpha <- v.alpha int) (alpha any))
  (with (((seg val) a))
    (typecase val
      (prim-extract val i seg)
      (get (flatten a) (elt-indices (v.length a) i))
      (tup (elt (v.first a) i)
	   (elt (v.second a) i))))
  :documentation
  "Extracts the element specified by index \\farg{i} from the
vector \\farg{a}.  Indices are zero-based."
  :document yes)

(ndefop (rep d v i) ((v.alpha <- v.alpha alpha int) (alpha any))
  (with (((seg vect) d))
    (typecase vect
      (tup seg (prim-replace vect i v seg))
      :not-implemented
      :not-implemented))
  :documentation
  "Replaces the \\farg{i}th value in the vector \\farg{v} with the
value \\farg{a}."
  :interface (rep v a i)
  :document yes
  :example
  ((a0 a1 a2 b0 a4) <- (a0 a1 a2 a3 a4) b0 3)
)

(ndefop (length v) ((int <- v.alpha) (alpha any))
  (len (first v))
  :documentation
"Returns the length of a vector."
  :document yes)

(ndefop (index-l (seg val)) ((v.int <- v.alpha) (alpha any))
  (tup seg (prim-+-scan (prim-dist 1 seg) seg)))

(odefop (index l) ((v.int <- int))
 (with ((segdes (vector l)))
   (tup segdes (prim-+-scan (prim-dist 1 segdes) segdes)))
  :documentation
"Given an integer, \\fun{index} returns a vector of
that length with consecutive integers starting at 0
in the elements."
  :example
  ((0 1 2 3 4 5 6 7) <- 8)
  :redefine no
  :type ((int <- int))
  :document yes)

(opsubsection "Scans and Reduces" "")

(odefop (+-scan a) ((v.int <- v.int)
		   (v.float <- v.float))
  ;; (tup (first a) (prim-+-scan (second a) (first a)))
  (((COPY 1 1) (+_SCAN INT)) ((COPY 1 1) (+_SCAN FLOAT)))
  :compound-prim t
  :documentation
"Given a vector of numbers, \\fun{+-scan} returns to each position
of a new equal-length vector, the sum of all previous
positions in the source."
  :example
  ((0 1 4 9 16 25 36 49) <-
   (1 3 5 7 9 11 13 15))
  :type ((v.alpha <- v.alpha) (alpha number))
  :document yes)

(odefop (max-scan a) ((v.int <- v.int)
		     (v.float <- v.float)
		     (v.char <- v.char))
  ;; (tup (first a) (prim-max-scan (second a) (first a)))
  (((COPY 1 1) (MAX_SCAN INT)) ((COPY 1 1) (MAX_SCAN FLOAT))
   ((COPY 1 1) (MAX_SCAN INT)))
  :compound-prim t
  :documentation
"Given a vector of ordinals, \\fun{max-scan} returns to each position
of a new equal-length vector, the maximum of all previous
positions in the source."
  :example
  ((minf 3 3 3 6 6 6) <-
   (3 2 1 6 5 4 8))
  :type ((v.alpha <- v.alpha) (alpha ordinal))
  :document yes)

(odefop (min-scan a) ((v.int <- v.int)
		     (v.float <- v.float)
		     (v.char <- v.char))
  ;; (tup (first a) (prim-min-scan (second a) (first a)))
  (((COPY 1 1) (MIN_SCAN INT)) ((COPY 1 1) (MIN_SCAN FLOAT))
   ((COPY 1 1) (MIN_SCAN INT)))
  :compound-prim t
  :documentation
"Given a vector of ordinals, \\fun{min-scan} returns to each position
of a new equal-length vector, the minimum of all previous
positions in the source."
  :type ((v.alpha <- v.alpha) (alpha ordinal))
  :document yes)

(odefop (or-scan a) ((v.bool <- v.bool))
  (((COPY 1 1) (OR_SCAN BOOL)))
  :compound-prim t
  :documentation
"A scan using logical-or on a vector of booleans."
  :type ((v.bool <- v.bool))
  :document yes)

(odefop (and-scan a) ((v.bool <- v.bool))
  (((COPY 1 1) (AND_SCAN BOOL)))
  :compound-prim t
  :documentation
"A scan using logical-and on a vector of booleans."
  :type ((v.bool <- v.bool))
  :document yes)

(odefop (enumerate flags) ((v.int <- v.bool))
  (+-scan (v.btoi flags))
  :documentation
"Given a boolean vector, \\fun{enumerate} returns
a vector of equal length with consecutive integers,
starting at 0, at the positions with a \\farg{t} in
the flag."
  :example
  ((0 1 1 1 2 2 3 4) <-
   (t f f t f t t f))
  :redefine no
  :type ((v.int <- v.bool)))

(odefop (iseq start stride end) ((v.int <- int int int))
   (v.+ (+-scan (dist stride 
		      (+ (/ (- (abs (- end start)) 1) (abs stride)) 1))) 
        v.start)
   :type ((v.int <- int int int))
   :document yes
   :documentation
"Returns a set of indices starting at \\farg{start}, increasing
by \\farg{stride}, and finishing before \\farg{end}."
   :example
   ((4 7 10 13) <- 4 3 15))

(odefop (+-reduce v) ((int <- v.int)
		     (float <- v.float))
  ;; (prim-+-reduce (second v) (first v))
  (((COPY 1 1) (+_REDUCE INT) (POP 1 1))
   ((COPY 1 1) (+_REDUCE FLOAT) (POP 1 1)))
  :compound-prim t
  :documentation
"Given a vector of numbers, \\fun{+-reduce} returns
their sum."
  :example
  (32 <- (7 2 9 11 3))
  :redefine no
  :type ((alpha <- v.alpha) (alpha number))
  :document yes)

(odefop (max-reduce v) ((int <- v.int)
		       (float <- v.float)
		       (char <- v.char))
  ;; (prim-max-reduce (second v) (first v))
  (((COPY 1 1) (MAX_REDUCE INT) (POP 1 1))
   ((COPY 1 1) (MAX_REDUCE FLOAT) (POP 1 1))
   ((COPY 1 1) (MAX_REDUCE INT) (POP 1 1)))
  :compound-prim t
  :documentation
  "Given a vector of ordinals, \\fun{max-reduce} returns
their maximum."
  :redefine no
  :type ((alpha <- v.alpha) (alpha ordinal))
  :document yes)

(odefop (min-reduce v) ((int <- v.int)
		       (float <- v.float)
		       (char <- v.char))
  ;; (prim-min-reduce (second v) (first v))
  (((COPY 1 1) (MIN_REDUCE INT) (POP 1 1))
   ((COPY 1 1) (MIN_REDUCE FLOAT) (POP 1 1))
   ((COPY 1 1) (MIN_REDUCE INT) (POP 1 1)))
  :compound-prim t
  :documentation
  "See max-reduce."
  :type ((alpha <- v.alpha) (alpha ordinal))
  :redefine no
  :document yes)

(odefop (or-reduce (s v)) ((bool <- v.bool))
  (prim-or-reduce v s)
  :interface (or-reduce v)
  :documentation
  "A reduce with logical or."
  :redefine no
  :type ((alpha <- v.alpha) (alpha logical))
  :document yes)

(odefop (and-reduce (s v)) ((bool <- v.bool))
  (prim-and-reduce v s)
  :interface (and-reduce v)
  :documentation
  "A reduce with logical and."
  :redefine no
  :type ((alpha <- v.alpha) (alpha logical))
  :document yes)


(odefop (count flags) ((int <- v.bool))
  (+-reduce (v.btoi flags))
  :documentation
"Counts the number of \\farg{t} flags in a boolean vector."
  :example
  (5 <- (t f t t f t f t))
  :redefine no
  :type ((int <- v.bool))
  :document yes)

(odefop (+-dist v) ((v.int <- v.int)
		   (v.float <- v.float))
  (dist-l (+-reduce v) v)
  :documentation
"Returns the sum of a vector to all positions of a vector.")

(odefop (max-dist v) ((v.int <- v.int)
		     (v.float <- v.float)
		     (v.char <- v.char))
  (dist-l (max-reduce v) v)
  :documentation
"Returns the maximum element to all positions of a vector.")

(odefop (min-dist v) ((v.int <- v.int)
		     (v.float <- v.float)
		     (v.char <- v.char))
  (dist-l (min-reduce v) v)
  :documentation
"Returns the minimum element to all positions of a vector.")

(odefop (max-index v) ((int <- v.int)
		      (int <- v.float)
		      (int <- v.char))
  (with ((l (length v)))
    (min-reduce (v.select (v.= (max-dist v) v) (index-l v) (dist-l l v))))
  :documentation
"Given a vector of ordinals, \\fun{max-index}
returns the index of the maximum value.  If several values
are equal, it returns the leftmost index."
  :example
  (4 <- (2 11 4 7 14 6 9 14))
  :redefine no
  :type ((int <- v.alpha) (alpha ordinal))
  :document yes)

(odefop (min-index v) ((int <- v.int)
		      (int <- v.float)
		      (int <- v.char))
  (with ((l (length v)))
    (min-reduce (v.select (v.= (min-dist v) v) (index-l v) (dist-l l v))))
  :documentation
"Given a vector of ordinals, \\fun{min-index}
returns the index of the minimum value.  If several values
are equal, it returns the leftmost index."
  :redefine no
  :type ((int <- v.alpha) (alpha ordinal))
  :document yes)

(opsubsection "Vector Reordering Functions" "")

(ndefop (seg-permute-index seg-lengths dest-seg-lengths i) 
  ((v.int <- v.int v.int v.int))
  (flatten (over ((offset     (get (+-scan dest-seg-lengths) i))
		  (seg-length seg-lengths))
	     (v.+ v.offset (index seg-length)))))

(ndefop (len-get-index seg-lengths i) 
  (((v.int v.int) <- v.int v.int))
  (with ((new-lengths (get seg-lengths i)))
    (tup (seg-permute-index new-lengths seg-lengths i)
	 new-lengths)))

(ndefop (get a i)  ((v.alpha <- v.alpha v.int) (alpha any))
  (with (((seg vals) a))
    (typecase vals
      (with (((segi valsi) i))
	(tup segi (bpermute vals valsi seg segi)))
      (with ((lengths (v.length a))
	     ((indices new-seg-lengths) (len-get-index lengths i)))
	(partition (get (flatten a) indices) new-seg-lengths))
      (v.tup (get (v.first a) i)
	     (get (v.second a) i))))
  :document yes
  :example
  ((a3 a5 a2 a6) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (3 5 2 6)
   (t t t t))
  :documentation
"Given a vector of \\farg{values} and a vector of \\farg{indices},
which can be of different lengths, \\fun{get} returns a vector which is
the same length as the \\farg{indices} vector and the same type 
as the \\farg{values} vector.  For each position in the \\farg{indices} 
vector, it extracts the value at that index of the \\farg{values} vector."
  :interface (get values indices)
  :redefine no)

(ndefop (cond-get (sseg v) (dseg i) (fseg flags))
	((v.alpha <- v.alpha v.int v.bool) (alpha any))
  (typecase v
    (tup dseg (bfpermute v i flags sseg dseg))
    :not-implemented
    :not-implemented)
  :documentation
"Similar to the \\fun{get} function, but the \\farg{flags} vector,
which must be the same length as the \\farg{indices} vector, masks
out positions where the flag is \\farg{f} such that nothing is fetched by
those positions and the identity is returned."
  :interface (cond-get values indices flags)
  :example
  ((a3 0 a5 a2 0 a6) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (3 1 5 2 5 6)
   (t f t t f t))
  :document yes
  :redefine no)

(ndefop (permute v i) ((v.alpha <- v.alpha v.int) (alpha any))
  (with (((seg values) v))
    (typecase values
      (tup seg (prim-permute values (second i) seg))
      (len-put v i (length v))
      (v.tup (permute (v.first v) i)
  	     (permute (v.second v) i))))
  :interface (permute values indices)
  :documentation  
"Given a vector of \\farg{values} and a vector of \\farg{indices},
which must be of the same length, \\fun{permute} permutes the
values to the given indices.  The permutation must be 1-to-1."
  :document yes
  :redefine no)

(odefop (put values indices defaults) ((v.int <- v.int v.int v.int)
				      (v.float <- v.float v.int v.float)
				      (v.char <- v.char v.int v.char)
				      (v.bool <- v.bool v.int v.bool))
  ;;(tup (first D)
  ;;(dpermute (second V) (second I) (second D) (first V) (first D)))
  (((COPY 1 1) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 7) (COPY 1 6) 
    (DPERMUTE INT) (POP 6 2))
   ((COPY 1 1) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 7) (COPY 1 6) 
    (DPERMUTE FLOAT) (POP 6 2))
   ((COPY 1 1) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 7) (COPY 1 6) 
    (DPERMUTE INT) (POP 6 2))
   ((COPY 1 1) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 7) (COPY 1 6) 
    (DPERMUTE BOOL) (POP 6 2)))
  :compound-prim t)

(ndefop (len-put-index seg-lengths i dest-len) 
  (((v.int v.int) <- v.int v.int int))
  (with ((new-lengths (len-put seg-lengths i dest-len)))
    (tup (seg-permute-index seg-lengths new-lengths i)
	 new-lengths)))

(ndefop (vselect flag v1 v2) ((v.alpha <- v.bool v.alpha v.alpha) (alpha any))
  (with (((seg val) v1))
  (typecase val
    (v.select flag v1 v2)
    (with ((nl (v.select flag (v.length v1) (v.length v2))))
      (partition (flag-merge (flatten (v.dist (v.not flag) nl))
			     (flatten (pack v1 flag))
			     (flatten (pack v2 (v.not flag))))
		 nl))
    (v.tup (vselect flag (v.first v1) (v.first v2))
           (vselect flag (v.second v1) (v.second v2))))))

(ndefop (put values indices defaults)
	((v.alpha <- v.alpha v.int v.alpha) (alpha any))
  (with ((dest-len (length defaults)))
    (vselect (len-put (dist t (length values)) indices dest-len)
	     (len-put values indices dest-len)
	     defaults))
  :documentation
"Given a vector of \\farg{values} and a vector of \\farg{indices},
which must be of the same length, \\fun{put} places the
values to the given indices in the \\farg{defaults} vector.  "
  :example
  ((b0 a0 a2 a3 b4 a1) <-
   (a0 a1 a2 a3)
   (1 5 2 3)
   (b0 b1 b2 b3 b4 b5))
  :redefine no
  :document yes)

(ndefop (const-put value indices defaults)
	((v.alpha <- alpha v.int v.alpha) (alpha any))
  (put (dist-l value indices) indices defaults)
  :documentation
"Given a vector of \\farg{indices}, \\fun{put} places the
constant \\farg{value} at each index in the \\farg{defaults} vector. "
  :redefine no
  :document yes)

(ndefop (cond-put v i flags d)  
	((v.alpha <- v.alpha v.int v.bool v.alpha) (alpha any))
  (put (pack v flags) (pack i flags) d)
  :documentation
"Similar to the \\fun{put} function, but the extra \\farg{flags} vector,
which must be the same length as the indices (\\farg{i}) vector, masks
out positions where the flag is \\farg{f} such that nothing is placed by
those positions."
  :example
  ((b0 a0 b2 a5) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (1 5 4 6 2 3 7 0)
   (t f f f f t f f)
   (b0 b1 b2 b3))
  :redefine no
  :document yes)

#|
(odefop (cond-len-put v i flags l) ((v.int <- v.int v.int v.bool int)
			  (v.bool <- v.bool v.int v.bool int)
			  (v.float <- v.float v.int v.bool int)
			  (v.char <- v.char v.int v.bool int))
  ;;(with ((dseg (vector l)))
  ;; (tup dseg (fpermute (second v) (second i) (second flags) (first v) dseg)))
  (((MAKE_SEGDES) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 5) (COPY 1 4)
    (FPERMUTE INT) (POP 6 2))
   ((MAKE_SEGDES) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 5) (COPY 1 4)
    (FPERMUTE BOOL) (POP 6 2))
   ((MAKE_SEGDES) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 5) (COPY 1 4)
    (FPERMUTE FLOAT) (POP 6 2))
   ((MAKE_SEGDES) (COPY 1 5) (COPY 1 4) (COPY 1 3) (COPY 1 5) (COPY 1 4)
    (FPERMUTE INT) (POP 6 2)))
  :type ((v.alpha <- v.alpha v.int v.bool int) (alpha any))
  :compound-prim t
  :example
  ((0 a0 0 a5) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (1 5 4 6 2 3 7 0)
   (t f f f f t f f)
   4)
  :redefine no)
|#

(odefop (len-put v i l) ((v.int <- v.int v.int int))
  (put v i (dist 0 l))
  :type ((v.alpha <- v.alpha v.int int) (alpha any))
  :redefine no)

(defop (len-put v i l) (v.float <- v.float v.int int)
  (put v i (dist 0.0 l)))

(defop (len-put v i l) (v.bool <- v.bool v.int int)
  (put v i (dist f l)))

(defop (len-put v i l) (v.char <- v.char v.int int)
  (put v i (dist #\null l)))

(ndefop (len-put a i l) ((v.alpha <- v.alpha v.int int) (alpha any))
  (with (((seg vals) a))
    (typecase vals
      (len-put a i l)
      (with ((lengths (v.length a))
	     ((indices new-seg-lengths) (len-put-index lengths i l)))
	(partition (permute (flatten a) indices) new-seg-lengths))
      (v.tup (len-put (v.first a) i l)
	     (len-put (v.second a) i l)))))

(ndefop (rotate a i) ((v.alpha <- v.alpha int) (alpha any))
  (with ((l   (length a))
	 (idx (index-l a)))
     (permute a (over ((idx idx)) (mod (+ i idx) l))))
  :documentation
  "Given a vector and an integer, \\fun{rotate} rotates the vector
around by \\farg{i} positions to the right.  If the integer is negative, then
the vector is rotated to the left."
  :example
  ((a5 a6 a7 a0 a1 a2 a3 a4) <-
   (a0 a1 a2 a3 a4 a5 a6 a7) 3)
  :redefine no
  :document yes)

(ndefop (previous a v) ((v.alpha <- v.alpha alpha) (alpha any))
  (rep (rotate a 1) v 0))

(ndefop (next a v) ((v.alpha <- v.alpha alpha) (alpha any))
  (rep (rotate a -1) v (- (length a) 1)))

(opsubsection "Vector Manipulation" "")

(odefop (prim-pack v flags) ((v.int <- v.int v.bool)
			     (v.float <- v.float v.bool)
			     (v.char <- v.char v.bool)
			     (v.bool <- v.bool v.bool))
	(((COPY 1 1) (POP 1 2) (PACK INT) (COPY 1 1) (POP 2 2))
	 ((COPY 1 1) (POP 1 2) (PACK FLOAT) (COPY 1 1) (POP 2 2))
	 ((COPY 1 1) (POP 1 2) (PACK INT) (COPY 1 1) (POP 2 2))
	 ((COPY 1 1) (POP 1 2) (PACK BOOL) (COPY 1 1) (POP 2 2)))
	:compound-prim t)

(ndefop (pack values flags) ((v.alpha <- v.alpha v.bool) (alpha any))
  (with (((seg v) values))
    (typecase v
      (prim-pack values flags)
      (with ((fvalues (flatten values))
	     (lengths (v.length values)))
	(partition (pack fvalues (flatten (v.dist flags lengths)))
		   (pack lengths flags)))
      (v.tup (pack (v.first values) flags)
	     (pack (v.second values) flags))))
  :documentation
  "Given a sequence of values, and an equal length boolean sequence of
flags, \\fun{pack} packs all the elements with a \\farg{t} in the
corresponding position in \\farg{flags} into consecutive elements, 
deleting elements with an \\farg{f}."
  :example
  ((a0 a2 a5 a6 a7) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (t f t f f t t t))
  :redefine no
  :document yes)

(odefop (pack-index flags) ((v.int <- v.bool))
  (pack (index-l flags) flags)
  :type ((v.int <- v.bool))
  :documentation
  "Given a boolean sequence of
flags, \\fun{pack-index} returns a vector containing the indices of
each of the true flags."
  :example
  ((0 2 5 6 7) <-
   (t f t f f t t t))
  :redefine no
  :document yes)

(ndefop (unpack v flags) ((v.alpha <- v.alpha v.bool) (alpha any))
  (cond-get v (enumerate flags) flags)
  :documentation
"Given a sequence of values, and longer or equal length sequence of flags, 
which has as many T's as values in \\farg{V}, \\fun{unpack} unpacks the 
elements into positions of the flags with a \\farg{t}."
  :example
  ((a0 0 a1 0 0 a2 a3 a4) <-
   (a0 a1 a2 a3 a4)
   (t f t f f t t t))
  :redefine no)

(ndefop (reverse a) ((v.alpha <- v.alpha) (alpha any))
  (with ((l (length a))
	 (l1 (- l 1))
	 (nindex (v.- v.l1 (index-l a))))
    (permute a nindex)))

(odefop (+-back-scan a) ((v.int <- v.int)
			(v.float <- v.float))
  (reverse (+-scan (reverse a)))
  :documentation
"Executes a \\fun{+-scan} in reverse order."
  :example
  ((63 60 55 48 39 28 15 0) <-
   (1 3 5 7 9 11 13 15)))

(odefop (max-back-scan a) ((v.int <- v.int)
			  (v.float <- v.float)
			  (v.char <- v.char))
  (reverse (max-scan (reverse a)))
  :documentation
"Executes a \\fun{max-scan} in reverse order.")

(odefop (min-back-scan a) ((v.int <- v.int)
			  (v.float <- v.float)
			  (v.char <- v.char))
  (reverse (min-scan (reverse A)))
  :documentation
"Executes a \\fun{min-scan} in reverse order.")

(ndefop (join v1 p1 v2 p2) 
	((v.alpha <- v.alpha v.int v.alpha v.int) (alpha any))
  (put v1 p1 (len-put v2 p2 (+ (length v1) (length v2))))
  :documentation
"Given two sequences of pointers and two sequences of values,
\\fun{join} merges the values into the positions specified
by the pointers.  \\farg{V1} and \\farg{P1} must be of the same
length, and \\farg{V2} and \\farg{P2} must be of the same length."
  :example
  ((b0 a0 b1 b2 a1 a2 b3 b4) <-
   (a0 a1 a2) (1 4 5)
   (b0 b1 b2 b3 b4) (0 2 3 6 7))
  :redefine no)

(ndefop (append v1 v2) ((v.alpha <- v.alpha v.alpha) (alpha any))
  (join v1 (index-l v1)
        v2 (v.+ (dist-l (length v1) v2)
		(index-l v2)))
  :documentation
  "Given two sequences, \\fun{append} appends them."
  :document yes
  :example
  ((a0 a1 a2 b0 b1) <-
   (a0 a1 a2) (b0 b1)))

(ndefop (cons a v)  ((v.alpha <- alpha v.alpha) (alpha any))
  (append (dist a 1) v)
  :documentation
  "Given a value \\farg{a} and a sequence of values \\farg{v}, \\fun{cons}
concatenates the value onto the front of the sequence."
  :document yes
  :example
  ((a0 b0 b1 b2 b3) <-
   a0 (b0 b1 b2 b3)))

(ndefop (snoc v a) ((v.alpha <- v.alpha alpha) (alpha any))
  (append v (dist a 1))
  :documentation
  "Given a sequence of values \\farg{v} and  a value \\farg{a}, \\fun{snoc}
concatenates \\farg{a} to the end of the sequence."
  :document yes
  :example
  ((b0 b1 b2 b3 a0) <-
   (b0 b1 b2 b3) a0))

(ndefop (vtup a b) ((v.alpha <- alpha alpha) (alpha any))
  (typecase a
    (rep (dist b 2) a 0)
    (partition (append a b) (vtup (length a) (length b)))
    (v.tup (vtup (first a) (first b))
           (vtup (second a) (second b))))
  :documentation
  "Given two values of the same type, \\fun{vtup} puts them together
into a sequence of length 2."
  :document yes
  :example
  ((a0 a1) <-
   a0 a1))

(ndefop (vsep a) (((alpha alpha) <- v.alpha) (alpha any))
  (tup (elt a 0) (elt a 1))
  :documentation
  "Given a vector of length 2, \\fun{vsep} returns a tuple
with the first vector element in the first position and
the second vector element in the second position."
  :document yes)

(ndefop (subseq v start end) ((v.alpha <- v.alpha int int) (alpha any))
  (get v (v.+ (index (- end start)) v.start))
  :documentation
"Given a sequence, \\fun{subseq} returns the subsequence 
starting at position \\farg{start} and ending one before
position \\farg{end}."
  :example
  ((a2 a3 a4 a5) <-
   (a0 a1 a2 a3 a4 a5 a6 a7) 2 6)
  :redefine no
  :document yes)

(ndefop (drop n v) ((v.alpha <- int v.alpha) (alpha any))
  (subseq v n (length v))
  :documentation
"Given a sequence, \\fun{drop} drops the first \\farg{n} items
from the sequence."
  :example
  ((a3 a4 a5 a6 a7) <-
   3 (a0 a1 a2 a3 a4 a5 a6 a7))
  :redefine no
  :document yes)

(ndefop (take n v) ((v.alpha <- int v.alpha) (alpha any))
  (subseq v 0 n)
  :documentation
"Given a sequence, \\fun{take} takes the first \\farg{n} items
from the sequence."
  :example
  ((a0 a1 a2) <-
   3 (a0 a1 a2 a3 a4 a5 a6 a7))
  :redefine no
  :document yes)

(ndefop (flag-merge flags v1 v2) 
	((v.alpha <- v.bool v.alpha v.alpha) (alpha any))
  (with ((i (index (length flags))))
    (join v2 (pack i flags)
          v1 (pack i (v.not flags))))
  :documentation
"Given two vectors of values and a boolean vector of flags,
\\fun{flag-merge} merges the values according to the flags.
Positions with a T in their flag will get values from the second vector,
and positions with a \\farg{f} in their flag will get values from the 
first vector.  The ordering is maintained."
  :example
  ((b0 a0 b1 a1 a2 b2 b3 b4) <-
   (a0 a1 a2) (b0 b1 b2 b3 b4)
   (t f t f f t t t))
  :redefine no)

(ndefop (odd-elts v) ((v.alpha <- v.alpha) (alpha any))
  (pack v (v.oddp (index (length v))))
  :documentation
"Returns the odd indexed elements of a vector.")

(ndefop (even-elts v) ((v.alpha <- v.alpha) (alpha any))
  (pack v (v.evenp (index (length v))))
  :documentation
"Returns the even indexed elements of a vector.")

(ndefop (interleave a b) ((v.alpha <- v.alpha v.alpha) (alpha any))
  (flag-merge (v.oddp (index (+ (length a) (length b)))) a b)
  :documentation
"Interleaves the elements of two vectors."
  :example
  ((a0 b0 a1 b1 a2 b2 a3 b3) <-
   (a0 a1 a2 a3) (b0 b1 b2 b3)))

#|
Some notes on how permutations are used.

permute -> Reverse -> back-scan
        -> Split

join -> append
     -> flag-merge -> interleave

pack -> flag-merge
     -> odd-elts

shift
|#
