;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

;;; Interface code between vcode and lisp.
;;; (First pass written by Timothy Freeman)

(in-package :nesl-lisp) 

(defconstant f nil)

(defun lift-quote (thing)
  (if (and (listp thing) (eq (car thing) 'quote))
      (cadr thing)
      thing))

;;; We use structures vcode-vector, vcode-tuple, and vcode-record
;;; instead of simple lists so we can figure out a unique vcode
;;; type for each lisp data structure.  
(defstruct (vcode-vector (:print-function print-vcode-vector))
  type
  data)

(eval-when (compile eval load)
  (defparameter vcode-vector-letter #\u
    "The letter used for low-level vcode-vectors."))

(defun print-vcode-vector (self stream depth)
  (declare (ignore depth))
  (if (and (= (length (vcode-vector-data self)) 1)
	   (not (eql (vcode-vector-type self) 'nesl::vector)))
      (format stream "~s" (elt (vcode-vector-data self) 0))
      (format stream "#~a.~s~:s"
	      vcode-vector-letter
	      (vcode-vector-type self)
	      (coerce (vcode-vector-data self) 'list))))

(defstruct (vcode-tuple (:print-function print-vcode-tuple)) data)

(defstruct (vcode-record (:print-function print-vcode-record)) type data)

(defun print-vcode-record (self stream depth)
  (declare (ignore depth))
  (format stream "~s" (cons (vcode-record-type self)
			    (vcode-record-data self))))

(defun booleanp (x) (not (null (member x '(t nil f)))))

(deftype boolean () '(satisfies booleanp))

;;; This guy can't call vvector or tup because it is called by them.
(defun coerce-lisp-constant (const)
  (cond
   ((booleanp const) (make-vcode-vector :type 'bool :data (list const)))
   ((integerp const) (make-vcode-vector :type 'int :data (list const)))
   ((floatp const)  (make-vcode-vector :type 'nesl::float :data (list const)));;%
   ((characterp const) (make-vcode-vector :type 'char :data (list const)))
   ((floatp const) (make-vcode-vector :type 'float :data (list const)))
   ((stringp const)
    (make-vcode-tuple :data
		      (list
		       (make-vcode-vector
			:type 'nesl::vector
			:data (list (length const)))
		       (make-vcode-vector
			:type 'char
			:data (coerce const 'list)))))
   (t const)))

(defun vcode-type-of (data)
  "Like type-of, except it figures the vcode type."
  (let ((data (coerce-lisp-constant data)))
    (etypecase data
      (vcode-tuple (mapcar #'vcode-type-of (vcode-tuple-data data)))
      (vcode-record (vcode-record-type data))
      (vcode-vector (vcode-vector-type data)))))

(defun vcode-primitive-type-p (type)
  (labels ((mapsymtype (name)
	     (if (and (symbolp type)
		      (or (equal (symbol-name type) name)
			  (equal (symbol-name type) (string (elt name 0)))))
		 (intern name 'nesl)
		 nil)))
    (cond
     ((mapsymtype "INT"))
     ((mapsymtype "FLOAT"))
     ((mapsymtype "BOOL"))
     ((mapsymtype "CHAR"))
     ((mapsymtype "VECTOR"))
     (t nil))))

(defun read-vcode-vector (stream subchar arg)
  (declare (ignore subchar arg))
  `',(multiple-value-call
      (if *read-suppress* #'(lambda (x y) nil) #'vvector)
      (vector-sequence-reader
       stream vcode-vector-letter
       #'(lambda (type)
	   (or (vcode-primitive-type-p type)
	       (nesl-error "~s should be either int, float, bool, or char, and it isn't."
		      type))))))

(eval-when (compile eval load)
  (set-dispatch-macro-character #\# vcode-vector-letter 'read-vcode-vector))

(defun evalmakes (x)
  (let ((x (lift-quote x)))
    (cond
     ((or (not (listp x)) (< (length x) 1)) x)
     ((eq (car x) 'tup)
      (apply #'tup (mapcar #'evalmakes (cdr x))))
     (t (apply #'make (car x) (mapcar #'evalmakes (cdr x)))))))

(defun vector-sequence-reader (stream letter map-type)
  ;; Stream is the stream to read from
  ;; Letter is the letter we're responding to.  (Needed for error messages.)
  ;; map-type is a function that maps the thing after the dot to a
  ;;    valid type, or signals an error if the thing after the dot is not
  ;;    valid.
  ;; We return two values, the type and the data.
  (if (char= (peek-char nil stream) #\.)
      (progn
	(read-char stream t nil t)
	(if *read-suppress*
	    (progn
	      (read stream t nil t)
	      (read stream t nil t)
	      (values nil nil))
	    (values (funcall map-type (read stream t nil t))
		    (mapcar #'evalmakes (read stream t nil t)))))
      (let ((firstread (read stream t nil t)))
	(cond
	 ((equal firstread '())
	  (if *read-suppress*
	      (values nil nil)
	      (nesl-error "Can't distinguish an empty integer vector from an empty ~
                    boolean vector.  Use #~a.int() or #~a.bool() instead."
		     letter letter)))
	 ((not (listp firstread))
	  (if *read-suppress*
	      (values nil nil)
	      (nesl-error "#~a must be followed by a list or .type for some type, not ~s."
		     letter
		     firstread)))
	 (t (if *read-suppress*
		(values nil nil)
		(values (vcode-type-of (evalmakes (car firstread)))
			(mapcar #'evalmakes firstread))))))))

(defun assert-list-types (name type list)
  (dolist (i list)
    (when (not (or (equal (vcode-type-of i) type)
		   (and (eql type 'nesl::vector) 
			(eql (vcode-type-of i) 'int))))
	  (nesl-error
	   "~%Can't make the list ~s into a vector of type ~a.~s,~%~
            because ~s isn't of type ~s."
	   list name type i type))))

(defun vvector (type list)
  (let ((list (coerce list 'list)))
    (assert-list-types #\U type list)
    (make-vcode-vector :type type :data list)))

;;(defun old-vsequence (type list)
;;  (assert-list-types "sequence" type list)
;;  (tup (nesl::vector (coerce-lisp-constant (length list)))
;;       (mash-vcode-things type (mapcar #'coerce-lisp-constant list))))

(defun vsequence (type list)
  (assert-list-types #\V type list)
  (tup (make-vcode-vector :type 'nesl::vector :data (list (length list)))
       (mash-vcode-things type (mapcar #'coerce-lisp-constant list))))

(defun vector-flatten (list)
  (if list
      (append (car list) (vector-flatten (cdr list)))
    nil))

(defun mash-vcode-things (type list)
  (cond
   ((member type '(int nesl::float bool char nesl::vector))
    (vvector type
	     (vector-flatten (mapcar #'(lambda (x)
					 (vcode-vector-data
					  (coerce-lisp-constant x)))
				     list))))
   ((and (listp type) (= (length type) 2))
    (tup (mash-vcode-things (first type) (mapcar #'nesl::first list))
	 (mash-vcode-things (second type) (mapcar #'nesl::second list))))
   ((symbolp type)
    (apply #'make type
	   (apply #'mapcar #'(lambda (type &rest stuff)
			       (mash-vcode-things type stuff))
		  (or (get-type type *definitions*)
		      (nesl-error "The type ~s is undefined." type))
		  (mapcar #'(lambda (x) (vcode-record-data x)) list))))
   (t (error "Bogus type ~s in mash-vcode-things." type))))

(defun nesl::first (tup)
  (first (vcode-tuple-data tup)))

(defun nesl::second (tup)
  (second (vcode-tuple-data tup)))

(eval-when (compile eval load)
(defparameter vcode-sequence-letter #\v
  "The letter used for higher-level nestable vcode-vectors."))

(defun read-vcode-sequence (stream subchar arg)
  (declare (ignore subchar arg))
  `',(multiple-value-call
      (if *read-suppress* #'(lambda (x y) nil) #'vsequence)
      (vector-sequence-reader
       stream vcode-sequence-letter
       #'(lambda (type)
	   (or (vcode-primitive-type-p type) type)))))

(eval-when (compile eval load)
  (set-dispatch-macro-character #\# vcode-sequence-letter 'read-vcode-sequence))

(defun tup (&rest data)
  (make-vcode-tuple :data (mapcar #'coerce-lisp-constant
				  (mapcar #'lift-quote data))))

(defun make (recordtype &rest data)
  (let ((types (get-type recordtype *definitions*))
	(data (mapcar #'coerce-lisp-constant (mapcar #'lift-quote data))))
    (assert (= (length types) (length data)))
    (mapc #'(lambda (type data) (assert (equal type (vcode-type-of data))))
	  types data))
  (make-vcode-record :type recordtype :data data))

(export 'vcode-equal)

(defun vcode-equal (v1 v2)
  (let ((v1 (coerce-lisp-constant v1))
	(v2 (coerce-lisp-constant v2)))
    (cond
     ((and (typep v1 'vcode-vector) (typep v2 'vcode-vector))
      (and (eq (vcode-vector-type v1) (vcode-vector-type v2))
	   (= (length (vcode-vector-data v1))
	      (length (vcode-vector-data v2)))
	   (every
	    (ecase (vcode-vector-type v1)
	      (bool #'(lambda (v1 v2)
			(or (and (eq v1 't) (eq v2 't))
			    (and (not (eq v1 't)) (not (eq v2 't))))))
	      (int #'=)
	      (nesl::float #'=)
	      (nesl::vector #'=)
	      (char #'char=))
	    (vcode-vector-data v1)
	    (vcode-vector-data v2))))
     ((and (typep v1 'vcode-tuple) (typep v2 'vcode-tuple))
      (and (= (length (vcode-tuple-data v1))
	      (length (vcode-tuple-data v2)))
	   (every #'vcode-equal
		  (vcode-tuple-data v1)
		  (vcode-tuple-data v2))))
     ((and (typep v1 'vcode-record) (typep v2 'vcode-record))
      (and (eq (vcode-record-type v1) (vcode-record-type v2))
	   (every #'vcode-equal
		  (vcode-record-data v1)
		  (vcode-record-data v2))))
     (t nil))))


(defparameter *print-vcode-sequences* t
  "Whether to print things of type (vector foo) as #v(...).")

;;; Return nil if thing isn't a vcode sequence, or return the length
;;; and the data if it is a vcode sequence.
(defun vcode-sequence-p (thing)
  (and
   (typep thing 'vcode-tuple)
   (let ((first (nesl::first thing)))
     (and
      (eq (vcode-vector-type first) 'nesl::vector)
      (values t
	      (vcode-vector-data first)
	      (nesl::second thing))))))
   
(defun print-vcode-tuple (self stream depth)
  (declare (ignore depth))
    (if (and *print-vcode-sequences* (vcode-sequence-p self))
	(print-vcode-sequence self stream 0)
	(format stream "~s" (cons 'tup (vcode-tuple-data self)))))

(defparameter *max-print-length* 100)

;;; Prints the which'th element of the thing to stream.  See the test
;;; cases for examples.
(defun print-vcode-sequence (thing stream which)
  (labels ((printbutend (list)
	     (cond
	      ((null list))
	      ((null (cdr list))
	       (print-vcode-sequence (car list) stream which))
	      (t
	       (print-vcode-sequence (car list) stream which)
	       (write-char #\space stream)
	       (printbutend (cdr list)))))
	   (printlist (first list)
	     (format stream "(~s " first)
	     (printbutend list)
	     (format stream ")")))	     
    (multiple-value-bind (vectorp length data) (vcode-sequence-p thing)
      (cond
       (vectorp
	(when (<= (length length) which)
	      (nesl-error "Unprintable Vector....."))
	(labels ((sumthem (list pos)
		   (if (= pos 0) 0
		       (+ (car list) (sumthem (cdr list) (- pos 1))))))
	  (let ((sum (sumthem length which))
		(thislength (elt length which)))
	    (if (equal (vcode-type-of data) 'char)
		(format stream "~s" (coerce
				     (subseq (vcode-vector-data data)
					     sum
					     (+ sum thislength))
				     'string))
		(progn	  
		  (write-string "#v" stream)
		  (when (= thislength 0)
		    (format stream ".~s" (vcode-type-of data)))
		  (write-char #\( stream)
		  (dotimes (i thislength)
		    (print-vcode-sequence data stream (+ i sum))
		    (unless (= i (- thislength 1))
		      (write-char #\space stream))
		    (when (= i *max-print-length*)
		      (write-string "...." stream)
		      (return)))
		  (write-char #\) stream))))))
       ((typep thing 'vcode-vector)
	(when (<= (length (vcode-vector-data thing)) which)
	  (nesl-error "Unprintable Vector....."))
	(format stream "~s" (elt (vcode-vector-data thing) which)))
       ((typep thing 'vcode-tuple) (printlist 'tup (vcode-tuple-data thing)))
       ((typep thing 'vcode-record)
	(printlist (vcode-record-type thing) (vcode-record-data thing)))
       (t (error "Got confused while printing vcode sequence ~s.~%"
		 thing))))))
      
(defun flatten-data (data)
  (let ((data (coerce-lisp-constant data)))
    (etypecase data
      (vcode-tuple (mapcan #'flatten-data (vcode-tuple-data data)))
      (vcode-record (mapcan #'flatten-data (vcode-record-data data)))
      (vcode-vector (list (coerce (vcode-vector-data data) 'vector))))))

(defun flatten-type (type)
  (cond
   ;; Primitive type
   ((not (listp (get-type-fields type *definitions*))) (list type))
   ;; tuple type
   ((listp type) (mapcan #'flatten-type type))
   ;; record type
   ((symbolp type) (mapcan #'flatten-type (get-type type *definitions*)))))

(defun gen-arbitrary-read (type)
  (cond
   ((equal type 'int) '(nesl::read-int))
   ((equal type 'nesl::float) '(nesl::read-float))
   ((equal type 'bool) '(nesl::read-bool))
   ((equal type 'char) '(nesl::read-char))
   ((equal type 'nesl::vector) '(nesl::read-segdes))
   ((listp type) `(tup . ,(mapcar #'gen-arbitrary-read type)))
   ((symbolp type) `(,type . ,(mapcar #'gen-arbitrary-read
				      (get-type type *definitions*))))))

(defun gen-arbitrary-write (variable type)
  (labels ((type-gen (rest-acces-types)
	     (if rest-acces-types
		 `(,@(gen-arbitrary-write
		      `(,(first (car rest-acces-types)) ,variable)
		      (second (car rest-acces-types)))
		     . ,(type-gen (cdr rest-acces-types)))
	       '())))
    (cond
     ;; added by guyb to make floats work 12/5/89
     ((member type '(int nesl::float bool char)
	      :test #'equal)
      `((,(gentemp) (nesl::write ,variable))))
     ((and (listp type) (= (length type) 2))
      (type-gen (mapcar #'list (list 'nesl::first 'nesl::second) type)))
     ((symbolp type)
      (type-gen (get-type-fields type *definitions*)))
     (t (error "Bogus type ~s in gen-arbitrary-write.~%" type)))))
   
;;; Returns two values, the thing grouped and the remainder of the
;;; ungrouped data. 
(defun group-data (data type)
  (labels ((looper (data typelist)
	     "Given a list of types and some ungrouped data,"
	     "Returns a list of the groupings of the data and the"
	     "remainder of the ungrouped data."
	     (if typelist
		 (multiple-value-bind (result moredata)
		     (group-data data (car typelist))
		   (multiple-value-bind (rest evenmoredata)
		       (looper moredata (cdr typelist))
		     (values (cons result rest) evenmoredata)))
		 (values nil data))))
    (cond
     ((member type '(int nesl::float bool nesl::vector))
      (values (vvector type (car data)) (cdr data)))
     ((equal type 'char)
      (values (vvector type (mapcar #'code-char (car data))) (cdr data)))
     ((listp type)
      (multiple-value-bind (result data) (looper data type)
	(values (apply #'tup result) data)))
     ((symbolp type)
      (multiple-value-bind (result data)
	  (looper data (get-type type *definitions*))
	(values (apply #'make type result) data))))))

(defun nesl-typep (x)
  (or (typep x 'vcode-tuple) (typep x 'vcode-vector) (typep x 'vcode-record)))

(defun nesl-value (x)
  (let ((val (coerce-lisp-constant (lift-quote x))))
    (if (nesl-typep val) val 
      (nesl-error "The type ~a is an invalid NESL type in: ~a" (type-of x) x))))

(defparameter *lastfuncall* nil)
(defparameter *laststubname* nil)

(defun call-vcode (functionname &rest args)
  (let* ((args (mapcar #'nesl-value args))
	 (argtypes (mapcar #'vcode-type-of args))
	 (resulttype (get-return-type functionname
				      (conv-type-list argtypes)
				      *definitions*))
	 (new nil))
    (when (not (equal (cons functionname argtypes) *lastfuncall*))
      (let ((stubname (gentemp "STUB"))
	    (argnames (mapcar #'(lambda (x) 
				  (declare (ignore x)) (gentemp)) args)))
	(insert-s-op `(,stubname)
		   `((,resulttype))
		   `(with (,@(mapcar #'(lambda (argname argtype)
					 `(,argname 
					   ,(gen-arbitrary-read argtype)))
				     argnames argtypes))
			  (,functionname . ,argnames))
		   *definitions*)
	(setq *laststubname* stubname)
	(setq new t)))
    (setq *lastfuncall* nil)
    (let ((result (values 
		   (group-data
		    (apply 'run-vcode *laststubname* (flatten-type resulttype)
			   new (mapcan #'flatten-data args))	     
		    resulttype))))
      (setq *lastfuncall* (cons functionname argtypes))
      result)))

(defparameter *interp* nil)

(defparameter *varlist* nil)

(defparameter *top-level-commands* 
  '(defop defrec load nesl::odefop nesl::sdefop progn nesl::ndefop
     nesl::use-machine nesl::cm-finger nesl::set-argcheck
     nesl::configuration nesl::set-memsize nesl::set-username))

(defparameter *top-level-funcs* '())

(defparameter *special-forms* 
  '(nesl::with nesl::over nesl::with nesl::if nesl::time nesl::tup
	       nesl::defop nesl::defrec nesl::first nesl::second))

(defparameter *exit-commands*
  '(nesl::exit nesl::lisp))

(defun describe-nesl (funname)
  (let ((fdef nil))
    (cond ((member funname *special-forms*)
	   (format t "~a is a special form; please see the NESL manual." 
		   funname))
	  ((member funname *top-level-commands*)
	   (format t "~a is a top level command; please use (help)." 
		   funname))
	  ((setq fdef 
		 (find funname *oplist* :key 
		       #'(lambda (k) (if (listp (cadr k)) (caadr k) nil))))
	   (let ((interface (second fdef))
		 (type (third fdef))
		 (documentation (fourth fdef)))
	     (format t "INTERFACE:~% ~a~%~%" (cadr fdef))
	     (when type (format t "TYPE:~% ~a~%~%" type))
	     (when documentation 
		   (format t "DOCUMENTATION:~%~% ~a~%" documentation))))
	  (t (nesl-error "Function ~a not found" funname)))
    funname))

(defun nesl-help ()
  (format t "Type any valid NESL form, or one of the following:~%  ~
    (DESCRIBE funname)    -- Describe a NESL function with funname.~%  ~
    (SET varname form)    -- Set the variable to result of form.~%  ~
    (LOAD filename)       -- Load a file.~%  ~
    (VERBOSE)             -- Toggle the verbose switch.~%  ~
    (SET-PRINT-LENGTH n)  -- Set the maximum number of elements that are
                             printed in a sequence.~%  ~
    (CONFIGURATION)       -- List the current configuration properties.~%  ~
    (USE-MACHINE config)  -- Use the machine with configuration CONFIG.~%  ~
    (SET-MEMSIZE n)       -- Set the memory size of the current configuration.~%  ~
    (LISP) or (EXIT)      -- Exit NESL and go to the Lisp interpreter.~%  ~
    (HELP)                -- Print this message."))

(defun add-to-variable-list (var-symbol value)
  (push (cons var-symbol value) *varlist*)
  value)

(defun get-from-variable-list (var-symbol)
  (cdr (assoc var-symbol *varlist*)))

(defmacro nesl::set (symbol value)
  `(nesl-lisp::add-to-variable-list ',symbol ,value))

(defun eval-toplevel (form)
  (if (listp form)
      (cond ((member (car form) *top-level-commands*)
	     (eval form))
	    ((member (car form) *top-level-funcs*)
	     (apply (car form) (mapcar 'eval-nesl (cdr form))))
	    ((member (car form) *exit-commands*)
	     :exit)
	    ((eql (car form) 'nesl::set)
	     (add-to-variable-list (second form) (eval-nesl (third form))))
	    ((eql (car form) 'nesl::help)
	     (nesl-help))
	    ((eql (car form) 'nesl::describe)
	     (describe-nesl (second form)))
	    ((eql (car form) 'nesl::set-print-length)
	     (setq *max-print-length* (second form)))
	    ((eql (car form) 'nesl::setq)
	     (nesl-error "SETQ is no longer valid, use SET instead."))
	    ((eql (car form) 'nesl::verbose)
	     (if (setq *verbose* (not *verbose*))
		 (format t "Verbose: On~%")
	       (format t "Verbose: Off~%")))
	    (t (eval-nesl form)))
      (eval-nesl form)))

;; This adds a line to the beginning and end of the code that prints
;; when interpretation is starting and ending.
(defun verbose-wrapper (form)
  (if *verbose*
   `(nesl::with ((start (nesl::print-string "Start of interpretation..."))
		 (cr (nesl::print-char #\newline))
		 (result ,form)
		 (end (nesl::print-string "End of interpretation..."))
		 (cr (nesl::print-char #\newline)))
		result)
   form))

(defun eval-nesl (form)
  (cond ((constantp form) (nesl-value form))
	((listp form)
	 (let* ((stack '((op-hold)))
		(l-c (trans-exp (conv-exp (verbose-wrapper form) nil nil) 
				stack *definitions*))
		(code (reverse (cons (list 'RET) (cdr l-c))))
		(result-type (car l-c)))
	   (add-op-def 'main (list result-type) code *definitions*)
	   (group-data (run-vcode 'main (flatten-type result-type) t nil)
		       result-type)))
	((symbolp form)
	 (or (get-from-variable-list form)
	     (nesl-error "~a is unbound" form)))
	(t (nesl-error "Invalid form: ~a" form))))
	 

(defun nesl ()
  (let ((*package* (find-package 'nesl))
	(*type-map* #'(lambda (x) x))
	(*interp* t))
    (loop
     (format t "~%<Nesl> ")
     (catch 'nesl-error 
       (let ((val (eval-toplevel (read))))
         (if (eql val :exit) 
	     (return)
	   (format t "~%~a" val)))))))

(defun user::nesl () (nesl))

(defun load-nesl (filename)
  (let ((*package* (find-package 'nesl))
	(*interp* t))
    (with-open-file (loadstr filename :direction :input)
      (loop
       (catch 'nesl-error
	 (let ((val (read loadstr nil :eof)))
	   (if (eql val :eof) (return)
	     (eval-toplevel val))))))))
