;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

(defparameter *current-fundef* nil)
(defparameter *type-map* nil)

(defun nesl-error (format-string &rest args)
  (when *current-fundef*
    (format t "In (DEFOP ~a ...)~%" *current-fundef*))
  (if *interp*
      (progn 
	(apply 'format t format-string args)
	(throw 'nesl-error :error))
    (apply 'error format-string args)))

(defun l-from-type (type definitions)
  (cond ((null type) 0)
	((atom type)
	 (cond ((get-type-length type definitions))
	       (t (nesl-error "~s is not a defined type." type))))
	((listp type)
	 (when (not (= (length type) 2))
	       (nesl-error "A tuple type must contain two elements, ~s" type))
	 (+ (l-from-type (first type) definitions)
	    (l-from-type (second type) definitions)))
	(t (nesl-error "Invalid type declaration ~s." type))))

(defun l-from-type-list (type-list definitions)
  (reduce #'+ (mapcar #'(lambda (a) (l-from-type a definitions))
		      type-list)))

(defun size-depth-type (symbol list definitions)
  (do* ((depth 0 (+ size depth))
	(rem-list list (cdr rem-list))
	(elt (car rem-list) (car rem-list))
	(size (l-from-type (second elt) definitions)
	      (l-from-type (second elt) definitions)))
       ((or (eql symbol (first elt)) (null (cdr rem-list)))
	(if (eql symbol (first elt))
	    (list size depth (second elt)) nil))))

(defun copy-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (if sdt
	(cons (third sdt) (list (list 'COPY (first sdt) (second sdt))))
      (let ((const (assoc symbol *varlist*)))
	(if const (trans-constant (cdr const))
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun delete-type (symbol list definitions)
  (do* ((rem-list list (cdr rem-list)))
       ((or (eql symbol (first (first rem-list))) (null (cdr rem-list)))
	(if (eql symbol (first (first rem-list)))
	    (rplacd (first rem-list) nil)))))

(defun copy-pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt
	(cons (third sdt) (list (list 'POP (first sdt) 
				      (+ (first sdt) (second sdt)))
				(list 'COPY (first sdt) (second sdt))))
      (let ((const (assoc symbol *varlist*)))
	(if const (trans-constant (cdr const))
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt (list 'POP (first sdt) (second sdt))
      (let ((const (assoc symbol *varlist*)))
	(if const nil
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun fun-def (fname arg-type definitions)
  (let ((return-type
	 (cond ((not (symbolp fname))
		(nesl-error "Function name ~s must be a symbol." fname))
	       ((get-return-type fname arg-type definitions))
	       ((eql 'tup fname) arg-type)
	       ((eql 'nesl::first fname)
		(if (not (listp (first arg-type)))
		    (nesl-error 
		     "First applied to type ~s which is not a tuple."
		     (first arg-type))
		  (first (first arg-type))))
	       ((eql 'nesl::second fname)
		(if (not (listp (first arg-type)))
		    (nesl-error 
		     "Second applied to type ~s which is not a tuple."
		     (first arg-type))
		  (second (first arg-type))))
	       ((assoc fname *funlist*)
		;(format t "Compiling function ~s on type ~a~%" fname arg-type)
		(let* ((func (cdr (assoc fname *funlist*))))
		  (when (eql (car func) :type)
			(typecheck (caaddr func) (cadr func) arg-type)
			(setq func (cddr func)))
		  (let* ((name (car func))
			 (*current-fundef* fname)
			 (body (conv-body (cdr name) (cdr func) nil))
			 (compiled (trans-op (cdr name) arg-type
					     body definitions))
			 (return-type (car compiled))
			 (code (cdr compiled)))
		    (add-op-def fname (cons return-type arg-type)
				code definitions)
		    return-type)))
	       (t (let ((valid-types (types-of-name fname definitions)))
		    (if valid-types
			(nesl-error "There is no definition for function ~a ~
                          with source type:~%    ~s.~%  ~
                          The only valid source types are:~%~{    ~a~%~}"
				    fname (funcall *type-map* arg-type)
				    (mapcar *type-map* (mapcar 'cddr valid-types)))
		      (nesl-error "Function ~a is undefined." fname))))))
	(call
	 (cond ((eql 'nesl::first fname)
		(list (list 'POP (l-from-type (second (first arg-type))
					      definitions)
			    0)))
	       ((eql 'nesl::second fname)
		(list (list 'POP
			    (l-from-type (first (first arg-type))
					 definitions)
			    (l-from-type (second (first arg-type))
					 definitions))))
	       ((eql 'tup fname))
	       ((primitivep fname arg-type definitions)
		(reverse (get-full-name fname arg-type definitions)))
	       ((get-return-type fname arg-type definitions)
		(list (list 'CALL (cons fname arg-type))))
	       (t (nesl-error "Internal error, please report bug")))))
    (cons return-type call)))

(defun trans-func (list stack definitions)
  (let* ((arg-t-c (trans-list (cdr list) stack definitions)) 
	 (arg-type (car arg-t-c))
	 (arg-code (cdr arg-t-c))
	 (fun-t-c (fun-def (first list) arg-type definitions)))
    (nconc fun-t-c arg-code)))

(defun trans-if (exp stack definitions)
  (let* ((cond-l-c (trans-exp (second exp) stack definitions))
	 (bstack (copy-tree stack))
	 (if-pops (mapcar #'(lambda (var) (pop-stack var stack definitions))
			  (set-difference (sixth exp) (fifth exp))))
	 (if-l-c (trans-exp (third exp) stack definitions))
	 (else-pops (mapcar #'(lambda (var) (pop-stack var bstack definitions))
			  (set-difference (fifth exp) (sixth exp))))
	 (else-l-c (trans-exp (fourth exp) bstack definitions)))
    (when (not (equal (car if-l-c) (car else-l-c)))
	  (nesl-error "If and else part must be of same type for ~s." exp))
    (when (not (equal (car cond-l-c) 'bool))
	  (nesl-error "~a in ~s is a ~s instead of a BOOL."
		 (second exp) exp (car cond-l-c) ))
    (cons (car if-l-c)
	  (nconc
	    (list (list 'ENDIF))
	    (cdr else-l-c)
	    (reverse else-pops)
	    (list (list 'ELSE))
	    (cdr if-l-c)
	    (reverse if-pops)
	    (list (list 'IF))
	    (cdr cond-l-c)))))

(defun match-names (names type)
  (if (listp names)
      (if (listp type)
	  (let ((first-match (match-names (first names) (first type)))
		(second-match (match-names (second names) (second type))))
	    (if (or (eql first-match :error) (eql second-match :error))
		:error
	      (append first-match second-match)))
	    :error)
    (list (list names type))))

(defun trans-with (exp stack definitions)
  (when (/= (length exp) 3)
	(nesl-error 
	 "A WITH form must contain one binding list and one expression"))
  (do* ((rem (second exp) (cdr rem))
	(code nil)
        (nstack (cons '(with-hold) stack)))
       ((not rem)
	(let ((body-l-c (trans-exp (third exp) nstack definitions)))
	  (cons (first body-l-c)
		(let ((depth (second (size-depth-type 'with-hold 
						      nstack definitions)))
		      (fullcode (nconc (cdr body-l-c) code))
		      (resultsize (l-from-type (first body-l-c) definitions)))
		  (if (plusp depth)
		      (cons (list 'POP depth resultsize) fullcode)
		    fullcode)))))
       (let* ((binding (car rem))
	      (exp (trans-exp (second binding) nstack definitions))
	      (exp-type (car exp))
	      (exp-code (cdr exp))
	      (name (first binding))
	      (matched-names (match-names name exp-type)))
	 (when (eql matched-names :error)
	       (nesl-error "Pattern ~a does not match assigned value ~a,~%~
                        which is of type ~a."
			   name (second binding) exp-type))
	 (setq nstack (append (reverse matched-names) nstack))
	 (let ((unused-pops 
		(mapcar #'(lambda (var) (pop-stack var nstack definitions))
			(cddr binding))))
	   (setq code (nconc (nconc (reverse unused-pops) exp-code) code))))))

(defun trans-constant (constant)
  (let ((vector (nesl-value constant)))
    (cons (vcode-type-of vector)
	  (list (list 'VCONST (flatten-type (vcode-type-of vector))
		              (flatten-data vector))))))

(defun trans-list (list stack definitions)
  (do* ((type nil)
	(rem list (cdr rem))
	(nstack stack)
	(code nil))
       ((not rem) (cons (reverse type) code))
       (let ((l-c (trans-exp (car rem) nstack definitions)))
	 (setq nstack (cons (list 0 (car l-c)) nstack))
	 (setq code (nconc (cdr l-c) code))
	 (setq type (cons (car l-c) type)))))

(defun trans-timer (expression stack definitions)
   (let ((subparse (trans-exp (second expression) stack definitions)))
     (cons (list (car subparse) 'nesl::float)
	   (append (list '(STOP_TIMER))
		   (cdr subparse)
		   (list '(START_TIMER))))))

(defun trans-typecase (expression stack definitions)
  (let* ((type (third (size-depth-type (second expression) 
				       stack definitions)))
	 (scalar-code (third expression))
	 (vector-code (fourth expression))
	 (tuple-code (fifth expression)))
    (trans-exp (cond ((and (atom type)
			   (atom (get-type-fields type definitions)))
		      scalar-code)
		     ((and (listp type) (eql (first type) 'nesl::vector))
		      vector-code)
		     ((listp type) tuple-code)
		     (t (nesl-error 
			 "Records don't work in trans-typecase.~%~
                    You probably havn't created one of the primitives for ~a."
			 type)))
	       stack definitions)))

(defun trans-exp (expression stack definitions)
  (cond ((or (constantp expression) (eql expression 'f))
	 (trans-constant expression))
	((symbolp expression)
	 (copy-stack expression stack definitions))
	((listp expression)
	 (cond ((eql (car expression) 'last)
		(copy-pop-stack (second expression) stack definitions))
	       ((eql (car expression) 'if)
		(trans-if expression stack definitions))
	       ((eql (car expression) 'with)
		(trans-with expression stack definitions))
	       ((eql (car expression) 'nesl::time)
		(trans-timer expression stack definitions))
	       ((eql (car expression) 'nesl::typecase)
		(trans-typecase expression stack definitions))
	       (t (trans-func expression stack definitions))))
	(t (nesl-error "Invalid expression, ~s." expression))))

(defun match-names-check (name type)
  (let ((matched-names (match-names name type)))
    (when (eql matched-names :error)
	  (nesl-error "Pattern ~a does not match specified type ~a."
		      name type))
    matched-names))

(defun trans-op (arguments source-type body definitions)
  (when (not (equal (length arguments) (length source-type)))
	(nesl-error 
	 "Number of source types does not match number of arguments.~%   ~
                 Arguments:    ~a~%   ~
                 Source Types: ~s"
	 (funcall *type-map* arguments)
	 (funcall *type-map* source-type)))
  (let ((stack (reverse (cons '(op-hold) (mapcan #'match-names-check
						 arguments source-type)))))
    (let* ((l-c (trans-exp body stack definitions))
	   (result-type (car l-c)))
      (cons result-type
	    (reverse 
	     (cons (list 'RET)
		   (let ((depth (second (size-depth-type 'op-hold 
							 stack definitions))))
		     (if (plusp depth)
			 (cons (list 'POP depth 
				     (l-from-type result-type definitions))
			       (cdr l-c))
		       (cdr l-c)))))))))

(defun insert-op (names types body definitions)
  (do* ((fname (first names))
	(nbody (cdr (free-exp body nil)))
	(rem types (cdr rem))
	(type (car rem) (car rem)))
       ((null rem) types)
       ;; this first one is to allow recursive routines
       (add-op-def fname type 'stub definitions)
       (let* ((compiled (trans-op (cdr names) (cdr type) nbody definitions))
	     (return-type (car compiled))
	     (code (cdr compiled)))
	 (when (not (equal return-type (car type)))
	   (nesl-error 
	    "Inferred body type and declared return type don't match.~%    ~
                Body type:            ~s~%    ~
                Declared return type: ~s"
	    return-type
	    (car type)))
	 (add-op-def fname type code definitions))))

(defun insert-rec (name fields definitions)
  (let* ((source-types (mapcar #'second fields))
	 (rfields (reverse fields))
	 (size (l-from-type-list source-types definitions)))

    ;; add the definition
    (add-type-def name size fields definitions)

    ;; add the constructor
    (add-op-def (pname name) (cons name (append source-types
						(list 'nesl::vector)))
		'((POP 1 0) ;;JJ
		  (RET)) definitions)

    (add-op-def name (cons name source-types) '((RET)) definitions)

    ;; add the accessors
    (dolist (field rfields)
      (let ((sdt (size-depth-type (first field) rfields definitions)))
	(add-op-def (pname (first field)) (list (second field) name
						'nesl::vector)
		    `((POP 1 0) ;;JJ
		      (COPY ,(first sdt) ,(second sdt))
		      (POP ,size ,(first sdt)) (RET))
		    definitions)
	(add-op-def (first field) (list (second field) name)
		    `((COPY ,(first sdt) ,(second sdt))
		      (POP ,size ,(first sdt)) (RET))
		    definitions)))))
