;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF tHIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl)

(odefop (len segdes) ((int <- vector))
  ((LENGTHS))
  :primitive t :pversion nil)

(odefop (vector int) ((vector <- int))
  ((MAKE_SEGDES))
  :primitive t :pversion nil)

(odefop (prim-dist v i) ((int <- int vector)
			(float <- float vector)
			(char <- char vector)
			(bool <- bool vector))
  ((DIST INT) (DIST FLOAT) (DIST INT) (DIST BOOL))
  :primitive t :pversion nil)

;; The following is just a stub, the actual definition is in
;; vector-ops
(odefop (prim-+-reduce a s) ((int <- int vector)) 0)

;; this is temporary
(defop (prim-dist A L) (vector <- vector vector) A)

(odefop (prim-length v) ((int <- int)
			(int <- float)
			(int <- char)
			(int <- bool))
  ((LENGTH INT) (LENGTH FLOAT) (LENGTH INT) (LENGTH BOOL))
  :primitive t :pversion nil)

(opsection "Scalar Functions" "\\seclabel{scalar-ops}")

(opsubsection "Logical Functions"
"All the logical functions work on either integers or booleans.
In the case of integers, they work bitwise over
the bit representation of the integer.")

(odefop (not a) ((bool <- bool)
		(int <- int))
  ((NOT BOOL) (NOT INT))
  :primitive t
  :documentation
  "Returns the logical inverse of the argument.  For integers, this
is the ones complement."
  :type ((alpha <- alpha) (alpha logical))
  :document yes)

(odefop (or a b) ((bool <- bool bool)
		  (int <- int int))
  ((OR BOOL) (OR INT))
  :primitive t
  :documentation
  "Returns the inclusive or of the two arguments."
  :type ((alpha <- alpha alpha) (alpha logical))
  :document yes)

(odefop (and a B) ((bool <- bool bool)
		  (int <- int int))
  ((AND BOOL) (AND INT))
  :primitive t
  :documentation
  "Returns the logical and of the two arguments."
  :type ((alpha <- alpha alpha) (alpha logical))
  :document yes)

(odefop (xor a b) ((bool <- bool bool)
		  (int <- int int))
  (or (and a (not b)) (and b (not a)))
  :documentation
  "Returns the exclusive or of the two arguments."
  :type ((alpha <- alpha alpha) (alpha logical))
  :document yes)

(odefop (nor a b) ((bool <- bool bool) (int <- int int))
  (not (or a b))
  :documentation
  "Returns the inverse of the inclusive or of the two arguments."
  :type ((alpha <- alpha alpha) (alpha logical))
  :document yes)

(odefop (nand a b) ((bool <- bool bool) (int <- int int))
  (not (and a b))
  :documentation
  "Returns the inverse of the and of the two arguments."
  :type ((alpha <- alpha alpha) (alpha logical))
  :document yes)

(opsubsection "Comparison Functions" 
"All comparison functions work on integers, floats and characters.")

(odefop (= a b) ((bool <- int int)
		(bool <- float float)
		(bool <- char char))
  ((= INT) (= FLOAT) (= INT))
  :primitive t
  :documentation
  "Returns \\farg{t} if the two arguments are equal."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (/= a b) ((bool <- int int)
		 (bool <- float float)
		 (bool <- char char))
  ((!= INT) (!= FLOAT) (!= INT))
  :primitive t
  :documentation
  "Returns \\farg{t} if the two arguments are not equal."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (< a b) ((bool <- int int)
		(bool <- float float)
		(bool <- char char))
  ((< int) (< float) (< int))
  :primitive t
  :documentation
  "Returns \\farg{t} if the first argument is strictly less than the second argument."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (> a b) ((bool <- int int)
		(bool <- float float)
		(bool <- char char))
  ((> INT) (> FLOAT) (> INT))
  :primitive t
  :documentation
  "Returns \\farg{t} if the first argument is strictly 
greater than the second argument."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (<= a b) ((bool <- int int)
		 (bool <- float float)
		 (bool <- char char))
  ((<= INT) (<= FLOAT) (<= INT))
  :primitive t
  :documentation
  "Returns \\farg{t} if the first argument is less than or equal to the second
argument."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (>= a b) ((bool <- int int)
		 (bool <- float float)
		 (bool <- char char))
  ((>= INT) (>= FLOAT) (>= INT))
  :primitive t
  :documentation
  "Returns \\farg{t} if the first argument is greater or equal to the second
argument."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (select flag a b) ((bool <- bool bool bool)
			  (float <- bool float float)
			  (char <- bool char char)
			  (int <- bool int int))
  ((SELECT BOOL) (SELECT FLOAT) (SELECT INT) (SELECT INT))
  :primitive t
  :documentation
  "Returns the second argument if the flag is \\farg{t} and
the third argument if the flag is F."
  :type ((alpha <- bool alpha alpha) (alpha any))
  :document yes)

(opsubsection "Predicates" "")

(odefop (plusp v) ((bool <- int))
  (> v 0)
  :documentation
  "Returns \\farg{t} if \\farg{v} is strictly greater than 0."
  :type ((bool <- alpha) (alpha number))
  :document yes)

(odefop (plusp v) ((bool <- float))
  (> v 0.0))

(odefop (minusp v) ((bool <- int))
  (< v 0)
  :documentation
  "Returns \\farg{t} if \\farg{v} is strictly less than 0."
  :type ((bool <- alpha) (alpha number))
  :document yes)

(odefop (minusp v) ((bool <- float))
  (< v 0.0))

(odefop (zerop v) ((bool <- int))
  (= v 0)
  :documentation
  "Returns \\farg{t} if \\farg{v} is equal to 0."
  :type ((bool <- alpha) (alpha number))
  :document yes)

(odefop (zerop v) ((bool <- float))
  (= v 0.0))

(odefop (oddp v) ((bool <- int))
  (plusp (and 1 v))
    :documentation
  "Returns \\farg{t} if \\farg{v} is odd (not divisible by two)."
  :type ((bool <- int))
  :document yes)

(odefop (evenp v) ((bool <- int))
  (zerop (and 1 v))
  :documentation
  "Returns \\farg{t} if \\farg{v} is even (divisible by two)."
  :type ((bool <- int))
  :document yes)

(opsubsection "Arithmetic Functions" "")

(odefop (+ a b) ((int <- int int)
		(float <- float float))
  ((+ INT) (+ FLOAT))
  :primitive t
  :documentation
  "Returns the sum of the two arguments."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

(odefop (- a b) ((int <- int int)
		(float <- float float))
  ((- INT) (- FLOAT))
  :primitive t
  :documentation
  "Subtracts the second argument from the first."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

(odefop (negate v) ((int <- int))
  (- 0 v)
  :documentation
  "Negates a number."
  :type ((alpha <- alpha) (alpha number))
  :document yes)

(odefop (negate v) ((float <- float))
  (- 0.0 v))

(odefop (abs x) ((int <- int)
		(float <- float))
  (select (plusp x) x (negate x))
  :documentation
  "Returns the absolute value of the argument."
  :type ((alpha <- alpha) (alpha number))
  :document yes)

(odefop (diff x y) ((int <- int int)
		  (float <- float float))
  (abs (- x y))
  :documentation
  "Returns the absolute value of the difference of the two arguments."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

(odefop (max a b) ((int <- int int)
		  (float <- float float)
		  (char <- char char))
  (select (> a b) a b)
  :documentation
  "Returns the argument that is greatest (closest to positive infinity)."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

(odefop (min a b) ((int <- int int)
		  (float <- float float)
		  (char <- char char))
  (select (< a b) a b)
  :documentation
  "Returns the argument that is least (closest to negative infinity)."
  :type ((alpha <- alpha alpha) (alpha ordinal))
  :document yes)

#|
(odefop (oldmax a b) ((int <- int int)
		  (float <- float float)
		  (char <- char char))
  ((max int) (max float) (max int))
  :primitive t)

(odefop (oldmin a b) ((int <- int int)
		  (float <- float float)
		  (char <- char char))
  ((min int) (min float) (min int))
  :primitive t)
|#

(odefop (* v d) ((int <- int int)
		(float <- float float))
  ((* INT) (* FLOAT))
  :primitive t
  :documentation
  "Returns the product of the two arguments."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

#|
(odefop (div V D) ((int <- int int)
		  (float <- float float))
  ((/ INT) (/ FLOAT))
  :primitive t)
|#

(odefop (/ v d) ((int <- int int)
		(float <- float float))
  ((/ INT) (/ FLOAT))
  :primitive t
  :documentation
  "Returns \\farg{v} divided by \\farg{d}.
If the arguments are integers, the result is truncated
towards 0."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

(odefop (rem v d) ((int <- int int)
		  (float <- float float))
  ((% INT) (% FLOAT))
  :primitive t
  :documentation
  "Returns the remainder after dividing \\farg{v} by \\farg{d}."
  :type ((alpha <- alpha alpha) (alpha number))
  :document yes)

(odefop (mod v d) ((int <- int int))
  (rem (+ v (* d (- 1 (/ v d)))) d)
  :documentation
  "Returns the modulus of the two arguments.")

(odefop (reciprocal v) ((float <- float))
  (/ 1.0 v)
  :documentation
  "Returns the reciprocal of \\farg{v}."
  :type ((float <- float)))

(odefop (lshift a b) ((int <- int int))
  ((LSHIFT))
  :primitive t
  :documentation
  "Returns the first argument logically shifted to the left by the integer
contained in the second argument.  Shifting will fill with 0-bits."
  :type ((int <- int int))
  :document yes)

(odefop (rshift a b) ((int <- int int))
  ((RSHIFT))
  :primitive t
  :documentation
  "Returns the first argument logically shifted to the right by the integer
contained in the second argument.  Shifting will fill with 0-bits or the
sign bit, depending on the implementation."
  :type ((int <- int int))
  :document yes)

(odefop (sqrt v) ((float <- float))
  ((SQRT))
  :documentation
  "Returns the square root of the argument.  The argument must be nonnegative."
  :type ((float <- float))
  :document yes
  :primitive t)

;; these are stubs.
(odefop (float v) ((float <- int)) ((I_TO_F)) :primitive t)
(odefop (floor v) ((int <- float)) ((FLOOR)) :primitive t)

(odefop (isqrt v) ((int <- int))
  (floor (sqrt (float v)))
  :documentation
  "Returns the greatest integer less than or equal to the exact square root of the integer argument."
  :type ((int <- int int))
  :document yes)

(odefop (ln v) ((float <- float))
  ((LOG))
  :documentation
  "Returns the natural log of the argument."
  :type ((float <- float))
  :document yes
  :primitive t)

(odefop (log v b) ((float <- float float)) 
  (/ (ln v) (ln b))
  :documentation
  "Returns the logarithm of \\farg{v} in the base \\farg{b}."
  :type ((float <- float float))
  :document yes)

(odefop (exp v) ((float <- float))
  ((EXP))
  :documentation
  "Returns {\\it e} raised to the power \\farg{v}."
  :type ((float <- float))
  :document yes
  :primitive t)

(odefop (expt v p) ((float <- float float))
  (exp (* (ln v) p))
  :documentation
  "Returns \\farg{v} raised to the power \\farg{p}."
  :type ((float <- float float))
  :document yes)

(set pi 3.14159)

(odefop (sin v) ((float <- float))
  ((SIN))
  :primitive t
  :documentation
  "Returns the sine of \\farg{v}, where \\farg{v} is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (cos v) ((float <- float))
  ((COS))
  :primitive t
  :documentation
  "Returns the cosine of \\farg{v}, where \\farg{v} is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (tan v) ((float <- float))
  ((TAN))
  :primitive t
  :documentation
  "Returns the tangent of \\farg{v}, where \\farg{v} is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (asin v) ((float <- float))
  ((ASIN))
  :primitive t
  :documentation
  "Returns the arc sine of \\farg{v}. The result is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (acos v) ((float <- float))
  ((ACOS))
  :primitive t
  :documentation
  "Returns the arc cosine of \\farg{v}. The result is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (atan v) ((float <- float))
  ((ATAN))
  :primitive t
  :documentation
  "Returns the arc tangent of \\farg{v}. The result is in radians."
  :type ((float <- float)) 
  :document yes)

(odefop (sinh v) ((float <- float))
  ((SINH))
  :primitive t
  :documentation
  "Returns the hyperbolic sine of \\farg{v} ($(e^x - e^{-x})/2$)."
  :type ((float <- float)) 
  :document yes)

(odefop (cosh v) ((float <- float))
  ((COSH))
  :primitive t
  :documentation
  "Returns the hyperbolic cosine of \\farg{v} ($(e^x + e^{-x})/2$)."
  :type ((float <- float)) 
  :document yes)

(odefop (tanh v) ((float <- float))
  ((TANH))
  :primitive t
  :documentation
  "Returns the hyperbolic tangent of \\farg{v} ($(e^x - e^{-x})/(e^x + e^{-x})$)."
  :type ((float <- float)) 
  :document yes)

;;(odefop (pow a) ((int <- int)) 
;;  (round (exp (* (float a) (ln 2.0)))))

(opsubsection "Conversion Functions" "")

(odefop (btoi a) ((int <- bool))
  ((B_TO_I))
  :primitive t
  :documentation
  "Converts the boolean values \\farg{t} and \\farg{f} into 1 and 0, respectively."
  :type ((int <- bool))
  :document yes)

(odefop (code-char a) ((char <- int))
  ((POP 0 0))
  :primitive t
  :documentation
  "Converts an integer to a character.  The integer must be the
code for a valid character."
  :type ((char <- int))
  :document yes)

(odefop (char-code a) ((int <- char))
  ((POP 0 0))
  :primitive t
  :documentation
  "Converts a character to its integer code."
  :type ((int <- char))
  :document yes)

(odefop (float v) ((float <- int))
  ((I_TO_F))
  :documentation
  "Converts an integer to a floating-point number."
  :primitive t
  :type ((float <- int))
  :document yes)

(odefop (ceil v) ((int <- float))
  ((CEIL))
  :documentation
  "Converts a floating-point number to an integer by truncating toward positive infinity."
  :primitive t
  :type ((int <- float))
  :document yes)

(odefop (floor v) ((int <- float))
  ((FLOOR))
  :documentation
  "Converts a floating-point number to an integer by truncating toward negative infinity."
  :primitive t
  :type ((int <- float))
  :document yes)

(odefop (trunc v) ((int <- float))
  ((TRUNC))
  :documentation
  "Converts a floating-point number to an integer by truncating toward zero."
  :primitive t
  :type ((int <- float))
  :document yes)

(odefop (round v) ((int <- float))
  ((ROUND))
  :documentation
  "Converts a floating-point number to an integer by rounding to the nearest integer; if the number is exactly halfway between two integers, then it is implementation specific to which integer it is rounded."
  :primitive t
  :type ((int <- float))
  :document yes)

(opsubsection "Other Scalar Functions" "")

(odefop (rand v) ((int <- int))
  ((RAND))
  :primitive t
  :documentation
  "Returns a random integer value between 0 and \\farg{v}."
  :type ((int <- int)) 
  :document yes)

