;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

(defun bindvars2 (varlist)
  (if (atom varlist) (list varlist)
    (append (bindvars2 (first varlist)) 
	    (bindvars2 (second varlist)))))

(defun ext-bindings (bindings env)
  (if (null bindings) nil
    (append (ext-exp (second (car bindings)) env)
	    (ext-bindings (cdr bindings) 
			  (append (bindvars2 (first (car bindings))) env)))))

(defun ext-with (exp env)
  (append (ext-bindings (second exp) env)
	  (ext-exp (third exp) (append (mapcan 
					#'(lambda (x) (bindvars2 (first x))) 
					(second exp)) env))))

(defun ext-func (exp env)
  (ext-list (cdr exp) env))

(defun ext-symbol (exp env)
  (if (not (find exp env)) (list exp) nil))

(defun ext-over (exp env)
  (ext-with exp env))

(defun ext-if (exp env)
  (ext-list (cdr exp) env))

(defun ext-list (list env)
  (if (null list) nil
    (append (ext-exp (car list) env)
	    (ext-list (cdr list) env))))

(defun ext-exp (exp env)
  (cond ((constantp exp) nil)
	((symbolp exp)
	 (ext-symbol exp env))
	((listp exp)
	 (let ((nexp (convert-vexp exp)))
	   (cond ((eql (car nexp) 'if)
		  (ext-if nexp env))
		 ((eql (car nexp) 'with)
		  (ext-with nexp env))
		 ((eql (car nexp) 'over)
		  (ext-with nexp env))
		 (t (ext-func nexp env)))))
	(t (nesl-error "Invalid expression, ~s." exp))))

(defun vprefixp (symbol)
  (let ((sstring (string symbol)))
    (and (>= (length sstring) 2)
	 (string-equal "v." symbol :end2 2))))

(defun add-vprefix (symbol)
  (let ((sstring (string symbol)))
    (intern (format nil "v.~s" sstring) (symbol-package symbol))))

(defun vpostfix (symbol)
  (let ((value (read-from-string (subseq (string symbol) 2))))
    (cond ((numberp value) value)
          ((symbolp value) (intern (string value) (symbol-package symbol)))
          (t (nesl-error "~s is an invalid form" symbol)))))

(defun convert-vexpt-list (list)
  (cond ((null list) (cons nil nil))
	((eql (car list) 'v.)
	 (let ((rest (convert-vexpt-list (cddr list))))
	   (cons (car rest) (cons (cadr list) (cdr rest)))))
	(t
	 (let ((rest (convert-vexpt-list (cdr list)))
	       (this (if (and (symbolp (car list)) (vprefixp (car list)))
			 (cons nil (vpostfix (car list)))
		       (let ((sym (gensym "ARG-")))
			 (cons (list (list sym (car list))) sym)))))	
	   (cons (append (car this) (car rest))
		 (cons (cdr this) (cdr rest)))))))

(defun convert-vexp (exp)
  (if (and (symbolp (car exp)) (vprefixp (car exp)))
      (let ((conv (convert-vexpt-list (cdr exp)))
	    (op (vpostfix (car exp))))
	(when (not (car conv))
	  (nesl-error "In expression ~s,~%~
             a v. expression must have at least one argument which is not~%~
             preceeded with a v." exp))
	`(over ,(car conv)
	       ,(cons op (cdr conv))))
    exp))

(defun conv-over (exp pflag env)
  (let* ((bindings (second exp))
	 (body (conv-exp (third exp) t env))
	 (free-body (remove-duplicates
		     (ext-exp (third exp) (mapcar #'first bindings)))))
    (when (zerop (length bindings))
      (nesl-error 
       "In an over expression, there must be at least one binding.~%~s"
       exp))
     `(with (((segdes ,(first (first bindings)))
	      ,(second (conv-exp (first bindings) pflag env)))
	     ,@(mapcar #'(lambda (a)
			   `(,(first a) 
			     (nesl::second ,(conv-exp (second a) pflag env))))
		       (cdr bindings))
	     ,@(mapcar (if pflag
			   #'(lambda (a) 
			       `(,a (neslp::prim-dist ,a segdes seg_len)))
			 #'(lambda (a) `(,a (nesl::prim-dist ,a segdes))))
		       free-body)
	     (seg_len 
	      ,(if pflag 
		   '(nesl::vector (nesl::prim-+-reduce (nesl::len segdes) seg_len))
		 'segdes)))
	 (tup segdes ,body))))

(defun conv-bindings (bindings pflag env)
  (cond ((null bindings) nil)
	(t (cons (list (first (car bindings))
		       (conv-exp (second (car bindings)) pflag env))
		 (conv-bindings (cdr bindings) pflag env)))))

(defun conv-with (expression pflag env)
  `(with ,(conv-bindings (second expression) pflag env)
     ,(conv-exp (third expression) pflag env)))

(defun conv-constant (constant pflag env)
  (if pflag
      `(nesl::prim-dist ,constant seg_len)
    constant))
  
(defun conv-list (list pflag env)
  (cond ((null list) nil)
	(t (cons (conv-exp (car list) pflag env)
		 (conv-list (cdr list) pflag env)))))

(defun pname (name)
  (if (not (symbolp name))
      (nesl-error "~a is an invalid function name." name)
    (intern (string name) 'neslp)))

(defun prim-fun (funname)
  (member funname '(nesl::first nesl::second tup nesl::time)))

(defun conv-func (expression pflag env)
  (let ((funname (car expression))
	(args (conv-list (cdr expression) pflag env)))
    (if (and pflag (not (prim-fun funname)))
	(cons (pname funname) (append args (list 'seg_len)))
      (cons funname args))))

(defun do-thing (a)
  `(,a (nesl::second (nesl::pack (tup seg_len ,a) vcond))))

(defun conv-if (expression pflag env)
  (if (not pflag)
      `(if ,@(conv-list (cdr expression) pflag env))
    (let ((cond (conv-exp (second expression) pflag env))
	  (then (conv-exp (third expression) pflag env))
	  (else (conv-exp (fourth expression) pflag env))
	  (then-vars (remove-duplicates (ext-exp (third expression) nil)))
	  (else-vars (remove-duplicates (ext-exp (fourth expression) nil))))
      `(with ((cond ,cond)
	      (vcond (tup seg_len cond))
	      (total-count (nesl::len seg_len))
	      (true-count (nesl::count vcond))
	      (false-count (nesl::- total-count true-count)))
         (if (nesl::= true-count total-count)
	     ,then
	   (if (nesl::= false-count total-count)
	       ,else
	     (nesl::second (nesl::flag-merge vcond
	       (with ((vcond (tup seg_len (nesl::not cond)))
		      ,@(mapcar #'do-thing else-vars)
		      (seg_len (nesl::vector false-count)))
		  (tup seg_len ,else))
	       (with (,@(mapcar #'do-thing then-vars)
		      (seg_len (nesl::vector true-count)))
		  (tup seg_len ,then))
	       ))))))))

(defun conv-exp (expression pflag env)
  (cond ((constantp expression) 
	 (conv-constant expression pflag env))
	((symbolp expression) expression)
	((listp expression)
	 (let ((nexp (convert-vexp expression)))
	   (cond ((eql (car nexp) 'if)
		  (conv-if nexp pflag env))
		 ((eql (car nexp) 'nesl::nif)
		  (nesl-error "NIF is outdated, please use IF."))
		 ((eql (car nexp) 'nesl::sif)
		  `(if ,@(conv-list (cdr nexp) pflag env)))
		 ((eql (car nexp) 'with)
		  (conv-with nexp pflag env))
		 ((eql (car nexp) 'over)
		  (conv-over nexp pflag env))
		 ((eql (car nexp) 'nesl::typecase)
		  `(nesl::typecase ,@(conv-list (cdr nexp) pflag env)))
		 (t (conv-func nexp pflag env)))))
	(t (nesl-error "Invalid expression, ~s." expression))))

(defun conv-body (variables body pflag)
  (if (not pflag)
      (conv-exp body nil nil)
    ;; If the body is parallel, then we have to make sure that 
    ;; the free variables get distributed over the current context.
    (let* ((flat-variables (mapcan 'bindvars2 variables))
	   (free-variables (remove-duplicates (ext-exp body flat-variables)))
	   (converted-body (conv-exp body t nil)))
      `(with (,@(mapcar #'(lambda (a) `(,a (nesl::prim-dist ,a seg_len)))
			free-variables))
	 ,converted-body))))

(defun conv-type-list (type-list)
  (cond ((null type-list) type-list)
	((symbolp type-list)
	 (if (vprefixp type-list)
	     (list 'nesl::vector (conv-type-list (vpostfix type-list)))
	   type-list))
	((atom type-list) type-list)
	((listp type-list)
	 (cond ((eql (car type-list) 'v.)
		(cons (list 'nesl::vector (conv-type-list (second type-list)))
		      (conv-type-list (cddr type-list))))
	       (t
		 (cons (conv-type-list (car type-list))
		       (conv-type-list (cdr type-list))))))))

(defun conv-type (type pflag)
  (if pflag
      (append type (list 'nesl::vector))
    type))

(defun conv-types (types pflag)
  (mapcar #'(lambda (a) (conv-type a pflag)) types))

(defun conv-names (names pflag)
  (if pflag
      (append (cons (pname (car names)) (cdr names)) (list 'seg_len))
    names))

(defun insert-p-op (names types body definitions)
  (let ((*current-fundef* names)
	(*type-map* 'butlast))
    (insert-op (conv-names names t) (conv-types types t)
	       (conv-body (cdr names) body t) definitions)
    (insert-op (conv-names names nil) (conv-types types nil)
	       (conv-body (cdr names) body nil) definitions)))

(defun insert-s-op (names types body definitions)
  (let ((*current-fundef* names)
	(*type-map* #'(lambda (x) x)))
    (insert-op (conv-names names nil) (conv-types types nil)
	       (conv-body (cdr names) body nil) definitions)))

(defun ptrans (pflag def)
  (pprint (list 'defop 
		(conv-names (second def) pflag) 
		(conv-types (third def) pflag)
		(conv-body (cdr (second def)) (fourth def) pflag)))
  nil)

(defun gen-dist (accessor type)
  `(with ((A (,accessor A)))
     ,(cond ((atom type)
	     '(nesl::prim-dist A L))
	    ((and (listp type) (= 2 (length type)))
	     `(tup ,(gen-dist 'nesl::first (first type))
		   ,(gen-dist 'nesl::second (second type)))))))

(defun gen-dist-list (fields)
  (if (null fields) nil
    (cons (gen-dist (first (car fields))
		    (conv-type-list (second (car fields))))
	  (gen-dist-list (cdr fields)))))

(defun gen-prim-dist (name fields definitions)
   (insert-p-op '(nesl::prim-dist A L) `((,name ,name nesl::vector))
	       `(,name . ,(gen-dist-list fields)) definitions)
   (insert-p-op '(nesl::dist A L) `(((nesl::vector, name) ,name int))
	       `(with ((segdes (nesl::vector L)))
                   (tup segdes (nesl::prim-dist A segdes)))
	       definitions))

(defun insert-p-rec (name fields definitions)
  (insert-rec name (mapcar #'(lambda (field)
			       (cons (car field) (conv-type-list (cdr field))))
			   fields)
	      definitions)
  (gen-prim-dist name fields definitions))

