;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl)

#|
Note that PARTITION and FLATTEN are the only user accessible functios that
deal with segments directly.  These should be considered the primitives
for moving between levels of nesting.  All other operations should
be built out of these (this will make it easier to update things
if the representation of segments change).
|#

(opsubsection "Nesting Vectors" " 
The two functions \\fun{partition} and
\\fun{flatten} are the primitives for moving between levels of
nesting.  All other functions for moving between levels of nesting
can be built out of these.  The functions \\fun{split} and \\fun{bottop}
are often useful for divide-and-conquer routines.")

(ndefop (partition values counts) 
	((v.v.alpha <- v.alpha v.int) (alpha any))
  (tup (first counts) (tup (vector (second counts)) (second values)))
  :documentation
"Given a sequence of values and another sequence of counts, \\fun{partition}
returns a nested sequence with each subsequence being of a length
specified by the counts.  The sum of the counts must equal the
length of the sequence of values."
  :example
  (((a0 a1 a2 a3) (a4) (a5 a6 a7)) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (4 1 3))
  :document yes)

(ndefop (flatten values) ((v.alpha <- v.v.alpha) (alpha any))
  (with ((counts (tup (first values) (len (first (second values))))))
    (tup (vector (+-reduce counts))
	 (second (second values))))
  :documentation
"Given a nested sequence of values, \\fun{flatten} flattens the sequence."
  :example
  ((a0 a1 a2 a3 a4 a5 a6 a7) <-
   ((a0 a1 a2) (a3 a4) (a5 a6 a7)))
  :document yes)

(defop (split-index flags) ((v.int v.int) <- v.bool)
  (with ((not-f  (v.btoi (v.not flags))) 
	 (i-down (+-scan not-f))
	 (sum-down (+-reduce not-f))
	 (i-up   (v.+ v.sum-down (+-scan (v.btoi flags)))))
    (tup (vtup sum-down (- (length flags) sum-down))
	 (v.select flags i-up i-down))))

(ndefop (split v flags) ((v.v.alpha <- v.alpha v.bool) (alpha any))
  (with (((counts index) (split-index flags)))
    (partition (permute v index) counts))
  :document yes
  :documentation
"Given a sequence of values \\farg{a} and a boolean sequence of
\\farg{flags}, \\fun{split} creates a nested sequence of length
2 with all the elements with an \\farg{f} in
their flag in the first element and elements with a \\farg{t} in
their flag in the second element."
  :example
  (((a1 a3 a4) (a0 a2 a5 a6 a7)) <-
   (a0 a1 a2 a3 a4 a5 a6 a7)
   (t f t f f t t t))
  :redefine no)

(ndefop (bottop values) ((v.v.alpha <- v.alpha) (alpha any))
  (with ((l (length values))
	 (botlen (rshift (+ l 1) 1)))
    (partition values (vtup botlen (- l botlen))))
  :document yes
  :documentation
"Given a sequence of values \\farg{values}, \\fun{bottop} 
creates a nested sequence of length
2 with all the elements from the bottom half of the sequence in
the first element and elements from the top half of the sequence in 
the second element."
  :example
  (((a0 a1 a2 a3) (a4 a5 a6)) <-
   (a0 a1 a2 a3 a4 a5 a6))
  :redefine no)

(ndefop (head-rest values) (((alpha v.alpha) <- v.alpha) (alpha any))
  (tup (elt values 0) (drop 1 values))
  :document yes
  :documentation
"Given a sequence of values \\farg{values} of length > 1, \\fun{head-rest} 
returns a tuple with the first element of the vector, and the
remaining elements of the vector."
  :redefine no)

(ndefop (rest-tail a) (((v.alpha alpha) <- v.alpha) (alpha any))
  (with ((n (length a)))
    (tup (take (- n 1) a) (elt a (- n 1)))))

(defop (bit v i) (bool <- int int)
  (= 1 (and (rshift v i) 1)))

(defop (radix-rank-r v i start end) (v.int <- v.int v.int int int)
  (if (= start end)
      (permute (index (length i)) i)
    (with ((index (second (split-index (v.bit v v.start)))))
      (radix-rank-r (permute v index) (permute i index) (+ start 1) end))))

(defop (radix-rank v bits) (v.int <- v.int int)
  (radix-rank-r v (index (length v)) 0 bits))

(defop (rank v) (v.int <- v.int)
  (radix-rank-r v (index (length v)) 0 32))

(defop (char-rank V) (v.int <- v.char)
  (radix-rank (v.char-code v) 7))
