;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DEFINITIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrec file-pointer (pnt int) (name v.char))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; WRITING TO STDOUT
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(odefop (write-scalar a) ((int <- int) (char <- char) 
			  (bool <- bool) (float <- float))
  (((COPY 1 0) (WRITE INT) (POP 2 0))
   ((COPY 1 0) (WRITE CHAR) (POP 2 0))
   ((COPY 1 0) (WRITE BOOL) (POP 2 0))
   ((COPY 1 0) (WRITE FLOAT) (POP 2 0)))
  :compound-prim t :pversion nil)

(odefop (write-segdes a) ((vector <- vector))
  (((COPY 1 0) (WRITE SEGDES) (POP 2 0)))
  :compound-prim t :pversion nil)

(defop (print-char v) (bool <- char)
  (with ((temp (write-scalar v))) t))

(defop (print-string v) (bool <- v.char)
  (with ((temp (write-scalar (second v)))) t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; OPENING AND CLOSING FILES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(odefop (prim-open-file filename mode) (((int (char bool)) <- char int))
  ((FOPEN))
  :primitive t :pversion nil)

(odefop (prim-close-file filep) (((char bool) <- int))
  ((FCLOSE))
  :primitive t :pversion nil)

(defop (file-error error-type filename message) (char <- v.char v.char char)
  (with ((s1 (print-string "ERROR while trying to "))
	 (s2 (print-string error-type))
	 (s3 (print-string ": "))
	 (s4 (print-string filename))
	 (s5 (print-char #\newline))
	 (s6 (print-char message))
	 (s7 (print-char #\newline)))
    message))

(defop (open-file filename mode) (file-pointer <- v.char int)
  (with (((pointer (message flag)) (prim-open-file (second filename) mode))
	 (junk (sif flag message (file-error "open file" filename message))))
    (file-pointer pointer filename)))

(defop (open-in-file filename) (file-pointer <- v.char)
  (open-file filename 1))

(defop (open-out-file filename) (file-pointer <- v.char)
  (open-file filename 2))

(defop (close-file filep) (bool <- file-pointer)
  (with (((message flag) (prim-close-file (pnt filep)))
	 (junk (sif flag message 
		 (file-error "close file" (name filep) message))))
    flag))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; READING FROM A FILE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(odefop (prim-readf-scalar val filep) (((int (char bool)) <- int int)
				      ((float (char bool)) <- float int)
				      ((bool (char bool)) <- bool int)
				      ((char (char bool)) <- char int))
  (((POP 1 1) (FREAD INT)) ((POP 1 1) (FREAD FLOAT))
   ((POP 1 1) (FREAD BOOL)) ((POP 1 1) (FREAD INT)))
  :compound-prim t :pversion nil)

(odefop (readf-scalar filep val) ((int <- file-pointer int)
				 (float <- file-pointer float)
				 (bool <- file-pointer bool)
				 (char <- file-pointer char))
  (with (((val (message flag)) (prim-readf-scalar val (pnt filep)))
	 (er (sif flag #\null 
	       (file-error "read from file" (name filep) message))))
    val))

(odefop (prim-readf-segdes filep) (((vector (char bool)) <- int))
  (((FREAD SEGDES)))
  :compound-prim t :pversion nil)

(odefop (readf-segdes filep) ((vector <- file-pointer))
  (with (((val (message flag)) (prim-readf-segdes (pnt filep)))
	 (er (sif flag #\null 
	       (file-error "read from file" (name filep) message))))
    val))

(ndefop (read-object filep a) ((alpha <- file-pointer alpha) (alpha any))
  (typecase a
     (readf-scalar filep a)
     (tup (readf-segdes filep)
	  (read-object filep (second a)))
     (tup (read-object filep (first a))
	  (read-object filep (second a)))))

(ndefop (read-object-from-file object-type filename)
	((alpha <- alpha v.char) (alpha any))
  (with ((filep (open-in-file filename))
	 (result (read-object filep object-type))
	 (closep (close-file filep)))
    result))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; WRITING TO A FILE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(odefop (prim-writef-scalar val filep) (((char bool) <- int int)
				       ((char bool) <- float int)
				       ((char bool) <- bool int)
				       ((char bool) <- char int))
  (((FWRITE INT)) ((FWRITE FLOAT)) ((FWRITE BOOL)) ((FWRITE INT)))
  :compound-prim t :pversion nil)

(odefop (writef-scalar filep val) ((int <- file-pointer int) 
				   (float <- file-pointer float)
				   (bool <- file-pointer bool)
				   (char <- file-pointer char))
  (with (((message flag) (prim-writef-scalar val (pnt filep)))
	 (er (if flag #\null 
	       (file-error "write to file" (name filep) message))))
    val)
  :pversion nil)

(odefop (prim-writef-segdes segdes filep) (((char bool) <- vector int))
  (((FWRITE SEGDES)))
  :compound-prim t :pversion nil)

(odefop (writef-segdes filep val) ((vector <- file-pointer vector))
  (with (((message flag) (prim-writef-segdes val (pnt filep)))
	 (er (if flag #\null
	       (file-error "write to file" (name filep) message))))
    val)
  :pversion nil)

(ndefop (write-object filep a) ((alpha <- file-pointer alpha) (alpha any))
  (typecase a
     (writef-scalar filep a)
     (tup (writef-segdes filep (first a))
	  (write-object filep (second a)))
     (tup (write-object filep (first a))
	  (write-object filep (second a)))))

(ndefop (write-object-to-file object filename) 
	((alpha <- alpha v.char) (alpha any))
  (with ((filep (open-out-file filename))
	 (result (write-object filep object))
	 (closep (close-file filep)))
    result))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; GENERATING STRINGS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;
;; BOOLS AND CHARS
;;;;;;;;;;;;;;;

(defop (string v) (v.char <- bool)
  (if v "t" "f"))

(defop (string v) (v.char <- char)
  (append "#\\" (dist v 1)))

;;;;;;;;;;;;;;;
;; INTEGERS
;;;;;;;;;;;;;;;

(set *hex-chars* "0123456789abcdef")

(defop (positive-int-string a base) (v.char <- int int)
  (if (= 0 a) ""
    (append (positive-int-string (/ a base) base) 
	    (dist (elt *hex-chars* (rem a base)) 1))))

(defop (string v) (v.char <- int)
  (if (zerop v) "0" 
    (if (plusp v)
	(positive-int-string v 10)
      (cons #\- (positive-int-string (abs v) 10)))))

;;;;;;;;;;;;;;;
;; FLOATS
;;;;;;;;;;;;;;;

(set *float-length* 5)

(defop (print-after-dot a count) (v.char <- int int)
  (if (= 0 count) ""
    (snoc (print-after-dot (/ a 10) (- count 1))
	  (elt *hex-chars* (rem a 10)))))

(defop (positive-float-string val) (v.char <- float)
  (with ((exp (+ 1 (floor (log val 10.0))))
	 (aprox (round (* val (expt 10.0 (float (- *float-length* exp))))))
	 (float-trunc (round (expt 10.0 (float (- *float-length* 1)))))
	 (head (positive-int-string (/ aprox float-trunc) 10))
	 (after-point (print-after-dot (rem aprox float-trunc)
				       (- *float-length* 1)))
	 (exp-str (cons #\e (string (- exp 1)))))
    (append head (cons #\. (append after-point exp-str)))))

(defop (string val) (v.char <- float)
  (if (zerop val) "0.0"
    (if (plusp val)
	(positive-float-string val)
      (cons #\- (positive-float-string (abs val))))))

;;;;;;;;;;;;;;;
;; VECTORS
;;;;;;;;;;;;;;;

(defop (pad str) (v.char <- v.char)
  (cons #\space str))

(defop (vector-string vect) (v.char <- v.v.char)
  (if (= (length vect) 0) "#v()"
     (append (append "#v(" (drop 1 (flatten (v.pad vect)))) ")")))

(defop (string v) (v.char <- v.char)
  (cons #\" (append v "\"")))

(ndefop (string v) ((v.char <- alpha) (alpha any))
  (typecase v
    (string v)
    (vector-string (v.string v))
    (append "(TUP " 
	    (append (string (first v))
		    (cons #\space (snoc (string (second v)) #\)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PRETTY PRINTING OBJECTS TO STDOUT
;;;;;;;;;;;;;;;;;;;;;;;;;

;; This prints an object followed by a space.
(ndefop (print v) ((alpha <- alpha) (alpha any))
  (with ((temp (print-string (snoc (string v) #\space))))
    v))

;; Same as print, but followed by a newline instead of a space.
(ndefop (printl v) ((alpha <- alpha) (alpha any))
  (with ((temp (print-string (snoc (string v) #\newline))))
    v))

(ndefop (print-debug str v) ((alpha <- v.char alpha) (alpha any))
  (with ((temp (print-string (append str (snoc (string v) #\newline)))))
    v))

