;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;;; The routines in this file take an NESL expression and mark all the
;;; final references to variables with LAST.   This, in turn, is
;;; used by the compiler to deallocate the variable whenever it
;;; sees a reference to last.  
;;; As well as taking the expression, the routines in the file take
;;; a list of continuation variables (variable references that appear
;;; in the continuation).  Any variable in this list will not be marked
;;; with LAST.
;;; Each of the routines in the file returns a cons in which the first item
;;; is a list of free variables that appear in the body, and the
;;; second is a new body with the appropriate transformations made.
;;;
;;; An example:
;;;  (free-exp 
;;;    '(with ((x (+ x y))
;;;            (z (pack w q)))
;;;       (- (+ (* x 7) (+_reduce z)) y))
;;;    '(w))
;;;
;;; will return:
;;;
;;;    ((Q W Y X) .
;;;     (WITH ((X (+ (LAST X) Y)) 
;;;            (Z (PACK W (LAST Q))))
;;;        (- (+ (* (LAST X) 7) (+_REDUCE (LAST Z))) (LAST Y)))) 
;;;
;;; The FREE-IF function also adds some information to if statements.
;;; In particular it adds a list of the free variables that are 
;;; freed (marked with LAST) in both the IF part and the ELSE part.

(defun bindvars (varlist)
  (if (atom varlist) (list varlist)
    (append (bindvars (first varlist)) 
	    (bindvars (second varlist)))))

(defun free-binds (binds body-vars cont-vars)
  (if (null binds) (cons body-vars nil)
    (let* ((bind (car binds))
	   (bind-vars (bindvars (first bind)))
	   (new-cont-vars (set-difference cont-vars bind-vars))
	   (fl (free-binds (cdr binds) body-vars new-cont-vars))
	   (unused-vars (set-difference bind-vars (car fl)))
	   (rem-bind-vars (set-difference (car fl) bind-vars))
	   (bind-code (cdr fl))
	   (fe (free-exp (second bind) (union cont-vars rem-bind-vars))))
      (cons (union rem-bind-vars (car fe))
	    (cons (append (list (car bind) (cdr fe)) unused-vars) 
		  bind-code)))))

(defun free-with (exp cont-vars)
  (let* ((bind-vars (mapcan #'(lambda (x) (bindvars (first x))) (second exp)))
	 (fbody (free-exp (third exp) (set-difference cont-vars bind-vars)))
	 (fbinds (free-binds (second exp) (first fbody) cont-vars)))
    (cons (union (set-difference (first fbody) bind-vars) (first fbinds))
	  (list 'with (cdr fbinds) (cdr fbody)))))

(defun free-symbol (exp cont-vars)
  (cons (list exp) 
	(if (find exp cont-vars) exp (list 'last exp))))

(defun free-list (list cont-vars)
  (if (null list) nil
    (let* ((fl (free-list (cdr list) cont-vars))
	   (listvars (car fl))
	   (listcode (cdr fl))
	   (fe (free-exp (car list) (union cont-vars listvars))))
      (cons (union listvars (car fe)) (cons (cdr fe) listcode)))))

(defun free-func (exp cont-vars)
  (let ((fl (free-list (cdr exp) cont-vars)))
    (cons (car fl) (cons (car exp) (cdr fl)))))

(defun free-if (exp cont-vars)
  (let* ((fif (free-exp (third exp) cont-vars))
	 (felse (free-exp (fourth exp) cont-vars))
	 (ifelsevars (union (car fif) (car felse)))
	 (fcond (free-exp (second exp) (union cont-vars ifelsevars))))
    (cons (union ifelsevars (car fcond))
	  (list 'if (cdr fcond) (cdr fif) (cdr felse)
		(set-difference (car fif) cont-vars)
		(set-difference (car felse) cont-vars)))))

(defun free-exp (exp cont-vars)
  (cond ((constantp exp) (cons nil exp))
	((symbolp exp)
	 (free-symbol exp cont-vars))
	((listp exp)
	 (cond ((eql (car exp) 'if)
		(free-if exp cont-vars))
	       ((eql (car exp) 'with)
		(free-with exp cont-vars))
	       (t (free-func exp cont-vars))))
	(t (error "Invalid expression, ~s." exp))))
