;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;; This is the only translator file with global variables

(defparameter *definitions* (make-definitions-table))

(defun add-primitive-types (type-list definitions)
  (dolist (type type-list)
    (add-primitive-type type definitions)))

;; *prim-types*
(add-primitive-types '(int bool char nesl::float nesl::vector) *definitions*)

(defun insert-prim (name types prim-calls compound definitions)
  (dolist (type-call (mapcar #'cons types prim-calls))
    (let ((code (if compound 
		    (cdr type-call) 
		  (list (cdr type-call)))))
      (add-primitive-op 
       (car name) (car type-call) code definitions)
      (add-primitive-op
       (pname (car name)) (append (car type-call) (list 'nesl::vector)) 
       (cons (list 'POP 1 0) code) definitions))))

(defun write-op (name source-type filename)
  (write-func name (conv-type-list source-type)
	      *definitions* filename))

(defun write-sop (name source-type filename)
  (write-func name source-type *definitions* filename))

(defparameter *oplist* nil)

(defun extract-type (types)
  (let ((long-types (conv-type-list types)))
    (if (eql (second long-types) '<-)
	(cons (car long-types) (cddr long-types))
      (nesl-error "The second element of a DEFOP typedef must be <-"))))

(defun nesl::opsection (name text) 
  (push (list "subsection" name text) *oplist*))

(defun nesl::opsubsection (name text) 
  (push (list "subsubsection" name text) *oplist*))

(defmacro defop (names type body)
  `(nesl::odefop ,names (,type) ,body :type (,type)))

(defmacro nesl::sdefop (names type body)
  `(nesl::odefop ,names (,type) ,body :pversion nil))

(defmacro nesl::odefop (names types body &key
		       documentation
		       example
		       (pversion t)
		       primitive
		       compound-prim
		       document
		       interface
		       bugs
		       type
		       redefine)
  (let ((rtypes (mapcar #'extract-type types))
	(name (car names))
	(vname (gen-v-name (car names))))
    (setq *lastfuncall* nil)
    (when (vprefixp (car names))
      (nesl-error 
       "You cannot define a function with a name that starts with v. :~%~
              (DEFOP (~S) ....)" names))
    (if (not (or compound-prim primitive))
	(if pversion
	    (insert-p-op names rtypes body *definitions*)
	  (insert-s-op names rtypes body *definitions*))
      (insert-prim names rtypes body compound-prim *definitions*))
    (when pversion
      (insert-v-version names rtypes))
    (push (list (if document 'opdef nil) 
		(if interface interface names)
		type documentation example) *oplist*)
    `(progn
       (unless (fboundp ',name)
         (defun ,name (&rest stuff)
	   (apply 'call-vcode ',name stuff)))
       ,(when pversion
	  `(unless (fboundp ',vname)
	    (defun ,vname (&rest stuff)
	      (apply 'call-vcode ',vname stuff))))
       ',(first names))))

(defparameter *funlist* nil)

(defmacro nesl::ndefop (names type body &key
			      documentation
			      example
			      document
			      interface
			      redefine)
  (let* ((serial-names (conv-names names nil))
	 (serial-body (conv-body (cdr names) body nil))
	 (parallel-names (conv-names names t))
	 (parallel-body (conv-body (cdr names) body t))
	 (flattype (extract-type (first type)))
	 (serial-type (cons (conv-type flattype nil) (second type)))
	 (parallel-type (cons (conv-type flattype t) (second type))))
    (push (list (if document 'opdef nil) 
		(if interface interface names)
		type documentation example) *oplist*)
    (push (cons (car serial-names) 
		(cons :type 
		      (cons serial-type
			    (cons serial-names serial-body))))
	  *funlist*)
    (push (cons (car parallel-names) 
		(cons :type
		      (cons parallel-type
			    (cons parallel-names parallel-body))))
	  *funlist*)
    `(car ',names)))

#|
(defmacro nesl::ndefop (names body &key
		       documentation
		       example
		       document
		       interface
		       redefine
		       type)
  (let ((serial-names (conv-names names nil))
	(serial-body (conv-body (cdr names) body nil))
	(parallel-names (conv-names names t))
	(parallel-body (conv-body (cdr names) body t)))
    (push (list (if document 'opdef nil) 
		(if interface interface names)
		type documentation example) *oplist*)
     (push (cons (car serial-names) 
		 (cons serial-names serial-body))
	   *funlist*)
     (push (cons (car parallel-names) 
		 (cons parallel-names parallel-body))
	   *funlist*)
     `(car ',names)))
|#

(defun gen-v-name (op)
  (intern (format nil "V.~a" (symbol-name op)) (symbol-package op)))

(defun insert-v-version (names types)
  (let ((vop (gen-v-name (car names)))
	(flat-args (mapcar #'(lambda (junk) (gensym)) (cdr names)))
	(vtypes (mapcar #'(lambda (type-list)
			    (mapcar #'(lambda (type) (list 'nesl::vector type))
				    type-list))
			types)))
    (insert-p-op (cons vop flat-args) vtypes (cons vop flat-args)
		 *definitions*)))

(defmacro defrec (name &rest fields)
  (let ((fnames (cons name (mapcar #'(lambda (a) (gentemp)) fields)))
	(types (list (cons name (mapcar #'second fields))))
	(vname (gen-v-name name)))
    (insert-p-rec name fields *definitions*)
    (insert-v-version fnames (conv-type-list types))
    `(progn
       (unless (fboundp ',name)
         (defun ,name (&rest stuff)
	   (apply #'make ',name stuff)))
       ,@(mapcar #'(lambda (field)
		     `(unless (fboundp ',(car field))
			(defun ,(car field) (&rest stuff)
			  (apply #'call-vcode ',(car field) stuff))))
		 fields)
       (unless (fboundp ',vname)
	 (defun ,vname (&rest stuff)
	   (apply 'call-vcode ',vname stuff)))
       ',name)))

(defmacro defrecs (name &rest fields)
  `(progn
     (unless (fboundp ',name)
       (defun ,name (&rest stuff)
	 (apply #'make ',name stuff)))
     ',(insert-rec name fields *definitions*)))

#|
(defmacro nesl () `(in-package 'nesl))
(defmacro user::nesl () `(in-package 'nesl))
(defmacro nesl::nesll () `(in-package 'nesl-lisp))
(defmacro user::nesll () `(in-package 'nesl-lisp))
|#
