/* 
 * Mach Operating System
 * Copyright (c) 1993 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */
/*
 * HISTORY
 * $Log:	cdrom_mach3_0.c,v $
 * Revision 2.2  93/05/31  16:03:02  mrt
 * 	Created a while back.
 * 	[93/05/08            af]
 * 
 */
/*
 *	File: cdrom_mach3_0.c
 * 	Author: Alessandro Forin, Carnegie Mellon University
 *	Date:	3/93
 *
 *	Mach 3.0 CDROM-DA access routines.
 */

# include <stdio.h>
# include <strings.h>
# include <mach.h>
# include <device/device_types.h>

# include <errno.h>	/* compat */

# include "debug.h"
# include "cdrom_callb.h"

#define private static

#define CDROM_DEVICE "cd_audio4"

extern char *getenv();
extern void	*calloc(), *malloc();

cdrom_info	cdi;

private char	*cdrom;

private device_t	cdrom_port = MACH_PORT_NULL;

struct cdrom_toc_header {
	int	data_length;
	int	start_track;
	int	end_track;
};

/* done */
int
cdrom_open() {
	kern_return_t   rc;
	int	n;
	mach_port_t     device_server_port, device_port;

	if (cdrom_port != MACH_PORT_NULL)
		return(cdi.cdi_curtrack);

	cdrom = getenv("CDROM");
	if (!cdrom) cdrom = CDROM_DEVICE;

	device_server_port = task_by_pid(-2);
	if (device_server_port == MACH_PORT_NULL) {
		errno = EPERM;
		perror("Getting device_server_port: ");
		return -1;
	}

	rc = device_open(device_server_port, 0, cdrom, &cdrom_port);
	if (rc != D_SUCCESS) {
		mach_error(cdrom, rc);
		return -1;
	}

	if (cdrom_get_times() == -1)
		return(-1);

	if ((n = cdrom_get_curtrack()) == -1)
		return(-1);

	cdi.cdi_curtrack = n;

	if (cdi.cdi_state & CDROM_STATE_SHUFFLE)
		shuffle_setup();

	return((cdi.cdi_curtrack == 0) ? 1 : cdi.cdi_curtrack);
}

/* done */
void
cdrom_close() {
	if (cdrom_port == MACH_PORT_NULL)
		return;

	if (cdi.cdi_times != NULL) {
		free((char *) cdi.cdi_times);
		cdi.cdi_times = NULL;
	}

	(void) device_close(cdrom_port);
	cdrom_port = MACH_PORT_NULL;
}

/* done */
int
cdrom_get_times() {
	struct cdrom_toc_header	sz;
	extern unsigned short	*ushort_malloc();
	unsigned int		trk, trk_total;
	char			buf[128];
	natural_t		num;
	char			*data, *p;

	if (cdrom_read_toc_sizes(&sz) == -1)
	  return -1;

	cdi.cdi_mintrack = sz.start_track;
	cdi.cdi_maxtrack = sz.end_track;

	cdi.cdi_times = ushort_malloc(cdi.cdi_maxtrack - cdi.cdi_mintrack + 1);

	/* ask for it */
	sprintf(buf, "Toc MSF %d", sz.start_track);
	num = strlen(buf)+1;
	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	/* get results back */
	num = round_page(sz.data_length);
	device_read(cdrom_port, 0, 0, num, &data, &num);

	if (debug) printf("%s\n", data);

	/* parse TOC */
	p = data;
	while (p = index(p, '\n'))
	{
		int	minutes, seconds, ignore;

		if (*++p == 0)
			break;
		sscanf(p, "%d %d %d %d %d %d",
			/* attributes */ &ignore,
			/* type */	 &ignore,
			/* trackno */	 &trk,
			&minutes, &seconds,
			/* frame */	 &ignore);

		if ((trk <= cdi.cdi_maxtrack) &&
		    (trk >= cdi.cdi_mintrack)) {
			trk_total = minutes * 60 + seconds;
			cdi.cdi_times[trk - cdi.cdi_mintrack] = trk_total;
		} 
		/* Get address of empty track --> total duration */
		else if (trk > cdi.cdi_mintrack) {
			trk_total = minutes * 60 + seconds;
			cdi.cdi_times[cdi.cdi_maxtrack] = trk_total;
		}
	}
	if (p == data) {
		fprintf(stderr, "%s\n%s\n",
			"Something screwed in this TOC:",
			data);
		fprintf(stderr, "%s\n%s\n",
			"Stopped parsing after:",
			p);
	}

	vm_deallocate(mach_task_self(), data, num);
	return 0;
}

private void
cdrom_print_toc() {
	unsigned long		trk, trk_total;

	for (trk = cdi.cdi_mintrack; trk <= cdi.cdi_maxtrack; trk++) {
		trk_total = cdi.cdi_times[trk - cdi.cdi_mintrack];
		debug_printf(1, "%02u:%02u\n", trk_total/60, trk_total%60);
	}
}


/* done */
int
cdrom_get_curtrack() {
	char		buf[128];
	natural_t	num;
	int		ii, minute, second, pos;

	sprintf(buf, "Get Position MSF");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);
	num = sizeof(buf);
	device_read_inband(cdrom_port, 0, 0, num, buf, &num);

	if (debug) printf("%s\n", buf);

	minute = second = 0;
	sscanf(buf, "MSF Position %d %d", &minute, &second);	/* ignore rest */
	pos = minute * 60 + second;

	for (ii = 0; ii < cdi.cdi_maxtrack - cdi.cdi_mintrack; ii++)
	{
	    if (pos < cdi.cdi_times[ii + 1])
	    {
		/* Hack to keep minutes/seconds display current -njl */
		cdi.cdi_dur = pos - (int)cdi.cdi_times[ii];
/*		fprintf(stderr, "cdi_dur = %d\n", cdi.cdi_dur); */
		if (cdi.cdi_dur > 10000) /* longer than any CD I've ever seen */
		    cdi.cdi_dur = 0;

		return ii + cdi.cdi_mintrack;
	    }
	}
	/* Second half of above hack -njl */
	cdi.cdi_dur = pos - (int) cdi.cdi_times[cdi.cdi_maxtrack-1];
	return cdi.cdi_maxtrack;
}

/* done */
int
cdrom_play_track(start_track, end_track)
	unsigned char		start_track;
	unsigned char		end_track;
{
	char		buf[128];
	natural_t	num;
	int		end_ix = 1;

	/* The following correctly plays the last track of a cd even
	   with the Toshiba 3101 BME drive that is shipped with
	   RISC/6000 systems.  It should work for better behaving
	   drives.  do12+@cs.cmu.edu */

	return cdrom_play_track_sec(start_track, 0, end_track+1, 0);

	/*
	 * Some like it red, some like it green...
	 * The issue here is how to specify "the end of the disc"
	 */
#if 0
	if ((start_track == end_track) &&
	    (start_track == cdi.cdi_maxtrack))

		return cdrom_play_track_sec(start_track, 0, end_track, 0);


	sprintf(buf, "Play TI %d 1 %d %d", start_track, end_track, end_ix);
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;
#endif
}

/* done */
int cdrom_play_track_sec(start_track, start_sec, end_track, end_sec)
    unsigned char start_track;
    int start_sec;
    unsigned char end_track;
    int end_sec;
{
	char		buf[128];
	natural_t	num;

	/*
	 * Some like it red, some like it green...
	 * The issue here is how to specify "the end of the disc"
	 */
	if ((start_track == end_track) &&
	    (start_sec == end_sec))
		end_track++;

	start_sec += cdi.cdi_times[start_track - cdi.cdi_mintrack];
	end_sec += cdi.cdi_times[end_track - cdi.cdi_mintrack];

	sprintf(buf, "Play A %d %d 0 %d %d 0",
		/* startM */ start_sec / 60,
		/* startS */ start_sec % 60,
		/* startF = 0 */
		/* endM */ end_sec / 60,
		/* endS */ end_sec % 60
		/* endF = 0 */ );
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;

}

/* done */
int
cdrom_start() {
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Start");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;
}

/* done */
int
cdrom_stop() {
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Stop");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;
}

/* done */
int
cdrom_eject() {
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Eject");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;
}

/* done */
int
cdrom_read_toc_sizes(toc_sizes)
	struct cdrom_toc_header	*toc_sizes;
{
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Get TH");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);
	num = sizeof(buf);
	device_read_inband(cdrom_port, 0, 0, num, buf, &num);

	if (debug) printf("%s\n", buf);

	sscanf(buf, "toc header: %d %d %d",
		&toc_sizes->data_length,
		&toc_sizes->start_track,
		&toc_sizes->end_track);

	return 0;
}

/* done */
int
cdrom_pause() {
	return cdrom_stop();
}

/* done */
int
cdrom_resume() {
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Resume");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	return 0;

}

/* done */
int
cdrom_volume(left_vol, right_vol)
	int			left_vol;
	int			right_vol;
{
	char			buf[128];
	natural_t		num;

#if 0
	/* Magic formulae -njl */
	left_vol = (left_vol>>2)+ 192;
	right_vol = (right_vol>>2) + 192;
	pbc.pc_chan0_volume = ((left_vol*left_vol*left_vol)>>17)+0x80;
	pbc.pc_chan1_volume = ((right_vol*right_vol*right_vol)>>17)+0x80;
#endif

	sprintf(buf, "Set V %d %d 0 0",
		/* vol0 = */ left_vol,
		/* vol1 = */ right_vol
		/* vol2 = 0 */
		/* vol3 = 0 */);
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

}

/* done */
int
cdrom_status() {
	char		buf[128];
	natural_t	num;

	sprintf(buf, "Get Status");
	num = strlen(buf)+1;

	device_write_inband(cdrom_port, 0, 0, buf, num, &num);

	num = sizeof(buf);
	device_read_inband(cdrom_port, 0, 0, num, buf, &num);

	if (debug) printf("%s\n", buf);

	switch (buf[0]) {
		case 'P':
			return(CDROM_PLAYING);

		case 'S':
			return(CDROM_PAUSED);

		case 'D':
			return(CDROM_COMPLETED);

		case 'E':
			return(CDROM_ERROR);

		case 'I':	/* Idle */
			return(CDROM_NO_STATUS);

		default:
			fprintf(stderr, "Unknown status: %s\n", buf);
	}

	return(-1);
}

unsigned short *
ushort_malloc(n)
	int		n;
{
	unsigned short	*ptr;

	ptr = (unsigned short *) calloc(n, sizeof(unsigned short));
	if (ptr == NULL) {
		perror("calloc");
		exit(1);
	}

	return(ptr);
}

