/* 
 * Mach Operating System
 * Copyright (c) 1993 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS 
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie the
 * rights to redistribute these changes.
 */
/*
 * monctl -- control program monitoring
 */
/* 
 * HISTORY
 * $Log:	monctl.c,v $
 * Revision 2.2  93/08/06  23:59:58  mrt
 * 		Created. Used Bob Baron's uxmon program as a model
 * 	[93/08/06            bershad]
 * 
 * 	
 *  File:	monctl.c
 *  Author:	Brian Bershad, Carnegie Mellon University
 *  Date:	July, 1993
 */

#include <stdio.h>
#include <strings.h>

#include <mach.h>
#include <mach/notify.h>

/*
 * prevent profiling libraries from feeding on ourselves
 */
int gprof_no_startup_from_crt0 = TRUE;
int gprof_no_control_server = TRUE;
int gprof_no_autoinitialization_from_crt0 = TRUE;

extern int gprof_debug;

mach_port_t gprof_server_port;

/* global vars set by program args */
int mon_state = 0;
int mon_pid = -1;
int mon_ux = FALSE;		/* UX is server */
int mon_off = FALSE;
char *gmon_out = NULL;
char *hostname = "";
char *exec_file_name = 0;
int become_proxy = FALSE;




struct flavor_info {
    char *name;
    int flavor;
    
};

struct flavor_info flavors[] = {
{ "periodic",  SAMPLED_PC_PERIODIC},
{ "vmzfill",   SAMPLED_PC_VM_ZFILL_FAULTS},
{ "vmcow",     SAMPLED_PC_VM_COW_FAULTS},
{ "vmfaults",  SAMPLED_PC_VM_FAULTS},
{ 0, 0}} ;
    


int
get_mon_state(s)
    char *s;
{
    int i;
    for (i = 0; flavors[i].name; i++)  {
	if (strcmp(flavors[i].name, s) == 0)  {
	    return flavors[i].flavor;
	}
    }

    return -1;
}

int
mon_state_to_name(s, name)
	int s;
	char *name;
{
    int i;
    *name = 0;

    if (s == 0) {
	strcat(name, "off");
    } else {
    	for (i = 0; flavors[i].name; i++)  {
		if (s & flavors[i].flavor)  {
		     if (*name) strcat(name, "|");
		     strcat(name, flavors[i].name);
		}
    	}
    }
    return *name;
}
	


void
usage(name)
	char *name;
{
    int i = 0;
    
    printf("usage: %s -host <host> -pid <pid> -on <flavor> -proxy -execfile <execfile> -off -monfile <monfile>\n", name);
    printf("flavor is one of:");
    for (i = 0; flavors[i].name; i++)  {
	printf("%s\t", flavors[i].name);
    }
    printf("\n");    
}
    

abort(x)
    int x;
{
    fflush(stdout);
    fflush(stderr);
    _exit(x); 

}

    


main(argc, argv)
	int argc;
	char **argv;
{
    kern_return_t kr = KERN_SUCCESS;
    int i;
    int res = 0;
    int x;

    for (i = 1; i < argc; i++)  {
	if (strcmp(argv[i], "-help") == 0)  {
	    usage(argv[0]);
	    abort(0);
	} else if ((strcmp(argv[i], "-host") == 0) && (i < argc-1))   {
	    hostname = argv[++i];	    
	} else if ((strcmp(argv[i], "-pid") == 0) && (i < argc-1))   {
	    mon_pid = atoi(argv[++i]);
	} else if ((strcmp(argv[i], "-on") == 0) && (i < argc-1)) {
	    x = get_mon_state(argv[++i]);
	    if (x < 0)  {
		usage(argv[0]);
		abort(-1);
	    } else {
		mon_state = x|mon_state;
	    }
	} else if ((strcmp(argv[i], "-off") == 0)) {
	    mon_off = TRUE;
	    mon_state = 0;
	} else if (strcmp(argv[i], "-status") == 0) {
	    mon_state = -1;
	} else if ((strcmp(argv[i], "-monfile") == 0) && (i < argc-1))  {
	    gmon_out = argv[++i];
	} else if (strcmp(argv[i], "-proxy") == 0) {
	    become_proxy = TRUE;
	} else if ((strcmp(argv[i], "-execfile") == 0) && (i < argc-1)) {
	    exec_file_name = argv[++i];
	} else  {
	    usage(argv[0]);
	    abort(-1);
	}
    }

    if (become_proxy)  {
	if (*hostname)  {
		fprintf(stderr,"Can only serve as proxy on the local host\n");
		kr = -1;
	} else {
	        kr = mon_proxy();
	}
    }  else {
	kr = mon_commands();
    }


    abort(kr);
}


mon_commands()	
{
    kern_return_t kr;
    kr = mon_get_port();
    if (kr != KERN_SUCCESS)  {
	mach_error("Could not get profiler port\n", kr);
	return -1;
    }
    if (*hostname && mon_ux)  {
	fprintf(stderr,"Caution. Can only control local UX, ignoring hostname %s directive\n", hostname);
    }

    if (mon_off || mon_state)  {
	if (mon_control() < 0) return -1;
    }
    
    if (gmon_out != (char *) NULL) {
	if (mon_dump_info() < 0)  {
	    return -1;
	}
    }
}

int
mon_dump_info()
{
	char *mon_data;
	int mon_data_cnt;
	FILE *f;
	kern_return_t kr;

	f = fopen(gmon_out, "w");
	if (f == (FILE *)NULL) {
	    perror("fopen");
	    return -1;
	}
		
	kr = mon_dump(gprof_server_port,
		      &mon_data, &mon_data_cnt);
	if (kr != KERN_SUCCESS) {
	    mach_error("gprof_mon_dump", kr);
	    return -1;
	}
		
	kr = fwrite(mon_data, sizeof(char), mon_data_cnt, f);
	if (kr != mon_data_cnt) {
	    perror("fwrite");
	    return -1;
	}
	fclose(f);

	printf("monitoring data written to %s\n", gmon_out);
	return 0;
}

int
mon_control()
{
    kern_return_t kr;
    int status_only;
    char state[256];
    int ok;

    status_only = (mon_state < 0) ? 1 : 0;

    kr = mon_switch(gprof_server_port, &mon_state);

    if (kr != KERN_SUCCESS) {
	mach_error("gprof_mon_switch", kr);
	return -1;
    }

    ok = mon_state_to_name(mon_state, state);
   
    printf("monitor state %s %s (%d)\n", 
	(status_only ? "is" : "was"),
	(ok ? state : "invalid"), 
	mon_state);

    return 0;
}


mon_switch(p, f)
    mach_port_t p;
    int *f;
{
    if (mon_ux)  {
	return bsd_mon_switch(p, f);	
    } else {
	return gprof_mon_switch(p, f);
    }
}

mon_dump(p, d, c)
    mach_port_t p;
    char **d;
    int *c;
{
    if (mon_ux)  {
	return bsd_mon_dump(p, d, c);
    } else {
	return gprof_mon_dump(p, d, c);
    }
}


mon_get_port()
{
    char namebuf[128];
    kern_return_t kr;
    
    if (mon_pid == 0)  {
	/* bootstrap port is the unix server */
	kr = task_get_bootstrap_port(mach_task_self(), &gprof_server_port);
	if (kr == KERN_SUCCESS)  {
	    int stat = -1;
	    mon_ux = TRUE;		/* guess... */
	    kr = mon_switch(gprof_server_port, &stat);
	    if (kr == KERN_SUCCESS)  {
		return kr;
	    }
	}
    }
    mon_ux = FALSE;

    /* else, try to contact through the name server. could be a proxy */
    sprintf(namebuf,"gprof.%d", mon_pid);
    kr = netname_look_up(name_server_port, hostname,
			       namebuf, &gprof_server_port);

    if (gprof_debug)
    	fprintf(stderr,"NETNAME LOOKUP RETURNED %s.%s %d\n", hostname, namebuf, kr);
    
    return kr;
}



mach_port_t
mach_privileged_host_port()
{
    mach_port_t p;
    p = task_by_pid(-1);
    if (p == MACH_PORT_NULL)  {
	fprintf(stderr,"Warning: can't get priv host port. Using self\n");
	p = mach_task_self();
    }
    return p;
}
    

int
get_target_task(target_task)
    mach_port_t *target_task;    
{
    kern_return_t kr;
    int size;

    if (mon_ux)  {
	mon_pid = 0;
    }

    *target_task = task_by_pid(mon_pid);
    if (! MACH_PORT_VALID(*target_task))  {
	fprintf(stderr, "Cannot map Unix pid %d to Mach task port", mon_pid);
	return -1;
    }
    return 0;
}





int
get_exec_info(low_pc, high_pc)
    vm_offset_t *low_pc;
    vm_offset_t *high_pc;
{
    int pipedes[2];
    int forkret;
    int i;
    char addrbuf[16];
    char typebuf[4];
    char namebuf[64];
    int needfirsttextsym = TRUE;

    /* exec nm to get the symbols... */
    if (pipe(pipedes) < 0) {
	fprintf(stderr,"gprof: pipe creation failed");
	abort(-1);
    }
    if (forkret = fork()) {
	/* parent */
	if (forkret < 0) {
	    fprintf(stderr,"gprof: fork failed");
	    abort(-1);
	}
	close(0);
	if (dup(pipedes[0]) != 0) {
	    fprintf(stderr,"gprof: dup failed for gprof");
	    abort(-1);
	}
	close(pipedes[1]);
    }
    else {
	/* child: do the nm */
	close(1);
	if (dup(pipedes[1]) != 1) {
	    fprintf(stderr,"gprof: dup failed for nm");
	    abort(-1);
	}
	execl("/bin/nm", "nm", "-ng", exec_file_name, 0);
	fprintf(stderr,"execl /bin/nm returned!\n");
	abort(-1);
    }

    /* take the symbols from the output of nm */
    while (1) {
	/* plan for triples */
	scanf("%s", addrbuf);
	scanf("%s", typebuf);
	scanf("%s", namebuf);
	if (feof(stdin)) {
	    break;
	}

	if (needfirsttextsym) {
	    if (strcmp(namebuf,"__start") != 0)  {
		continue;
	    } else {
		sscanf(addrbuf,"%x", low_pc);
		needfirsttextsym = FALSE;
	    }
	} else if (typebuf[0] == 'T' || typebuf[0] == 't')  {
	    sscanf(addrbuf,"%x", high_pc); /* guess */
	} else break;
    }
    
    if (needfirsttextsym)  {
	fprintf(stderr, "failure. Can't find text symbols for %s\n",
		exec_file_name);
	return -1;
    }
    return 0;

}


int mon_get_pid()
{
    return mon_pid;
}

    


int
mon_proxy()
{
    mach_port_t task;
    vm_offset_t low_pc;
    vm_offset_t high_pc;
    kern_return_t kr;
    extern int (*gprof_get_export_pid)();
    

    kr = mon_get_port();
    if (kr == KERN_SUCCESS)  {
	fprintf(stderr,"Failure.  Target task already has a profiler port\n");
	return -1;
    }
    
    if (get_target_task(&task) < 0) {
	fprintf(stderr,"get_target_task failed\n");		
	return -1;
    }

    if (get_exec_info(&low_pc, &high_pc) < 0)  {
	fprintf(stderr,"get_exec_info failed\n");	
	return -1;
    }

    fprintf(stderr, "Task range is 0x%x 0x%x\n", low_pc, high_pc);
    
    if (gprof_set_target_info(task, low_pc, high_pc) < 0)  {
	fprintf(stderr,"gprof_set_target_task failed\n");
	return -1;
    }

    notify_init(task);

    gprof_get_export_pid = mon_get_pid;

    monstartup();
    
    if (gprof_control_server_init() != KERN_SUCCESS)  {
	fprintf(stderr,"detach_as_proxy: ");
	return -1;
    }

    if (mon_commands() < 0)  {
	fprintf(stderr,"mon commands failed\n");
	abort(-1);
    }

    notify_wait(task);

}


mach_port_t notify_port;
mach_port_t notify_task;



notify_init(task)
    mach_port_t task;
{
    kern_return_t kr;
    mach_port_t prev_port;

    kr = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_RECEIVE,
			    &notify_port);
    if (kr != KERN_SUCCESS)  {
	mach_error("mach_port_allocate, notify_port", kr);
	abort(-1);
    }

    kr = mach_port_request_notification(mach_task_self(),
					task,
					MACH_NOTIFY_DEAD_NAME,
					TRUE,
					notify_port,
					MACH_MSG_TYPE_MAKE_SEND_ONCE,
					&prev_port);
    if (kr != KERN_SUCCESS)  {
	mach_error("mach_port_request_notification, notify_port", kr);
	abort(-1);
    }
    return 0;
}


notify_wait(task)
    mach_port_t task;
{
    extern void notify_server();
    kern_return_t kr;
    
    notify_task = task;
    kr = mach_msg_server(notify_server, 8192, notify_port);
    mach_error("notify_wait notofy_server", kr);
}


    
    
    
kern_return_t
do_mach_notify_dead_name (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
    if (name != notify_task)  {
	fprintf(stderr, "Strange: task  died, but it was not our target task");
    }   else	        {
	notify_task = MACH_PORT_NULL;
	fprintf(stderr,"Warning: target died. Attempting to write profiling information to gmon.out\n");
	gmon_out = "gmon.out";
	if (mon_dump_info() < 0)  {
	    fprintf(stderr,"write failed\n");
	} else {
	    fprintf(stderr, "write succeeds\n");
	}
    }
    abort(2);
}


kern_return_t
do_mach_notify_msg_accepted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  printf ("do_mach_notify_msg_accepted : notify %x, name %x", notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_no_senders (notify, mscount)
     mach_port_t notify;
     mach_port_mscount_t mscount;
{
  printf ("do_mach_notify_no_senders : notify %x, mscount %x",
	   notify, mscount);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_deleted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  printf ("do_mach_notify_port_deleted : notify %x, name %x",
	   notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_destroyed (notify, rights)
     mach_port_t notify;
     mach_port_t rights;
{
  printf ("do_mach_notify_port_destroyed : notify %x, rights %x",
	   notify, rights);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_send_once (notify)
     mach_port_t notify;
{
#ifdef DUMP_SYSCALL
  /* MANY of these are generated. */
  printf ("do_mach_notify_send_once : notify %x", notify);
#endif
  return KERN_SUCCESS;
}    
    
    
    

