/*******************************************************************************
+
+  LEDA 3.5
+
+  window.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/

#ifndef LEDA_WINDOW_H
#define LEDA_WINDOW_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350136
#include <LEDA/REDEFINE_NAMES.h>
#endif

#include <LEDA/base_window.h>
#include <LEDA/plane.h>

/*{\Moptions usesubscripts=yes}*/

/*{\Manpage {window} {} {Windows}}*/

class window : public BASE_WINDOW {

/*{\Mdefinition
The data type $window$ provides an interface for graphical input and output 
of basic two-dimensional geometric objects.
Application programs using this data type have to be linked with {\em libW.a} 
and (on UNIX systems) with the X11 base library {\em libX11.a}
(cf.~section~1.6):

CC $prog.c$ -lW -lP -lG -lL -lX11 -lm

An instance $W$ of type $window$ is an iso-oriented rectangular window in the 
two-dimensional plane. The default representation of $W$ on the screen is a 
square of maximal possible edge length positioned in the upper right corner 
of the display. 

In general, a window consists of two rectangular sections,
a {\em panel section} in the upper part and 
a {\em drawing section} in the rest of the window. 
The panel section  contains {\em panel items} such as sliders, choice
fields, string items and buttons. They have to be created before
the window is opened by special panel operations described in section
\ref{panel-operations}.

The drawing section can be used for the output of geometric objects such
as points, lines, segments, arrows, circles, polygons, graphs, \dots and 
for the input of all these objects using the mouse input device.  
All drawing and input operations in the drawing section use a
coordinate system that is defined by three parameters of type $double$: 
$xmin$, the minimal x-coordinate, $xmax$, the maximal x-coordinate, 
and $ymin$, the minimal y-coordinate. The two parameters $xmin$ and $xmax$
define the scaling factor $scaling$  as $w/(xmax-xmin)$, where $w$ is the width 
of the drawing section in pixels. The maximal y-coordinate $ymax$ of the
drawing section is equal to $ymin + h \cdot  scaling$ and depends on the 
actual shape of the window on the screen. Here, $h$ is the height of the
drawing section in pixels.


A list of all window parameters:

\begin{enumerate}

\item
The {\em foreground color} parameter (default $black$) defines 
the default color to be used in all drawing operations.
There are 16 predefined colors (enumeration in \<LEDA/incl/impl/x\_window.h\>):
$black$, $white$, $red$, $green$, $blue$, $yellow$, $violet$, $orange$,
$cyan$, $brown$, $pink$, $green2$, $blue2$, $grey1$, $grey2$, $grey3$. 
Note that all drawing operations have an optional color argument that can
be used to override the default foreground color.
On monochrome systems all colors different from $white$ are displayed as 
$black$. 


\item
The {\em background color} parameter (default $white$) defines
the default background color (e.g. used by $W$.clear()).

\item
The {\em text font} parameter defines the name of the font to be used
in all text drawing operations.

\item
Minimal and maximal coordinates of the drawing area
$xmin$ (default 0), $xmax$ (default 100), $ymin$ (default 0).

\item
The {\em grid width} parameter (default 0) defines the width of the
grid that is used in the drawing area. A grid width of 0 indicates
that no grid is to be used.

\item
The {\em frame label} parameter defines a string to be displayed
in the frame of the window.

\item
The {\em show coordinates} flag  (default $true$) determines whether
the current coordinates of the mouse cursor in the drawing section
are displayed in the upper right corner.

\item
The {\em flush output} flag  (default $true$) determines whether
the graphics output stream is flushed after each draw action.

\item
The {\em line width} parameter (default value 1 pixel) defines the width of all
kinds of lines (segments, arrows, edges, circles, polygons).

\item
The {\em line style} parameter defines the style of lines. Possible line
styles are |solid| (default), |dashed|, and |dotted|.

\item
The {\em node width} parameter (default value 8 pixels) defines the diameter
of nodes created by the draw\_node and draw\_filled\_node operations.

\item
The {\em text mode} parameter defines how text is inserted into the window.
Possible values are |transparent| (default)  and |opaque|.

\item
The {\em drawing mode} parameter defines the logical operation that is used
for setting pixels in all drawing operations. Possible values are
|src_mode| (default) and |xor_mode|. In |src_mode| pixels are set to
the respective color value, in |xor_mode| the value is bitwise added to the
current pixel value.

\item
The {\em redraw function} parameter is a pointer to a function of type 
void ($*$F)(window$*$). It is called with a pointer to the corresponding window
as argument to redraw (parts of) the window whenever a redrawing 
is necessary, e.g., if the shape of the window is changed or previously
hidden parts of it become visible. 

\item
The {\em buttons per line} parameter (default $\infty$) defines
the maximal number of buttons in one line of the panel section.


\end{enumerate} }*/


private: 

// windows cannot be copied, use reference parameters! 

window(const window&) { }
window& operator=(const window&) { return *this; }

char* access_str(GenPtr p) { return ((string*)p)->cstring(); }
void  assign_str(GenPtr p, const char* s) { *((string*)p) = s; }

public:

enum { min = -1, center =-2, max = -3 };

static color fgcol;
static color bgcol;

static void win_error_handler(int i, const char* s);

#if defined(OLD_WINDOW_TYPE) 
#define INIT_BUTTONS  old_buttons()
#define OPEN_WINDOW   BASE_WINDOW::display(window::max,window::min,0)
#else
#define INIT_BUTTONS  std_buttons()
#define OPEN_WINDOW 
#endif


/*{\Mcreation W }*/ 

window();
/*{\Mcreate creates a squared window with maximal possible edge length
            (minimum of width and height of the display).}*/


window(const char* label);
/*{\Mcreate creates a maximal squared window with frame label $label$.}*/

window(float w, float h);
/*{\Mcreate creates a window $W$ of physical size $w$ pixels 
            $\times$ $h$ pixels .}*/


window(float w, float h, const char* label);
/*{\Mcreate creates a window $W$ of physical size $w$ pixels 
            $\times$ $h$ pixels and frame label $label$.}*/




#if defined(OLD_WINDOW_TYPE)
window(float w, float h, float x, float y, const char* label = LEDA::copyright_string) : BASE_WINDOW(int(w),int(h),label)
{ set_error_handler(win_error_handler); 
  old_buttons();
  if (w>0 && h>0) BASE_WINDOW::display(int(x),int(y),0);
 }
#endif



/*{\Mtext

\medskip
All four variants initialize the coordinates of $W$ to $xmin = 0$,
$xmax = 100$ and $ymin = 0$. The $init$ operation (see  below) can later
be used to change the window coordinates and scaling. Please note, that a 
window is not displayed before the function $display$ is called for it.}*/


~window() {}   // ~BASE_WINDOW does this job
 


/*{\Moperations 3.1 4.4}*/

/*{\Mtext
\headerline{3.1 Initialization} }*/

void init(double x0, double x1, double y0) { BASE_WINDOW::init(x0,x1,y0); }
/*{\Mopl     sets $xmin$ to $x_0$, $xmax$ to $x_1$, and $ymin$ to $y_0$, 
             the scaling factor $scaling$ to $w / (xmax-xmin)$, and 
             $ymax$ to $ymin + h \cdot scaling$. Here $w$ and $h$ are the
             width and height of the drawing section in pixels. }*/


void set_grid_mode(int d) {BASE_WINDOW::set_grid_mode(d);}
/*{\Mopl     sets the grid distance parameter of $W$ to $d$ pixels. }*/


void init(double x0, double x1, double y0, int d) 
{ BASE_WINDOW::init(x0,x1,y0,d);}
/*{\Mopl     same as $W$.init($x_0,x_1,y_0$) followed by 
             $W$.set\_grid\_mode($d$).}*/



void display() { BASE_WINDOW::display(window::max,window::min,0); }
/*{\Mopl      opens $W$ and displays it with its right upper corner in the 
              upper right corner of the 
              screen. Note that $W$.display() has to be called before
              all drawing operations and that all operations adding panel 
              items to $W$ (cf. \ref{panel-operations}) have to be called 
              before the first call of $W$.display(). }*/

void display(int x, int y) { BASE_WINDOW::display(x,y,0); }
/*{\Mopl      opens $W$ and displays it with its left upper corner at position
              $(x,y)$. Special 
              values for $x$ and $y$ are  $window::min$, $window::center$, 
              and $window::max$ for positioning $W$ at the minimal or maximal 
              $x$ or $y$ coordinate or centering it in the $x$ or $y$ 
              dimension. }*/

/*{\Moptions nextwarning=no}*/
void display(window& w, int x, int y) { BASE_WINDOW::display(x,y,&w); }
void display(int x, int y, window& W0) { BASE_WINDOW::display(x,y,&W0); }
/*{\Mopl      opens $W$ and displays it with its left upper corner at position 
              $(x,y)$ relative to the upper left corner of  window $W_0$.   }*/ 
/*{\Mtext
$W.open$\dots can be used as a synonym for $W.display$\dots\ Note,
that the $open$ operation for panels (cf. \ref{Panels}) is defined 
slightly different.}*/

void open()                        { display(); }
void open(int x, int y)            { display(x,y); }
void open(window& w, int x, int y) { display(w,x,y); }
void open(int x, int y, window& w) { display(x,y,w); }

void close() { BASE_WINDOW::close(); }
/*{\Mopl      closes $W$ by removing it from the display. }*/


void clear(color c=window::bgcol) { BASE_WINDOW::clear(c); };
/*{\Mopl      clears $W$ by filling it with color $c$ and
              empties the list of messages (see |W.message()|).}*/



/*{\Mtext
\headerline {3.2 Setting parameters}\label{parameters} }*/

color set_fg_color(color c) { return BASE_WINDOW::set_fg_color(c); }

color set_color(color c) { return BASE_WINDOW::set_fg_color(c); }
/*{\Mopl     sets the foreground color parameter to $c$ and
	     returns its previous value.}*/


color set_bg_color(color c) { return BASE_WINDOW::set_bg_color(c); }
/*{\Mopl     sets the background color parameter to $c$ and
	     returns its previous value.}*/

int set_line_width(int pix){return BASE_WINDOW::set_line_width(pix);}
/*{\Mopl     sets the line width parameter to $pix$ pixels and
	     returns its previous value.}*/

line_style set_line_style(line_style s)
{return BASE_WINDOW::set_line_style(s);}
/*{\Mopl     sets the line style parameter to $s$ and returns its
	     previous value.}*/

int set_node_width(int pix) {return BASE_WINDOW::set_node_width(pix);}
/*{\Mopl     sets the node width parameter to $pix$ pixels and
	     returns its previous value.}*/

text_mode set_text_mode(text_mode m)
{return BASE_WINDOW::set_text_mode(m);}
/*{\Mopl     sets the text mode parameter to $m$ and returns
	     its previous value.}*/

drawing_mode set_mode(drawing_mode m)
{return BASE_WINDOW::set_mode(m);}
/*{\Mopl     sets the drawing mode parameter to $m$ and returns 
	     its previous value.}*/

void set_show_coordinates(bool b) { BASE_WINDOW::set_show_coordinates(b); }
/*{\Mopl     sets the show coordinates flag to $b$. }*/

void set_frame_label(string s){ BASE_WINDOW::set_frame_label(s); }
/*{\Mopl     makes $s$ the window frame label. }*/

void reset_frame_label() { BASE_WINDOW::reset_frame_label(); }
/*{\Mop      restores the standard LEDA frame label. }*/

void set_show_coord_handler(void (*F)(window*,double,double)) 
{ BASE_WINDOW::set_coord_handler((coord_handler_func)F);}
/*{\Mopl     sets the show coordinate handler function parameter to $F$.}*/

void set_redraw(void (*F)(window*)) 
{ BASE_WINDOW::set_redraw((win_redraw_func1)F);}
/*{\Mopl     sets the redraw function parameter to $F$.}*/

void set_redraw(void (*F)(window*,double,double,double,double)=0) 
{ BASE_WINDOW::set_redraw((win_redraw_func2)F);}
/*{\Mopl     sets the redraw function parameter to $F$.}*/

void set_redraw(void (*F)()) { BASE_WINDOW::set_redraw((win_redraw_func0)F);}
/*{\Xopl     for backward compatibility. }*/

void start_timer(int msec, void (*F)(window*)=0) 
{ BASE_WINDOW::start_timer(msec,(win_redraw_func1)F);}
/*{\Mopl     }*/

void stop_timer() { BASE_WINDOW::stop_timer(); }
/*{\Mopl     }*/


void set_flush(bool b)     { BASE_WINDOW::set_flush(b); }
/*{\Mopl     sets the flush parameter to $b$.}*/


bool load_text_font(string fn)    { return BASE_WINDOW::load_text_font(fn); }
/*{\Xopl      loads $X11$ font $fn$ as text font. Returns true on success and 
              false if the font is not available.}*/

bool load_bold_font(string fn)    { return BASE_WINDOW::load_bold_font(fn); }
/*{\Xopl      loads $X11$ font $fn$ as bold font. Returns true on success and 
              false if the font is not available.}*/

bool loadfixed_font(string fn) { return BASE_WINDOW::load_fixed_font(fn); }
/*{\Xopl      load $X11$ font $fn$ as fixed font. Returns true on success 
              and false if the font is not available.}*/


/*{\Mtext
\headerline {3.3 Reading parameters} }*/

int get_line_width() {return BASE_WINDOW::get_line_width();}
/*{\Mop      returns the current line width.}*/ 

line_style get_line_style() {return BASE_WINDOW::get_line_style();}
/*{\Mop      returns the current line style.}*/ 

int get_node_width() {return BASE_WINDOW::get_node_width();}
/*{\Mop      returns the current node width.}*/ 

text_mode get_text_mode() {return BASE_WINDOW::get_text_mode();}
/*{\Mop      returns the current text mode.}*/ 

drawing_mode get_mode() {return BASE_WINDOW::get_mode();}
/*{\Mop      returns the current drawing mode.}*/ 

double xmin() {return BASE_WINDOW::xmin();}
/*{\Mop      returns the minimal x-coordinate of the drawing area of $W$.}*/ 

double ymin() {return BASE_WINDOW::ymin();}
/*{\Mop      returns the minimal y-coordinate of the drawing area of $W$.}*/ 

double xmax() {return BASE_WINDOW::xmax();}
/*{\Mop      returns the maximal x-coordinate of the drawing area of $W$.}*/ 

double ymax() {return BASE_WINDOW::ymax();}
/*{\Mop      returns the maximal y-coordinate of the drawing area of $W$.}*/ 

double scale() {return BASE_WINDOW::scale();}
/*{\Mop      returns the scaling factor of the drawing area of $W$, i.e. the 
             number of pixels of a unit length line segment.}*/

int          get_grid_mode()  { return BASE_WINDOW::get_grid_mode(); }
/*{\Mop      returns the width of the current grid in pixels (zero if
             no grid is used). }*/


operator void*() { return (state==0) ? 0 : this; }


/*{\Mtext
\headerline {3.4 Drawing Operations}
All drawing operations have an optional color argument at the end of the
parameter list. If this argument is omitted the current foreground color 
(cf. section \ref{parameters}) of $W$ is used.
}*/



// points

/*{\Mtext
\headerline {3.4.1 Drawing points} 
\setopdims 1.2 4.4
}*/

void draw_point(double x, double y, color c=window::fgcol);
/*{\Mopl     draws the point $(x,y)$ (a cross of two short segments). }*/

void draw_point(const point& p, color c=window::fgcol);
/*{\Mopl     draws point $p$. }*/

/*{\Moptions nextwarning=no}*/
void draw_pix(double x, double y, color c=window::fgcol);
void draw_pixel(double x, double y, color c=window::fgcol) { draw_pix(x,y,c); }
/*{\Mopl     sets the color of the pixel at position $(x,y)$ to $c$. }*/


/*{\Moptions nextwarning=no}*/
void draw_pix(const point& p, color c=window::fgcol);
void draw_pixel(const point& p, color c=window::fgcol) { draw_pix(p,c); }
/*{\Mopl     sets the color of the pixel at position $p$ to $c$. }*/

void window::draw_pixels(const list<point>& L, color c=window::fgcol);
/*{\Mopl     sets the color of all pixels in $L$ to $c$. }*/

void window::draw_pixels(int n, double* xcoord, double* ycoord,
                                                color c=window::fgcol);
/*{\Mopl     draws all pixels $(xcoord[i],ycoord[i])$ for $0 \le i \le n-1$. }*/



// segments

/*{\Mtext
\headerline {3.4.2 Drawing line segments}
}*/

void draw_segment(double x1, double y1, double x2, double y2, color c=window::fgcol);
/*{\Mopl     draws a line segment from $(x_1,y_1)$ to $(x_2,y_2)$.}*/

void draw_segment(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws a line segment from point $p$ to point $q$.}*/

void draw_segment(const segment& s, color c=window::fgcol);
/*{\Mopl     draws line segment $s$.}*/

void window::draw_segments(const list<segment>& L, color c=window::fgcol);
/*{\Mopl     draws all segments in $L$.}*/


// lines

/*{\Mtext
\headerline {3.4.3 Drawing lines}
}*/

void draw_line(double x1, double y1, double x2, double y2, color c=window::fgcol);
/*{\Mopl     draws a straight line passing through points $(x_1,y_1)$ and 
	     $(x_2,y_2)$.}*/

void draw_line(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws a straight line passing through points $p$ and $q$.}*/

void draw_line(const segment& s, color c=window::fgcol);
/*{\Mopl     draws the line supporting segment $s$.}*/


void draw_line(const line& l, color c=window::fgcol);
/*{\Mopl     draws line $l$.}*/

void draw_hline(double y, color c=window::fgcol);
/*{\Mopl     draws a horizontal line with y-coordinate $y$. }*/

void draw_vline(double x, color c=window::fgcol);
/*{\Mopl     draws a vertical line with x-coordinate $x$. }*/



/*{\Mtext
\headerline {3.4.4 Drawing Rays}
}*/

void draw_ray(double x1, double y1, double x2, double y2, color c=window::fgcol);
/*{\Mopl     draws a ray starting in $(x_1,y_1)$ and passing through 
	     $(x_2,y_2)$.}*/

void draw_ray(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws a ray starting in $p$ and passing through $q$.}*/


void draw_ray(const segment& s, color c=window::fgcol);
/*{\Mopl     draws a ray starting in $s.source()$ containing $s$. }*/


void draw_ray(const ray& r, color c=window::fgcol);
/*{\Mopl     draws ray $r$.}*/



// drawing arcs

void draw_arc(double x1, double y1, double x2, double y2, double r, color c=window::fgcol);
/*{\Mopl     draws a circular arc with radius r from $(x_1,y_1)$ to 
             $(x_2,y_2)$ with the center lying  to the left of the
             directed segment $(x_1,y_1)\longrightarrow (x_2,y_2)$.  }*/

void draw_arc(const point& p, const point& q, double r, color c=window::fgcol);
/*{\Mopl     draws a circular arc with radius r from p to q 
             with the center lying  to the left of the
             directed segment $p\longrightarrow q$.  }*/

void draw_arc(const segment& s, double r, color c=window::fgcol);
/*{\Mopl     same as $W$.draw\_arc$(s.source(),s.target(),r,c)$ }*/



// arrows

/*{\Mtext
\headerline{3.4.5 Drawing arrows}
}*/

point draw_arrow_head(const point& p, double dir, color c=window::fgcol);
/*{\Mopl     draws an arrow head at position $p$ pointing to direction $dir$.}*/


void draw_arrow(double x1, double y1, double x2, double y2, color c =window::fgcol);
/*{\Mopl     draws an arrow pointing from $(x_1,y_1)$ to $(x_2,y_2)$.}*/

void draw_arrow(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws an arrow pointing from point $p$ to point $q$.}*/

void draw_arrow(const segment& s, color=window::fgcol);
/*{\Mopl     draws an arrow pointing from $s$.start() to $s$.end().}*/


void draw_arc_arrow(double x1, double y1, double x2, double y2, double r, color c=window::fgcol);
/*{\Mopl     draws a circular arc arrow with radius r pointing from $(x_1,y_1)$
             to $(x_2,y_2)$ with the center lying to the left of the
             directed segment $(x_1,y_1) \longrightarrow (x_2,y_2)$.  }*/

void draw_arc_arrow(const point& p, const point& q, double r, color c=window::fgcol);
/*{\Mopl     draws a circular arc arrow with radius r pointing from 
             p to q with the center lying  to the left of the
             directed segment $p\longrightarrow q$.  }*/

void draw_arc_arrow(const segment& s, double r, color c=window::fgcol);
/*{\Mopl     same as $W$.draw\_arc\_arrow$(s.source(),s.target(),r,c)$ }*/



//circles

/*{\Mtext
\headerline{3.4.6 Drawing circles}
}*/

void draw_circle(double x, double y, double r, color c=window::fgcol);
/*{\Mopl     draws the circle with center $(x,y)$ and radius $r$.}*/

void draw_circle(const point& p, double r, color c=window::fgcol);
/*{\Mopl     draws the circle with center $p$ and radius $r$.}*/

void draw_circle(const circle& C, color c=window::fgcol);
/*{\Mopl     draws circle $C$.}*/

void draw_ellipse(double x, double y, double r1, double r2, color c=window::fgcol);
/*{\Mopl     draws the ellipse with center $(x,y)$ and radii $r_1$ and $r_2$.}*/

void draw_ellipse(const point& p, double r1, double r2, color c=window::fgcol);
/*{\Mopl     draws the ellipse with center $p$ and radii $r_1$ and $r_2$.}*/



/*{\Mtext
\headerline{3.4.7 Drawing discs}
}*/

void draw_disc(double x, double y, double r, color c=window::fgcol);
/*{\Mopl     draws a filled circle with center $(x,y)$ and radius $r$.}*/

void draw_disc(const point& p, double r, color c=window::fgcol);
/*{\Mopl     draws a filled circle with center $p$ and radius $r$.}*/

void draw_disc(const circle& C, color c=window::fgcol);
/*{\Mopl     draws filled circle $C$.}*/


void draw_filled_ellipse(double x, double y, double r1, double r2, color c=window::fgcol);
/*{\Mopl  draws a filled ellipse with center $(x,y)$ and radii $r_1$ and $r_2$.}*/

void draw_filled_ellipse(const point& p, double r1, double r2, color c=window::fgcol);
/*{\Mopl  draws a filled ellipse with center $p$ and radii $r_1$ and $r_2$.}*/


//polygons 

/*{\Mtext
\headerline{3.4.8 Drawing polygons }
}*/

void draw_polygon(const list<point>& lp, color c=window::fgcol);
/*{\Mopl     draws the polygon with vertex sequence $lp$.}*/

void draw_polygon(const polygon& P, color c=window::fgcol);
/*{\Mopl     draws polygon $P$.}*/

void draw_filled_polygon(const list<point>& lp, color c=window::fgcol);
/*{\Mopl     draws the filled polygon with vertex sequence $lp$.}*/

void draw_filled_polygon(const polygon& P, color c=window::fgcol);
/*{\Mopl     draws filled polygon $P$.}*/


void draw_rectangle(double x0, double  y0, double x1, double y1, color=window::fgcol);
/*{\Mopl     draws a rectangle with lower left corner $(x_0,y_0)$ and upper
             right corner $(x_1,y_1)$.\\ \precond $x_0 < x_1$ and $y_0 < y_1$. }*/

void draw_rectangle(point p, point q, color=window::fgcol);
/*{\Mopl     draws a rectangle with lower left corner $p$ and upper
             right corner $q$.\\ \precond $p < q$. }*/


void draw_filled_rectangle(double,double,double,double,color=window::fgcol);

void draw_box(double x0, double y0, double x1, double y1,color c=window::fgcol)
{ draw_filled_rectangle(x0,y0,x1,y1,c); }
/*{\Mopl     draws a filled rectangle with lower left corner $(x_0,y_0)$ and upper
             right corner $(x_1,y_1)$.\\ \precond $x_0 < x_1$ and $y_0 < y_1$. }*/

void draw_filled_rectangle(point p, point q, color=window::fgcol);

void draw_box(point p, point q, color c=window::fgcol)
{ draw_filled_rectangle(p,q,c); }
/*{\Mopl     draws a filled rectangle with lower left corner $p$ and upper
             right corner $q$.\\ \precond $p < q$. }*/


void draw_triangle(point a, point b, point c, color=window::fgcol);
/*{\Mopl     draws triangle $(a,b,c)$. }*/

void draw_filled_triangle(point a, point b, point c, color=window::fgcol);
/*{\Mopl     draws filled triangle $(a,b,c)$. }*/



// functions

/*{\Mtext
\headerline{3.4.9 Drawing functions}
}*/

void plot_xy(double x0, double x1, win_draw_func F, color c=window::fgcol);
/*{\Mopl     draws the graph of function $F$ in the x-range $[x_0,x_1]$, i.e., 
             all pixels $(x,y)$ with $y = F(x)$ and $x_0\le x\le x_1$.}*/

void plot_yx(double y0, double y1, win_draw_func F, color c=window::fgcol);
/*{\Mopl     draws the graph of function $F$ in the y-range $[y_0,y_1]$, i.e., 
             all pixels $(x,y)$ with $x = F(y)$ and $y_0\le y\le y_1$.}*/




// text

/*{\Mtext
\headerline{3.4.10 Drawing text}
}*/

void draw_text(double x, double y, string s, color c=window::fgcol);
/*{\Mopl     writes string $s$ starting at position $(x,y)$.}*/

void draw_text(const point& p, string s, color c=window::fgcol);
/*{\Mopl     writes string $s$ starting at position $p$.}*/

void draw_ctext(double x, double y, string s, color c=window::fgcol);
/*{\Mopl     writes string $s$ centered at position $(x,y)$.}*/

void draw_ctext(const point& p, string s, color c=window::fgcol);
/*{\Mopl     writes string $s$ centered at position $p$.}*/

void draw_ctext(string s, color c=window::fgcol);
/*{\Mopl     writes string $s$ centered in window $W$.}*/


void message(string s) {BASE_WINDOW::message(s);};
/*{\Mop      displays the message $s$ (each call adds a new line).}*/

/*{\Moptions nextwarning=no}*/
void del_message()  { BASE_WINDOW::del_messages(); };
void del_messages() { BASE_WINDOW::del_messages(); };
/*{\Mop      deletes the text written by all previous message 
	     operations.}*/



// nodes

/*{\Mtext
\headerline{3.4.11 Drawing nodes}
Nodes are represented by circles of diameter $node\_width$.
}*/

void draw_node(double x0, double y0, color c=window::fgcol);
/*{\Mopl     draws a node at position $(x_0,y_0)$.}*/

void draw_node(const point& p, color c=window::fgcol);
/*{\Mopl     draws a node at position $p$.}*/

void draw_filled_node(double x0, double y0, color c=window::bgcol);
/*{\Mopl     draws a filled node at position $(x_0,y_0)$.}*/

void draw_filled_node(const point& p, color c=window::bgcol);
/*{\Mopl     draws a filled node at position $p$.}*/

void draw_text_node(double x, double y, string s, color c=window::bgcol);
/*{\Mopl     draws a node with label $s$ at position $(x,y)$. }*/

void draw_text_node(const point& p, string s, color c=window::bgcol);
/*{\Mopl     draws a node with label $s$ at position $p$. }*/

void draw_int_node(double x, double y, int i, color c=window::bgcol);
/*{\Mopl     draws a node with integer label $i$ at position 
	     $(x,y)$. }*/

void draw_int_node(const point& p, int i, color c=window::bgcol);
/*{\Mopl     draws a node with integer label $i$ at position  $p$. }*/



// edges

/*{\Mtext
\headerline{3.4.12 Drawing edges}
Edges are drawn as straigth line segments or arrows with a clearance of 
$node\_width/2$ at each end.
}*/

void draw_edge(double x1, double y1, double x2, double y2, color c=window::fgcol);
/*{\Mopl     draws an edge from $(x_1,y_1)$ to $(x_2,y_2)$.}*/

void draw_edge(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws an edge from $p$ to $q$.}*/

void draw_edge(const segment& s, color c=window::fgcol);
/*{\Mopl     draws an edge from $s$.start() to $s$.end().}*/

void draw_edge_arrow(double x1, double y1, double x2, double y2, color c=window::fgcol);
/*{\Mopl     draws a directed edge from $(x_1,y_1)$ to $(x_2,y_2)$.}*/

void draw_edge_arrow(const point& p, const point& q, color c=window::fgcol);
/*{\Mopl     draws a directed edge from $p$ to $q$.}*/

void draw_edge_arrow(const segment& s, color c=window::fgcol);
/*{\Mopl     draws a directed edge from $s$.start() to $s$.end().}*/ 

void draw_arc_edge(double x1, double y1, double x2, double y2, double r, color c=window::fgcol);
/*{\Mopl     draws a circular edge arc with radius $r$ from $(x_1,y_1)$ to 
             $(x_2,y_2)$ with the center lying  to the left of the
             directed segment $(x_1,y_1)\longrightarrow (x_2,y_2)$.  }*/

void draw_arc_edge(const point& p, const point& q, double r, color c=window::fgcol);
/*{\Mopl     draws a circular edge arc with radius r from p to q 
             with the center lying  to the left of the
             directed segment $p\longrightarrow q$.  }*/

void draw_arc_edge(const segment& s, double r, color c=window::fgcol);
/*{\Mopl     same as $W$.draw\_arc\_edge$(s.source(),s.target(),r,c)$. }*/ 


void draw_arc_edge_arrow(double x1, double y1, double x2, double y2, double r, color c=window::fgcol);
/*{\Mopl     draws a circular directed edge arc with radius $r$ from $(x_1,y_1)$
             to $(x_2,y_2)$ with the center lying  to the left of the
             directed segment $(x_1,y_1)\longrightarrow (x_2,y_2)$.  }*/

void draw_arc_edge_arrow(const point& p, const point& q, double r, color c=window::fgcol);
/*{\Mopl     draws a circular directed edge arc with radius r from p 
             to q with the center lying  to the left of the
             directed segment $p\longrightarrow q$.  }*/

void draw_arc_edge_arrow(const segment& s, double r, color c=window::fgcol);
/*{\Mopl     same as $W$.draw\_arc\_edge\_arrow$(s.source(),s.target(),r,c)$. }*/ 



/*{\Mtext
\headerline{3.4.13 Bitmaps and Pixrects}
}*/

char* create_bitmap(int w, int h, char* bm_data)
{ return BASE_WINDOW::create_bitmap(w,h,bm_data); }
/*{\Mopl   creates a bitmap (monochrome pixrect) of width $w$, height $h$, 
           from the bits in |data|. }*/

char* create_pixrect(char** xpm)
{ return BASE_WINDOW::create_pixrect(xpm); }


char* create_pixrect(int w, int h, char* bm_data, int fg=window::fgcol, 
                                               int bg=window::bgcol)
{ return BASE_WINDOW::create_pixrect(w,h,bm_data,fg,bg); }
/*{\Mopl   creates a pixrect of width $w$, height $h$, foreground color 
           $fg$, and background color $bg$ from bitmap |data|. }*/


char* get_pixrect(double x1, double y1, double x2, double y2)
{ return BASE_WINDOW::get_pixrect(x1,y1,x2,y2); }
/*{\Mopl   creates a color pixrect of width $w=x_2-x_1$, height $h=y2-y1$,
           and  copies all pixels from the rectangular area $(x_1,x_2,y_1,y_2)$
           of $W$ into it. }*/

void  put_pixrect(double x, double y, char* pr)
{ BASE_WINDOW::put_pixrect(x,y,pr); }
/*{\Mopl   copies the contents of pixrect $pr$ with lower left corner at
           position $(x,y)$ into $W$. }*/

void  put_bitmap(double x, double y, char* bm, color c=window::fgcol)
{ BASE_WINDOW::put_bitmap(x,y,bm,c); }
/*{\Mopl   draws all pixels corresponding to 1-bits in $bm$ with color $c$
           here the lower left corner of $bm$ corresponds to the pixel at
           position $(x,y)$ in $W$. }*/

void  put_pixrect(double x, double y, char* pr,int x0, int y0, int w, int h)
{ BASE_WINDOW::put_pixrect(x,y,pr,x0,y0,w,h); }
/*{\Mopl   copies (pixel) rectangle $(x0,y0,x0+w,y0+h)$ of $pr$ with lower left 
           corner at position $(x,y)$ into $W$. }*/


void  put_pixrect(char* pr) { BASE_WINDOW::put_pixrect(pr); }


void  del_bitmap(char* bm) { BASE_WINDOW::del_bitmap(bm); }
/*{\Mopl   destroys bitmap $bm$. }*/

void  del_pixrect(char* pr) { BASE_WINDOW::del_pixrect(pr); }
/*{\Mopl   destroys pixrect $pr$. }*/


void  copy_rect(double x1, double y1, double x2, double y2, double x, double y)
{ BASE_WINDOW::copy_rect(x1,y1,x2,y2,x,y); }
/*{\Mopl   same as |W.put_pixrect(x,y,W.get_pixrect(x1,y1,x2,y2))|. }*/




// mouse input

/*{\Mtext
\headerline{3.5 Input}
The main input operation for reading positions, mouse clicks, and buttons
from a window $W$ is the operation $W$.read\_mouse(). This operation is 
blocking, i.e., waits for a button to be pressed which is either a ``real''
button on the mouse device pressed inside the drawing area of $W$ or a 
button in the panel section of $W$.
In both cases, the number of the selected button is returned. Mouse
buttons have pre-defined numbers MOUSE\_BUTTON(1) for the left button, 
MOUSE\_BUTTON(2) for the middle button, and MOUSE\_BUTTON(3) for the
right button. The numbers of the panel buttons can be defined by the
user. If the selected button has an associated action function or sub-window 
this function/window is executed/opened (cf. \ref{panel-operations} for
details).

There is also a non-blocking version $W$.get\_mouse() which returns 
the constant NO\_BUTTON if no button was pressed. 

The window data type also provides two more general input operations 
$W$.read\_event() and $W$.get\_event() for reading events. They return 
the event type (enumeration in \<LEDA/impl/x\_window.h\>), the value of 
the event, the position of the event in the drawing section, and a time 
stamp of the event.

}*/

/*{\Mtext
\headerline{3.5.1 Read Mouse}
}*/

int read_mouse();
/*{\Mop      displays the cursor on the screen until a mouse button is pressed
             inside of the drawing area or until a button of the panel section
             is selected.  In both cases, the number $n$ of the button is 
             returned which is one of the predefined constants 
             MOUSE\_BUTTON($i$) with $i\in\{1,2,3\}$ for mouse buttons and
             a user defined value (defined when adding the button with 
             $W$.button()) for panel buttons.
             If the button has an associated action function this function is 
             called with parameter $n$. If the button has an associated
             window $M$ it is opened and $M$.read\_mouse() is returned. }*/

int read() { return read_mouse(); }

int read_mouse(double& x, double& y);
/*{\Mopl     If a button is pressed inside the drawing area the 
             current position of the cursor is assigned to $(x,y)$. 
             The operation returns the number of the pressed button
             (see $W$.read\_mouse().)}*/ 

int read_mouse(point& p);
/*{\Mopl     If a button is pressed inside the drawing area the 
             current position of the cursor is assigned to $p$.
             The operation returns the number of the pressed button
             (see $W$.read\_mouse().)}*/ 


int read_mouse_seg(double x0, double y0, double& x, double& y);
/*{\Mopl     displays a line segment from $(x_0,y_0)$ to the
	     current cursor position until a mouse button is
	     pressed inside the drawing section of $W$. When a 
             button is pressed the current position is assigned to $(x,y)$ 
             and the number of the pressed button is returned.}*/

int read_mouse_seg(const point& p, point& q);
/*{\Mopl     displays a line segment from $p$ to the current 
             cursor position until a mouse button is pressed 
             inside the drawing section of $W$.
             When a button is pressed the current position is 
             assigned to $q$ and the number of the pressed button 
             is returned.}*/

int read_mouse_rect(double x0, double y0, double& x, double& y);
/*{\Mopl     displays a rectangle with diagonal from $(x_0,y_0)$ 
     	     to the current cursor position until a mouse button 
	     is pressed inside the drawing section of $W$.
             When a button is pressed the current 
	     position is assigned to $(x,y)$ and the number of the 
             pressed button is returned.}*/

int read_mouse_rect(const point& p, point& q);
/*{\Mopl     displays a rectangle with diagonal from $p$ 
     	     to the current cursor position until a mouse button 
	     is pressed inside the drawing section of $W$.
             When a button is pressed the current 
	     position is assigned to $q$ and the number of the
             pressed button is returned.}*/

int read_mouse_circle(double x0, double y0, double& x, double& y);
/*{\Mopl     displays a circle with center $(x_0,y_0)$ passing 
	     through the current cursor position until a mouse 
	     button is pressed inside the drawing section of $W$.
             When a button is pressed the 
	     current position is assigned to $(x,y)$ and the 
	     number of the pressed button is returned.}*/

int read_mouse_circle(const point& p, point& q);
/*{\Mopl     displays a circle with center $p$ passing 
	     through the current cursor position until a mouse 
	     button is pressed inside the drawing section of $W$.
             When a button is pressed the 
	     current position is assigned to $q$ and the 
	     number of the pressed button is returned.}*/

int read_mouse_action(mouse_action_func, double&, double&);
int read_mouse_action(mouse_action_func, point&);


int get_mouse();
/*{\Mop     non-blocking read operation, i.e., if a button was pressed 
            its number is returned, otherwise the constant NO\_BUTTON is 
            returned. }*/

int get_mouse(double& x, double& y);
/*{\Mopl    if a mouse button was pressed the corresponding position is
            assigned to $(x,y)$ and the button number is returned, 
            otherwise the constant NO\_BUTTON is returned. }*/

int get_mouse(point& p);
/*{\Mop     if a mouse button was pressed the corresponding position is
            assigned to $p$ and the button number is returned, 
            otherwise the constant NO\_BUTTON is returned. }*/


// for backward compatibility
int get_button()                     { return get_mouse(); }
int get_button(double& x, double& y) { return get_mouse(x,y); }
int get_button(point& p)             { return get_mouse(p); }


/*{\Mtext
\bigskip
{\bf 3.5.2 Events}
}*/

int  read_event(int& val, double& x, double& y, unsigned long& t)
{ return BASE_WINDOW::read_event(val,x,y,t); }

int  read_event(int& val, double& x, double& y, unsigned long& t, int timeout)
{ return BASE_WINDOW::read_event(val,x,y,t,timeout); }

int  read_event(int& val, double& x, double& y)
{ return BASE_WINDOW::read_event(val,x,y); }
/*{\Mopl   waits for next event in window $W$ and returns it. 
           Assigns the button or key to $val$ and the position
           in $W$ to  $(x,y)$. Possible events are 
           (cf. \<LEDA/impl/x\_window.h\>): 
           key\_press\_event, key\_release\_event,
           button\_press\_event, button\_release\_event, 
           configure\_event, motion\_event, destroy\_event. }*/

int  get_event(int& val, double& x, double& y)
{ return BASE_WINDOW::get_event(val,x,y); }
/*{\Mopl   if there is an event for window $W$ in the event queue a 
           $W.read\_event$ operation is performed, otherwise the integer 
           constant $no\_event$ is returned. }*/


// get shift key info for last handled mouse button event

bool shift_key_down() { return BASE_WINDOW::shift_key_down(); }
/*{\Mop   returns $true$ if a {\em shift} key was pressed during
          the last handled mouse button event. }*/ 

bool ctrl_key_down()  { return BASE_WINDOW::ctrl_key_down(); }
/*{\Mop   returns $true$ if a {\em ctrl} key was pressed during
          the last handled mouse button event. }*/ 

bool alt_key_down()  { return BASE_WINDOW::alt_key_down(); }
/*{\Mop   returns $true$ if an {\em alt} key was pressed during
          the last handled mouse button event. }*/ 


int button_press_time()
{ return (int)BASE_WINDOW::button_press_time(); }
/*{\Mop     returns $X11$ time-stamp of last button press event. }*/


int button_release_time()
{ return (int)BASE_WINDOW::button_release_time(); }
/*{\Mop     returns $X11$ time-stamp of last button release event. }*/
        
 

/*{\Mtext
\bigskip
{\bf 3.6 Panel Input}\\
The operations listed in this section are useful for simple input of
strings, numbers, and Boolean values.
}*/

bool    confirm(string s);
/*{\Mop      displays string $s$ and asks for confirmation. 
	     Returns true iff the answer was ``yes''.}*/

/*{\Moptions nextwarning=no}*/
void    notice(string s);
void    acknowledge(string s);
/*{\Mopl     displays string $s$ and asks for acknowledgement.}*/


int     read_panel(string h, int n, string* S);
/*{\Mopl     displays a panel with header $h$ and an array of $n$ 
             buttons with labels $S[0..n-1]$, returns the index of 
             the selected button.}*/

int     read_vpanel(string h, int n, string* S);
/*{\Mopl     like read\_panel with vertical button layout.}*/

string  read_string(string p);
/*{\Mop      displays a panel with prompt $p$ for string input, 
    	     returns the input.}*/

double  read_real(string p);
/*{\Mop      displays a panel with prompt $p$ for double input 
 	     returns the input.}*/

int     read_int(string p);
/*{\Mop      displays a panel with prompt $p$ for integer input, 
	     returns the input.}*/




// I/O operators
/*{\Mtext
\bigskip
{\bf 3.7 Input and output operators}\\
For input and output of basic geometric objects in the plane such as points, 
lines, line segments, circles, and polygons the \<\< and \>\>
 operators can be used. Similar to \CC input streams windows have an internal 
state indicating whether there is more input to read or not. Its initial value 
is true and it is turned to false if an input sequence is terminated  by 
clicking the right mouse button (similar to ending stream input by the eof 
character). In conditional statements, objects of type $window$ are 
automatically converted to boolean by returning this internal state. Thus, 
they can be used in conditional statements in the same way as \CC input 
streams. For example, to read a sequence of points terminated by a right 
button click,  use `` {\bf while} ($W\ \>\>\ p$) $\{\  \dots\ \}$ ''. 
}*/

/*{\Mtext
\headerline{3.7.1 Output}
\setopdims 2.5 4.4
}*/

window& draw(const point& p,color c=window::fgcol)   
{ draw_point(p,c); return *this; }

window& draw(const segment& s,color c=window::fgcol) 
{ draw_segment(s,c); return *this;}

window& draw(const ray& r,color c=window::fgcol) 
{ draw_ray(r,c); return *this;}

window& draw(const line& l,color c=window::fgcol)    
{ draw_line(l,c); return *this; }

window& draw(const circle& C,color c=window::fgcol)  
{ draw_circle(C,c); return *this;}

window& draw(const polygon& P,color c=window::fgcol)
{ draw_polygon(P,c); return *this;}

 

window& operator<<(const point& p)   { return draw(p); }
/*{\Mbinop   like $W$.draw\_point($p$).}*/

window& operator<<(const segment& s) { return draw(s); }
/*{\Mbinop   like $W$.draw\_segment($s$).}*/

window& operator<<(const ray& r) { return draw(r); }
/*{\Mbinop   like $W$.draw\_ray($r$).}*/

window& operator<<(const line& l)    { return draw(l); }
/*{\Mbinop   like $W$.draw\_line($l$).}*/

window& operator<<(const circle& C)  { return draw(C); }
/*{\Mbinop   like $W$.draw\_circle($C$).}*/

window& operator<<(const polygon& P) { return draw(P); }
/*{\Mbinop   like $W$.draw\_polygon($P$).}*/

/*{\Mtext
\bigskip
{\bf 3.7.2 Input}
}*/

window& read(point&);
window& read(segment&);
window& read(ray&);
window& read(line&);
window& read(circle&);
window& read(polygon&);

window& operator>>(point& p);
/*{\Mbinop   reads a point $p$: clicking the left button 
	     assigns the current cursor position to $p$.}*/

window& operator>>(segment& s);
/*{\Mbinop   reads a segment $s$: use the left button to input 
	     the start and end point of $s$.}*/

window& operator>>(ray& r);
/*{\Mbinop   reads a ray $r$: use the left button to input 
	     the start point and a second point on $r$.}*/

window& operator>>(line& l);
/*{\Mbinop   reads a line $l$: use the left button to input 
	     two different points on $l$.}*/

window& operator>>(circle& C);
/*{\Mbinop   reads a circle $C$: use the left button to input 
	     the center of $C$ and a point on $C$.}*/

window& operator>>(polygon& P);
/*{\Mbinop   reads a polygon $P$: use the left button to input 
	     the sequence of vertices of $P$, end the sequence 
	     by clicking the right button.}*/

/*{\Mtext
As long as an input operation has not been completed the last read point can 
be erased by simultaneously pressing the shift key and the left mouse button.
}*/


/*{\Mtext
\bigskip
{\bf 3.8 Non-Member Functions} 
}*/

friend int  read_event(window*& w, int& val, double& x, double& y)
{ return BASE_WINDOW::read_event((BASE_WINDOW*&)w,val,x,y); }
/*\{\Mfuncl waits for next event and returns it. Assigns the window to
            $w$, the button or key to $val$ and the position in $w$ to 
            $(x,y)$. Possible events are (cf. <LEDA/impl/x_window.h>): 
            key_press_event, key_release_event,
            button_press_event, button_release_event, 
            configure_event,motion_event, destroy_event. }*/


friend int  get_event(window*& w, int& val, double& x, double& y)
{ return BASE_WINDOW::get_event((BASE_WINDOW*&)w,val,x,y); }
/*\{\Mfuncl if the event queue is not empty a $read_event$ operation
            is performed, otherwise the integer constant $no\_event$ is 
            returned. }*/


friend int read_mouse(window*& w, double& x, double& y)
{ return BASE_WINDOW::read_mouse((BASE_WINDOW*&)w,x,y); }
/*{\Mfuncl  waits for mouse input, assigns a pointer to the 
            corresponding window to $w$ and the position in 
            $*w$ to $(x,y)$ and returns the pressed button. }*/

friend int get_mouse(window*& w, double& x, double& y)
{ return BASE_WINDOW::get_mouse((BASE_WINDOW*&)w,x,y); }
/*{\Mfuncl  non-blocking variant of |read_mouse|, returns
            NO\_BUTTON if no button was pressed. }*/
 
        
friend void put_back_event() { BASE_WINDOW::put_back_event(); }
/*{\Mfunc   puts last handled event back to the event queue. }*/


//------------------------------------------------------------------------------
// panel operations
//------------------------------------------------------------------------------

/*{\Mtext
\bigskip
{\bf 3.9 Panel Operations \label{panel-operations} }

The panel section of a window is used for displaying text messages and 
for updating the values of variables. It consists of a list of panel items and 
a list of buttons.  
The operations in this section add panel items or buttons to the panel section 
of $W$. Note that they have to be called before the window is displayed the
first time.

In general, a panel item consists of a string label and an associated variable
of a certain type (int, bool, string, double, color). The value of this variable
can be manipulated through the item.
Each button has a label (displayed on the button) and an associated number.
The number of a button is either defined by the user or is the rank of the
button in the list of all buttons. If a button is pressed  (i.e. selected
by a mouse click) during a $read\_mouse$ operation its number is returned.

{\em Action functions} can be associated with buttons and some items 
(e.g. slider items) whenever a button with an associated action function
is pressed this function is called with the number of the button as
actual parameter. Action functions of items are called whenever the
value of the corresponding variable is changed with the new value
as actual parameter. All action functions must have the type
$void\ \ func(int)$. 

Another way to define a button is to associate another window with it.
In this case the button will have a menu sign and
as soon as it is pressed the attached window will open. 
This method can be used to implement pop-up menues.
The return value of the current $read\_mouse$ operation will be 
the number associated with the button in the menu.
}*/

void buttons_per_line(int n) { BASE_WINDOW::buttons_per_line(n); }
/*{\Mop  defines the maximal number $n$ of buttons per line. }*/


panel_item text_item(string s);
/*{\Mop      adds a text\_item $s$ to $W$.}*/

panel_item bool_item(string s, bool& x, const char* hlp=0);
/*{\Mopl     adds a boolean item with label $s$ and variable $x$ to $W$.}*/

panel_item bool_item(string s, bool& x, void (*F)(int), const char* hlp=0);
/*{\Mopl     as above with action function |F|.}*/

panel_item real_item(string s, double& x, const char* hlp=0);
/*{\Mopl     adds a real item with label $s$ and variable $x$ to $W$.}*/

panel_item double_item(string s, double& x, const char* hlp=0);

panel_item color_item(string s, color& x, const char* hlp=0);
/*{\Mopl     adds a color item with label $s$ and variable $x$ to $W$.}*/

panel_item color_item(string s, color& x, void (*F)(int), const char* hlp=0);
/*{\Mopl     as above with action function |F|.}*/

panel_item lstyle_item(string s, line_style& x, const char* hlp=0);
/*{\Mopl     adds a line style item with label $s$ and variable $x$ to $W$.}*/

panel_item int_item(string s, int& x, const char* hlp=0);
/*{\Mopl     adds an integer item with label $s$ and variable $x$ to $W$.}*/

panel_item int_item(string s, int& x, int l, int h, int step,const char* hlp=0);
/*{\Mopl     adds an integer choice item with label $s$, variable $x$, 
	     range $l$,\dots, $h$, and step size $step$ to $W$.}*/

panel_item int_item(string s, int& x, int l, int h, int step,void (*F)(int),
                                                             const char* hlp=0);
/*{\Mopl     adds an integer choice item with label $s$, variable $x$, 
	     range $l$,\dots, $h$, and step size $step$ to $W$. Function 
             $F(x)$ is executed whenever the value of $x$ is changed. }*/

panel_item int_item(string s, int& x, int l, int h, const char* hlp=0);
/*{\Mopl     adds an integer slider item with label $s$, variable $x$, and 
	     range $l$,\dots,$h$ to $W$.}*/

panel_item int_item(string s, int& x, int l, int h, void (*F)(int),const char* hlp=0);
/*{\Mopl     adds an integer slider item with label $s$, variable $x$, and 
	     range $l$,\dots,$h$ to $W$. Function $F(x)$ is executed whenever
             the value of $x$ has changed by moving the slider. }*/

panel_item string_item(string s, string& x, void (*F)(char*), 
                                            const char* hlp=0);


panel_item string_item(string s, string& x, const char* hlp=0);
/*{\Mopl     adds a string item with label $s$ and variable $x$ to $W$.}*/


panel_item string_item(string s, string& x, const list<string>& L, 
                                            void (*F)(char*),const char* hlp=0);

panel_item string_item(string s, string& x, const list<string>& L, const char* hlp=0);
/*{\Mopl     adds a string item with label $s$, variable $x$, and menu $L$ 
	     to $W$.}*/


void add_menu(panel_item it, const list<string>& L);


// choice items

panel_item choice_item(string s, int& x, const list<string>& L, 
                       void (*F)(int)=0, const char* hlp=0);
/*{\Mopl     adds an integer item with label $s$, variable $x$, and choices 
	     from $L$ to $W$.}*/

panel_item  choice_item(string label,int& x,const char* hlp,int n, ... );

panel_item choice_item(string,int& x,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,char*,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,char*,char*,char*,char*,
                       const char* =0);
panel_item choice_item(string,int& x,char*,char*,char*,char*,char*,char*,char*,
                       char*, const char* =0);

/*{\Moptions nextwarning=no}*/
/*
panel_item choice_item(string s, int& x, string s1,...,string sk, const char* hlp=0);
*/
/*{\Mopl     adds an integer item with label $s$, variable $x$, and choices 
	     $s_1$, \dots, $s_k$ to $W$ ($k \le 8$).}*/

panel_item choice_mult_item(string s,int& x,const list<string>& L, const char* hlp=0);
/*{\Mopl  }*/

panel_item choice_mult_item(string s,int& x,const list<string>& L,void (*F)(int), 
                                                            const char* hlp=0);
/*{\Mopl  }*/




// bitmap choice items

panel_item choice_item(string s, int& x, int n, int w, int h, char** bm, 
                                                        const char* hlp=0);
/*{\Mopl     adds an integer item with label $s$, variable $x$, and $n$
             bitmaps |bm[0]|, \dots, |bm[n-1]| each of width $w$ 

             and height $h$.}*/ 

panel_item choice_item(string s, int& x, int n, int w, int h,char** bm, 
                                                             void (*F)(int), 
                                                             const char* hlp=0);
/*{\Mopl  }*/


panel_item choice_mult_item(string s, int& x, int n, int w, int h, char** bm, 
                                                             const char* hlp=0);
/*{\Mopl  }*/

panel_item choice_mult_item(string s, int& x, int n, int w, int h, char** bm, 
                                                             void (*F)(int), 
                                                             const char* hlp=0);
/*{\Mopl  }*/




int  button(string s, int n, const char* hlp=0);
/*{\Mop      adds a button with label $s$ and number $n$ to $W$.}*/

int  button(string s, const char* hlp=0);
/*{\Mop   adds a new button to $W$ with label $s$ and number equal to its
          position in the list of all buttons (starting with $0$).}*/

int  button(int w, int h, char* bm, string s, int n, const char* hlp=0);
/*{\Mop    adds a button with bitmap $bm$, label $s$, and number $n$ to $W$.}*/

int  button(int w, int h, char* bm, string s,const char* hlp=0);
/*{\Mop   adds a new button to $W$ with bitmap $bm$, label $s$, and number equal
          to its position in the list of all buttons (starting with $0$).}*/


int  button(string s, int n, void (*F)(int), const char* hlp=0);
/*{\Mopl  adds a button with label $s$, number $n$ and action 
          function $F$ to $W$. Function $F$ is called with actual
          parameter $n$  whenever the button is pressed. }*/

int  button(int w, int h, char* bm, string s, int n, void (*F)(int), 
                                                     const char* hlp=0);
/*{\Mopl  adds a button with bitmap $bm$, label $s$, number $n$ and action 
          function $F$ to $W$. Function $F$ is called with actual
          parameter $n$  whenever the button is pressed. }*/

int  button(string s, void (*F)(int), const char* hlp=0);
/*{\Mopl  adds a button with label $s$, number equal to its rank and action 
          function $F$ to $W$. Function $F$ is called with the value of the
          button as argument whenever the button is pressed. }*/

int  button(int w, int h, char* bm, string s, void (*F)(int),const char* hlp=0);
/*{\Mopl  adds a button with bitmap $bm$, label $s$, number equal to its rank 
          and action function $F$ to $W$. Function $F$ is called with the 
          value of the button as argument whenever the button is pressed. }*/

int  button(string s, int n, window& M, const char* hlp=0);
/*{\Mopl  adds a button with label $s$, number $n$  and attached sub-window
          (menu) $M$ to $W$. Window $M$ is opened whenever the button is
          pressed. }*/

int  button(int w, int h, char* bm, string s, int n, window& M,
                                                     const char* hlp=0);
/*{\Mopl  adds a button with bitmap $bm$, label $s$, number $n$  and attached 
          sub-window (menu) $M$ to $W$. Window $M$ is opened whenever the 
          button is pressed. }*/


int  button(string s, window& M, const char* hlp=0);
/*{\Mopl  adds a button with label $s$ and attached sub-window $M$ to $W$. 
          The number returned by $read\_mouse$ is the number of the
          button selected in sub-window $M$. }*/

int  button(int w, int h, char* bm, string s, window& M, const char* hlp=0);
/*{\Mopl  adds a button with bitmap $bm$, label $s$ and attached sub-window 
          $M$ to $W$. The number returned by $read\_mouse$ is the number of 
          the button selected in sub-window $M$. }*/


int menu_button(string s, int n, window& M, const char* hlp=0);
/*{\Mopl  adds a {\em menu} button with label $s$, number $n$  and 
          attached sub-window (menu) $M$ to $W$. }*/


int menu_button(string s, window& M, const char* hlp=0);
/*{\Mopl  adds a {\em menu} button with label $s$ and attached sub-window $M$ 
          to $W$. The number returned by $read\_mouse$ is the number of the
          button selected in sub-window $M$. }*/




window* get_window(string s);
/*{\Mopl     looks for a button with label $s$ in window $W$.
             Returns attached subwindow on success and NULL otherwise.}*/

window* set_window(string s, window* M);
/*{\Mopl     Associates subwindow (menu) $M$ with button with label $s$.
	     Returns previously attached subwindow on success 
	     and NULL otherwise.}*/

static window* get_call_window() 
{ return (window*)BASE_WINDOW::call_window; } 

};


 

/*{\Mexample
Example programs can be found on LEDA/prog/window and LEDA/prog/demo. 
}*/


/*{\Moptions usesubscripts=no}*/


#include <LEDA/panel.h>
#include <LEDA/menu.h>


#if LEDA_ROOT_INCL_ID == 350136
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif
