/*******************************************************************************
+
+  LEDA 3.5
+
+  rat_line.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/

#ifndef LEDA_RAT_LINE_H
#define LEDA_RAT_LINE_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350109
#include <LEDA/REDEFINE_NAMES.h>
#endif


#include <LEDA/rat_point.h>
#include <LEDA/rat_segment.h>
#include <LEDA/rat_ray.h>
#include <LEDA/line.h>

//------------------------------------------------------------------------------
// straight rat_lines
//------------------------------------------------------------------------------


class rat_line_rep : public handle_rep {

friend class rat_line;
friend class rat_circle;

  rat_segment  seg; 

public:
   
  rat_line_rep() {}

  rat_line_rep(const rat_segment& s) : seg(s)
  { if (s.is_trivial())
      error_handler(1,"rat_line: cannot construct line from trivial segment");
    }

 ~rat_line_rep() {}

friend inline int cmp_slopes(const rat_line&l1, const rat_line& l2);
friend inline int orientation(const rat_line& l, const rat_point& p);

};
   
/*{\Manpage {rat_line} {} {Straight Rational Lines}}*/

class rat_line   : public handle_base 
{

friend class circle;

/*{\Mdefinition
An instance $l$ of the data type $rat\_line$ is a directed straight line
in the two-dimensional plane. 
}*/

rat_line_rep* ptr() const { return (rat_line_rep*)PTR; }

public:

/*{\Mcreation l }*/

 rat_line(const rat_point& p, const rat_point& q);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
line passing through points $p$ and $q$ directed form $p$ to $q$.\\
\precond $p \not= q$.}*/


 rat_line(const rat_segment& s);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
line supporting segment $s$.\\
\precond $s$ is nontrivial.}*/


rat_line(const rat_point& p, const rat_vector& v);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
line passing through points $p$ and $p+v$.\\
\precond $v$ is a nonzero vector.}*/



 rat_line(const rat_ray& r);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
line supporting ray $r$.}*/



 rat_line();
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|.}*/


 rat_line(const rat_line& l) : handle_base(l) {};
 rat_line& operator=(const rat_line& l) 
 { handle_base::operator=(l); return *this; }

~rat_line() {}



/*{\Moperations 2 4.5 }*/

line to_line() const;
/*{\Mop  returns a floating point approximation of |\Mvar|.}*/


rat_point   point1() const { return ptr()->seg.source(); }
/*{\Mop     returns a point on |\Mvar|.}*/

rat_point   point2() const { return ptr()->seg.target(); }
/*{\Mop     returns a second point on |\Mvar|.}*/

rat_segment seg()   const  { return ptr()->seg; }
/*{\Mop     returns a segment on |\Mvar|.}*/

bool is_vertical() const    { return ptr()->seg.is_vertical();  }
/*{\Mopl    decides whether |\Mvar| is vertical. }*/

bool is_horizontal() const  { return ptr()->seg.is_horizontal();}
/*{\Mopl    decides whether |\Mvar| is horizontal. }*/


bool intersection(const rat_line& g, rat_point& inter) const;
/*{\Mopl    returns true if $l$ and $g$ intersect. In case of intersection a common point is return in |inter|.}*/

bool intersection(const rat_segment& s, rat_point& inter) const;
/*{\Mopl    returns true if $l$ and $s$ intersect. In case of intersection a common point is return in |inter|.}*/

rat_line translate(const rational& dx, const rational& dy) const
{ return ptr()->seg.translate(dx,dy); }
/*{\Mopl    returns |\Mvar| translated by vector $(dx,dy)$.}*/


rat_line translate(integer dx, integer dy, integer dw) const
{ return ptr()->seg.translate(dx,dy,dw); }
/*{\Mopl    returns |\Mvar| translated by vector $(dx/dw,dy/dw)$.}*/

rat_line translate(const rat_vector& v)  const 
{ return ptr()->seg.translate(v); }
/*{\Mopl   returns |\Mvar| translated by vector $v$.\\
           \precond $v$.dim() = 2.}*/ 

rat_line  operator+(const rat_vector& v) const { return translate(v); }
/*{\Mbinop   returns |\Mvar| translated by vector $v$.}*/

rat_line  operator-(const rat_vector& v) const { return translate(-v); }
/*{\Mbinop   returns |\Mvar| translated by vector $-v$.}*/



rat_line rotate90(const rat_point& q) const
{ return ptr()->seg.rotate90(q); }
/*{\Mopl    returns |\Mvar| rotated about $q$ by an angle of $90$ degrees.}*/


rat_line reflect(const rat_point& p, const rat_point& q) const
{ return ptr()->seg.reflect(p,q); }
/*{\Mop     returns |\Mvar| reflected  across the straight line passing
            through $p$ and $q$.}*/

rational sqr_dist(const rat_point& q) const;
/*{\Mop     returns the square of the distance between |\Mvar| and |q|.}*/


rat_segment perpendicular(const rat_point& p) const;
/*{\Mop    returns the segment perpendicular to |\Mvar| with source $p$.
           and target on |\Mvar|.}*/

inline int side_of(const rat_point& p) { return orientation(ptr()->seg,p); }
/*{\Mop     computes orientation($a$, $b$, $p$), where $a \not= b$
and $a$ and $b$ appear in this order on line $l$.}*/


bool contains(const rat_point&) const;
bool contains(const rat_segment&) const;


bool operator==(const rat_line& g) const 
{ return ( contains(g.ptr()->seg) && 
          ( point2() - point1() )*( g.point2() - g.point1() ) > 0 ); }
/*{\Mbinop returns true if the |l| and |g| are equal as oriented lines. }*/

bool operator!=(const rat_line& g) const { return !operator==(g); }

friend bool equal_as_sets(const rat_line& l, const rat_line& g) 
{ return l.contains(g.seg()); }
/*{\Mfunc returns true if the |l| and |g| are equal as unoriented lines. }*/


friend ostream& operator<<(ostream& out, const rat_line& l);
friend istream& operator>>(istream& in, rat_line& l);  



/*{\Mtext
{\bf Non-Member Functions}
\smallskip
}*/

friend int orientation(const rat_line& l, const rat_point& p)
{ return orientation(l.ptr()->seg,p); }
/*{\Mfunc      computes orientation($a$, $b$, $p$), where $a \not= b$
and $a$ and $b$ appear in this order on line $l$.}*/

friend int cmp_slopes(const rat_line& l1, const rat_line& l2)
{ return cmp_slopes(l1.ptr()->seg,l2.ptr()->seg); }
/*{\Mfunc      returns compare(slope($l_1$), slope($l_2$)).}*/


};


inline bool parallel(const rat_line& l1, const rat_line& l2)
{ return cmp_slopes(l1,l2) == 0; }


extern rat_line p_bisector(const rat_point& p, const rat_point& q);
/*{\Mfunc  returns the perpendicular bisector of |p| and |q|. The bisector has |p| on its left.\\
\precond $p \not= q$.
}*/


inline char* leda_tname(const rat_line*) { return "rat_line"; }

#if LEDA_ROOT_INCL_ID == 350109
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif



