/*******************************************************************************
+
+  LEDA 3.5
+
+  _leda.c
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/
#include <LEDA/basic.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <fstream.h>


#if defined(unix)

#include <unistd.h>
#include <sys/time.h>
#include <sys/times.h>

#if defined(__svr4__)

#include <sys/systeminfo.h>
static void get_host_info(char* name, int sz, unsigned long& hostid)
{ sysinfo(SI_HOSTNAME,name,sz);
  char buf[32];
  sysinfo(SI_HW_SERIAL,buf,32);
  hostid = strtoul(buf,0,0);
}

#else

static void get_host_info(char* name, int, unsigned long& hostid)
{ strcpy(name,"????");
  hostid = 0;
 }

/*
extern "C" long gethostid();
extern "C" long gethostname(char*, int);
static void get_host_info(char* name, int sz, unsigned long& hostid)
{ gethostname(name,sz);
  hostid = (unsigned long)gethostid();
}
*/

#endif

#elif defined(__win_32__)

#if defined(bool)
#undef bool
#endif

#include <windows.h>

static void get_host_info(char* name, int sz, unsigned long& hostid)
{ char volname[64];
  char filesysname[64];
  unsigned long serialno;
  unsigned long maxcomp;
  unsigned long filesysflag;

  GetVolumeInformation("c:\\",volname,64,&serialno,&maxcomp,&filesysflag, 
                                                            filesysname,64);
  hostid = serialno;
  strcpy(name,"????");
}

#else

static void get_host_info(char* name, int sz, unsigned long& hostid)
{ strcpy(name,"????");
  hostid = 0;
 }

#endif




void LEDA::write_log(const char* s)  
{ char fname[256];
  sprintf(fname,"%s/LOGFILE",rootdir_name);
  ofstream log(fname,ios::nocreate|ios::app);
  if (log.good()) log << "\t" << s << endl;
 }


void LEDA::user_log()  
{ 
  char fname[256];
  sprintf(fname,"%s/LOGFILE",rootdir_name);

  ofstream log(fname,ios::nocreate|ios::app);

  if (!log.good()) return; 

  time_t clock; 
  time(&clock);

  char* usr_id = 0;

#if defined(unix)
  usr_id = cuserid(0);
#endif

  if (usr_id == 0) usr_id = "unknown";

  char host_name[64];
  unsigned long host_id;
  get_host_info(host_name,64,host_id);

  log << string("%8s@%-12s ", usr_id,host_name);
  log << string("%s (%s %s)  ",version_string,compiler_string,compiler_date);
  log << ctime(&clock);
}


#if defined(CHECK_LICENSE)
#include "/LEDA/LICENSE/compute_lic.c"
#else
static void compute_lic(unsigned long, char* buf) { buf[0] = 0; }
#endif


void LEDA::check_license()  
{ 
  char licfile_name[128];
  sprintf(licfile_name,"%s/LICENSE_FILE",rootdir_name);

  ifstream lic(licfile_name);

  unsigned long host_id = 0;
  char host_name[64];
  get_host_info(host_name,64,host_id);

  char lic_id[64];
  compute_lic(host_id,lic_id);

  if (lic_id[0] == 0) // no checking
     return;

  int lic_ok = 0;

  if (!lic_ok && !lic.good()) {
    cerr << endl;
    cerr << "ERROR: Cannot open license file " << licfile_name << endl;
    if (getenv("LEDAROOT")) 
       cerr << "       (check path in LEDAROOT environment variable)" << endl;
    cerr << endl;
  }

  char x[64];

  while (lic >> x)
   if (strcmp(x,lic_id) == 0) 
   { lic_ok = 1;
     break;
    }


  if (!lic_ok)
  { cerr << endl;
    cerr << "ERROR: No valid LEDA license found for host \""<< host_name<< "\"";
    cerr << hex << " ( id = " << host_id << " )" << endl;
    cerr << endl;
    cerr << "Please contact LEDA Software GmbH to obtain a license for" << endl;
    cerr << "this host and for more information on the license terms." << endl;
    cerr << endl;
    cerr << "\t LEDA Software GmbH" << endl;
    cerr << "\t Postfach 151101" << endl;
    cerr << "\t 66041 Saarbruecken" << endl;
    cerr << "\t Germany" << endl;
    cerr << "\t email:leda@mpi-sb.mpg.de" << endl;
    cerr << "\t fax: +49 681 842502" << endl;
    cerr << endl;
    exit(1);
   }
}



void LEDA::read_options()  
{ 
  //char* rdn = getenv("LEDAROOT");
  //if (rdn) rootdir_name = rdn;

  //option_list = getenv("LEDAOPTS"); 
  option_list = getenv("LEDA_INIT"); 

  char* s1 = getenv("LEDA_NODE_DATA");
  char* s2 = getenv("LEDA_EDGE_DATA");

  if (s1) node_data_slots = atoi(s1);
  if (s2) edge_data_slots = atoi(s2);

  if (s1 || s2)
  { fprintf(stderr,"\n");
    fprintf(stderr,"node_data_slots = %d\n",node_data_slots);
    fprintf(stderr,"edge_data_slots = %d\n",edge_data_slots);
    fprintf(stderr,"\n");
   }

}



//------------------------------------------------------------------------------
// LEDA::used_time
//------------------------------------------------------------------------------

float LEDA::used_time()
{ 
#if defined(unix)
#if !defined(HZ)
#define HZ 60
#endif
  tms x;
  times(&x);
  return  float(x.tms_utime)/HZ;
#else
  return  float(clock())/CLOCKS_PER_SEC;
#endif
}


float LEDA::used_time(float& T)
{ float t = T;
  T = LEDA::used_time();
  return  T-t;
}


//------------------------------------------------------------------------------
// LEDA::sleep
//------------------------------------------------------------------------------


#if defined(unix)

#if defined(_AIX)
#include<strings.h>
#include<sys/select.h>
#endif

/*
#if defined(sun) && !defined(__svr4__)
extern "C" int select(int,void*,void*,void*,void*);
#endif
*/

void LEDA::sleep(float sec) 
{ int usec = int(1000000*sec);
  timeval delay;
  delay.tv_sec  = usec / 1000000;
  delay.tv_usec = usec % 1000000;
  select(0, NULL, NULL, NULL, &delay); 
}

#else

void LEDA::sleep(float) {} 

#endif

