/*******************************************************************************
+
+  LEDA 3.5
+
+  rat_ray.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/

#ifndef LEDA_RAT_RAY_H
#define LEDA_RAT_RAY_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350112
#include <LEDA/REDEFINE_NAMES.h>
#endif


#include <LEDA/rat_point.h>
#include <LEDA/rat_segment.h>
#include <LEDA/ray.h>

//------------------------------------------------------------------------------
// rat_rays
//------------------------------------------------------------------------------


class rat_ray_rep : public handle_rep {

friend class rat_ray;
friend class rat_line;
friend class circle;

  rat_segment  seg; 

public:
   
  rat_ray_rep() {}
  rat_ray_rep(const rat_segment& s) : seg(s) {}

 ~rat_ray_rep() {}

friend inline int cmp_slopes(const rat_ray&, const rat_ray&);
friend inline int orientation(const rat_ray&, const rat_point&);

};
   
/*{\Manpage {rat_ray} {} {Rational Rays }}*/

class rat_ray   : public handle_base 
{

friend class rat_line;
friend class circle;

/*{\Mdefinition
An instance $r$ of the data type $rat\_ray$ is a directed straight ray defined
by two points with rational coordinates in the two-dimensional plane.
}*/

rat_ray_rep* ptr() const { return (rat_ray_rep*)PTR; }

public:

/*{\Mcreation r }*/

 rat_ray(const rat_point& p, const rat_point& q);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
ray starting at point $p$ and passing through point $q$. \\
\precond $p \not= q$.}*/


 rat_ray(const rat_segment& s);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
$(rat\_ray(s.source(),s.target())$.\\
\precond $s$ is nontrivial. }*/

 rat_ray();
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|.}*/


 rat_ray(const rat_ray& r) : handle_base(r) {};
 rat_ray& operator=(const rat_ray& r) { handle_base::operator=(r); return *this; }
~rat_ray() {}



/*{\Moperations 2 4.5 }*/

ray to_ray() const;
/*{\Mop     returns a floating point approximation of |\Mvar|. }*/


rat_point  source() const   { return ptr()->seg.source(); }
/*{\Mop     returns the source of |\Mvar|.}*/

rat_point  point1() const   { return ptr()->seg.source(); }
/*{\Mop     returns the source of |\Mvar|.}*/

rat_point  point2() const   { return ptr()->seg.target(); }
/*{\Mop     returns a point on |\Mvar| different from |\Mvar.source()|.}*/


bool is_vertical() const    { return ptr()->seg.is_vertical();  }
/*{\Mop     returns true iff |\Mvar| is vertical.}*/

bool is_horizontal() const  { return ptr()->seg.is_horizontal();}
/*{\Mop     returns true iff |\Mvar| is horizontal.}*/


rat_segment seg()  const     { return ptr()->seg; }


bool intersection(const rat_ray& s, rat_point& inter) const;
/*{\Mopl    returns true if |\Mvar| and $s$ intersect. If so, a point of intersection is returned in |inter|. }*/

bool intersection(const rat_segment& s, rat_point& inter) const;
/*{\Mopl    returns true if |\Mvar| and $s$ intersect. If so, a point of intersection is returned in |inter|. }*/


rat_ray translate(const rational& dx, const rational& dy) const
{ return ptr()->seg.translate(dx,dy); }
/*{\Mopl    returns |\Mvar| translated by vector $(dx,dy)$.}*/


rat_ray translate(integer dx, integer dy, integer dw) const
{ return ptr()->seg.translate(dx,dy,dw); }
/*{\Mopl    returns |\Mvar| translated by vector $(dx/dw,dy/dw)$.}*/

rat_ray  translate(const rat_vector& v)  const 
{ return ptr()->seg.translate(v); }
/*{\Mop     returns $r+v$, i.e., $r$ translated by vector $v$.\\
            \precond $v$.dim() = 2.}*/ 

rat_ray  operator+(const rat_vector& v) const { return translate(v); }
/*{\Mbinop   returns |\Mvar| translated by vector $v$.}*/

rat_ray  operator-(const rat_vector& v) const { return translate(-v); }
/*{\Mbinop   returns |\Mvar| translated by vector $-v$.}*/


rat_ray  rotate90(const rat_point& q) const
{ return ptr()->seg.rotate90(q); }
/*{\Mopl    returns |\Mvar| rotated about $q$ by an angle of $90$ degrees.}*/


rat_ray  reflect(const rat_point& p, const rat_point& q) const
{ return ptr()->seg.reflect(p,q); }
/*{\Mop     returns |\Mvar| reflected  across the straight line passing
            through $p$ and $q$.\\
\precond $p \not= q$.}*/

rat_ray  reflect(const rat_point& p)
{ return ptr()->seg.reflect(p); }
/*{\Mop     returns |\Mvar| reflected  across point $p$. }*/


bool contains(const rat_point& p) const;
/*{\Mopl    decides whether |\Mvar| contains $p$. }*/

bool contains(const rat_segment& s) const;
/*{\Mopl    decides whether |\Mvar| contains $s$. }*/


bool operator==(const rat_ray& g) const;
bool operator!=(const rat_ray& g) const { return !operator==(g); }


/*{\Mtext
{\bf Non-Member Functions}
\smallskip
}*/


friend int orientation(const rat_ray& r, const rat_point& p);
/*{\Mfunc      computes orientation($a$, $b$, $p$), where $a \not= b$
and $a$ and $b$ appear in this order on ray $r$. }*/

friend int cmp_slopes(const rat_ray& r1, const rat_ray& r2);
/*{\Mfunc      returns compare(slope($r_1$), slope($r_2$)).}*/


friend ostream& operator<<(ostream& out, const rat_ray& r);
friend istream& operator>>(istream& in, rat_ray& r);  

};

inline  int orientation(const rat_ray& r, const rat_point& p)
{ return orientation(r.ptr()->seg,p); }

inline  int cmp_slopes(const rat_ray& r1, const rat_ray& r2)
{ return cmp_slopes(r1.ptr()->seg,r2.ptr()->seg); }

inline bool parallel(const rat_ray& r1, const rat_ray& r2)
{ return cmp_slopes(r1,r2) == 0; }



inline char* leda_tname(const rat_ray*) { return "rat_ray"; }

#if LEDA_ROOT_INCL_ID == 350112
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif



