/*******************************************************************************
+
+  LEDA 3.5
+
+  face_array.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/
#ifndef LEDA_FACE_ARRAY_H
#define LEDA_FACE_ARRAY_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350042
#include <LEDA/REDEFINE_NAMES.h>
#endif


//------------------------------------------------------------------------------
// face arrays
//
// S. Naeher (1996)
//------------------------------------------------------------------------------

#include <LEDA/graph.h>


/*{\Manpage {face_array} {E} {Face Arrays}}*/

template <class E>
class face_array : public graph_map
{

/*{\Mdefinition
An instance $A$ of the parameterized data type $face\_array\<E\>$
is a partial mapping from the face set of a graph $G$ to the set of 
variables of type $E$, called the element type of the array. The domain 
$I$ of $A$ is called the index set of $A$ and $A(v)$ is called the element 
at position $f$. $A$ is said to be valid for all faces in $I$.}*/

E def;

void clear_entry(GenPtr& x) const { LEDA_CLEAR(E,x); }
void copy_entry(GenPtr& x)  const { LEDA_COPY(E,x);  }
void init_entry(GenPtr& x)  const { x = leda_copy(def); }
  
public:

const char* elem_type_name() const { return LEDA_TYPE_NAME(E); }


/*{\Mcreation A }*/

face_array() : graph_map(nil,0,2) {}
/*{\Mcreate creates an instance $A$ of type $face\_array\<E\>$ with empty
            index set.}*/

face_array(const graph& G) : graph_map(&G,G.max_face_index()+1,2) { init_table(); }
/*{\Mcreate creates an instance $A$ of type $face\_array\<E\>$ and initializes
            the index set of $A$ to the current face set of graph $G$. }*/


face_array(const graph& G, E x) : graph_map(&G,G.max_face_index()+1,2) { def = x; init_table(); }
/*{\Mcreate creates an instance $A$ of type $face\_array\<E\>$, sets the
            index set of $A$ to the current face set of graph $G$ and 
            initializes $A(f)$ with $x$ for all faces $f$ of $G$. }*/


face_array(const graph& G, int n, E x) : graph_map(&G,n,2) { def = x; init_table(); }
/*{\Mcreate creates an instance $A$ of type $face\_array\<E\>$ valid for
            up to $n$ faces of graph $G$ and initializes $A(f)$ with $x$ 
            for all faces $f$ of $G$.\\
	    \precond $n \ge \Labs{V}$. $A$ is also
            valid for the next $n - \Labs{V}$ faces added to $G$.}*/


face_array(const face_array<E>& A) : graph_map(A)   {}
face_array<E>& operator=(const face_array<E>& A) 
{ graph_map::operator=(A);  
  return *this;
 }
~face_array() { clear_table(); }



/*{\Moperations 1.3 4.7 }*/

const E& operator[](face f) const
{ return LEDA_CONST_ACCESS(E,graph_map::array_read(f)); }

E& operator[](face f) {  return LEDA_ACCESS(E,graph_map::array_access(f)); }
/*{\Marrop    returns the variable $A(f)$.\\
	      \precond $A$ must be valid for $f$.}*/

E& entry(int i)     { return LEDA_ACCESS(E,graph_map::access(i));}
E  inf(int i) const { return LEDA_ACCESS(E,graph_map::read(i));}
E& entry(face f)    { return LEDA_ACCESS(E,graph_map::array_access(f)); }
E  inf(face f) const{ return LEDA_ACCESS(E,graph_map::array_read(f)); }

void init()  { graph_map::init(nil,1,2); }

void init(const graph& G)      
{ graph_map::init(&G,G.max_face_index()+1,2); 
  init_table();
 }
/*{\Mop       sets the index set $I$ of $A$ to the face  
	      set of $G$, i.e., makes $A$ valid for all faces 
              of $G$.}*/

void init(const graph& G, E x) 
{ graph_map::init(&G,G.max_face_index()+1,2); 
  def = x;
  init_table();
}
/*{\Mop       makes $A$ valid for all faces of $G$
	      and sets $A(f) = x$ for all faces $f$ of $G$.}*/

void init(const graph& G, int n, E x) 
{ graph_map::init(&G,n,2); 
  def = x;
  init_table();
}
/*{\Mopl      makes $A$ valid for at most $n$ faces  of $G$ 
              and sets $A(f) = x$ for all faces $f$  of $G$.
              \precond {$n \ge \Labs{V}$}. $A$ is also valid for 
              the next  $n - \Labs{V}$ faces added to $G$. }*/


/*{\Mimplementation
Node arrays for a graph $G$ are implemented by \CC vectors and an
internal numbering of the faces and edges of $G$. The access operation
takes constant time, $init$ takes time $O(n)$, where $n$ is the number of
faces in $G$. The space requirement is $O(n)$.
    
{\bf Remark}: A face array is only valid for a bounded number of
faces of $G$. This number is either the number
of faces of $G$ at the moment of creation of the array or it is explicitely 
set by the user. Dynamic face arrays can be realized by face 
maps (cf. section \ref{Node Maps}). }*/

};



#if LEDA_ROOT_INCL_ID == 350042
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif
