/* Possible Changes :
   -> temp variables in functions?
   -> redundant distrib/distribution syntax !
   -> UI ability to change label - element, component, unit, etc
   -> Function overloading?
   -> When it is 'key mass', the origin should default to the current component,
      not root
   -> add keyword parent, as a reference, eg key mass of parent
*/

default mass = real 0 kg;
default power = real 0 watt;
default cost = real 0 megadollar;
default efficiency = real 1;
default temp = 0 K;
default duration = 0 sec;
default speed = 0 m/s;
default freq = 1 hz;
default theta = 0 radian;
default t = 0 radian;

/* #include "general.c" */
Pi = 3.14159265358979323846 radian;
G  = 9.81; /* m/s^2 */
triple y_axis = triple( 0, 1, 0 );

Component LunarIce {
  
  Component Mission /* Baseline */
    {
      sortieDuration = 20 hour; /* hours */
      duration = 30 day;    /* hours */
      distrib speed = distribution(min=0.20, max=0.4, avg=0.25) m/s;
      distance = duration*get_min(Speed)*2;
      Component Launcher
	{
	  FairingDia = 2.3 meter; /* 2.3 meter payload fairing */
	  InjectionError = 1.5 degree; /* deg */
	}
    }
    
  component Earth 
    {
      /* this would normally be added by the robot design rather
	 than in the component itself. Do when we have ability
	 to reference & include components */
      type CommLink; 

      g = 9.81 m/s2;
      
      Component Atmosphere { CommAmplitude = -0.3; }
      Component Ionosphere { CommAmplitude = -0.1; }
      Component Rain {
	distrib CommAmplitude = distribution(max=0, min=-1, avg=-0.2); }
    }

  component Moon {
    RevPeriod = 28.45 day; /* in seconds */
    distrib EarthDist = Distrib(avg=384.0E6, max=406.697E6, min=356.410E6) m;
    
    cycle = 27.322 hour; /* approx max "day" hrs - sun cycle, see p57 LSB */
    g = 9.81 m/s2 / 6;
  }
  
  component Space {
    Temp = 3 kelvin; /* degree Kelvin */
  }

  Component Spacecraft {
    shape body_shape = shape "booster_shape.txt" * ( 4.8, 1, 4.8 );

    transparent mass = sum( key mass of Spacecraft ) : 
      1500 kg, clock coriolis, main_clock;
    triple position = triple( 0, 12.5, 0 ) kilometer;
    triple velocity = -triple( 0, 50, 0 ) m/s;
    altitude = dot( y_axis, spacecraft.position ) - 1 m, clock coriolis;

    weight = ( mass - srm.mass ) * moon.g, clock coriolis;

    /* gravity */
    link spacecraft : force spacecraft.weight * -y_axis, triple ( 0, 0, 0 );

    Component Thrust_Control/*, clock coriolis*/ {
      /* target velocity for spacecraft */
      real vtarget = if( spacecraft.altitude > 2 kilometer, -150 m/s,
			 spacecraft.altitude > 1 kilometer,
			 -( ( spacecraft.altitude - 1 kilometer ) * 0.1 / 1 s
			    + 50 m/s ), spacecraft.altitude > 5 m,
			 -( spacecraft.altitude - 5 m ) / 20 s - .25 m/s,
			 -0.25 m/s ), clock coriolis;
      
      /* if dropping too quickly then delta_v negative */
      real delta_v = dot( spacecraft.velocity, y_axis ) - vtarget, 
	clock coriolis;
      
      /* dv1 >= 0 */
      dv1 = if( delta_v >= 0 m/s, 0 m/s, -delta_v * 1.2 ), clock coriolis;
      /* error < 1 */
      error = if (delta_v >= 0 m/s, 0 m/s, error * .7 + dv1 * .1) : 0 m/s, 
        clock coriolis;
      
      /* desired_thrust >= 0? */
      thrust = ( dv1 * 1.2 s/m + error * 1.4 s/m ) * spacecraft.weight, 
	clock coriolis;
            
      transparent maxThrust = 3300 newton, clock coriolis;

      /* real thrust values in newtons */
      w_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      e_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      nw_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      ne_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      sw_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      se_thrust = max( min( maxThrust, thrust / 6 ), 0 newton ), clock coriolis;
      total_thrust = sum( key "*_thrust" of Thrust_Control ) : 0 newton, 
	clock coriolis;
      
      /* thrusters -- two sets of three.  center thruster lies in the xy plane,
	 at a 10 degree angle from the negative y axis.the side two are 10 
	 degrees from the negative y axis in the xy plane and 15 
	 degrees from the negative y axis in the yz plane */
      x_thruster_angle = 10 degree;
      z_thruster_angle = 15 degree;
      
      x_tan = tan( x_thruster_angle );
      z_tan = tan( z_thruster_angle );
      
      y_side = 1 / sqrt( x_tan * x_tan + 1 + z_tan * z_tan );
      x_side = x_tan * y_side;
      z_side = z_tan * y_side;
      
      y_main = 1 / sqrt( x_tan * x_tan + 1 );
      x_main = x_tan * y_main;
      
      /* thrusters are labeled east_wast if in the pos/neg x direction, or 
	 north_south for the pos/neg z direction */
      triple thrust_west = gvec( x_main, y_main, 0 ), clock coriolis;
      triple thrust_east = gvec(-x_main, y_main, 0 ), clock coriolis;
      
      triple thrust_northwest = gvec( x_side, y_side, -z_side ), clock coriolis;
      triple thrust_northeast = gvec(-x_side, y_side, -z_side ), clock coriolis;
      triple thrust_southwest = gvec( x_side, y_side, z_side ), clock coriolis;
      triple thrust_southeast = gvec(-x_side, y_side, z_side ), clock coriolis;
      
      triple wthrust_bp = triple( -2.4, -.30, 0 ), clock coriolis; /* -.50, -.30 scaled */
      triple ethrust_bp = triple( 2.4, -.30, 0 ), clock coriolis;  /* .50, -.30 scaled */
      
      /* forces due to thrusters */
      link spacecraft : force w_thrust * thrust_west, wthrust_bp;
      link spacecraft : force e_thrust * thrust_east, ethrust_bp;
      link spacecraft : force nw_thrust * thrust_northwest, wthrust_bp;
      link spacecraft : force ne_thrust * thrust_northeast, ethrust_bp;
      link spacecraft : force sw_thrust * thrust_southwest, wthrust_bp;
      link spacecraft : force se_thrust * thrust_southeast, ethrust_bp;
    }
    
    Component Icebreaker {
	
      transparent mass = sum(key mass of icebreaker) : 392 kg;
      transparent power = sum(key power of icebreaker) : 400 watt; /* W */

      Component Locomotion {
	Component Spec 
	  {
	    slopemax = 22 degree; 
	    obstacleMax = 0.25 m;   /* meters  */
	    turnmin = 6.0 m; /* min outer turning radius */
	  }

	Component Soil 
	  { /* Lunar Soil LSS model, LSB, P526,8,.. */
	    /* Pressure sinkage characterstics under wheel load */
	    n = 1; /* regolith sinkage exponent */
	    Kc = 1382.32 newton/m2; /* cohesive modulus N/m^(n+1) */
	    Kphi = 816331.27 newton/m3; /* frictional modulus N/m^(n+2) */
	    
	    c = 172.79 Newton/m2; /* cohesion of regolith N/m^2 */
	    K = 0.01778 m; /* coeff of slip  meters */
	    phi = 35 degree; /* angle of internal friction/ repose 35 */
	    fr  = 0.1; /* coeffecient of rolling friction */
	    pressureMax = 6000 pascal; /* Pascals */
	  }

	CGz = 1.0 m; /* meters offset from bottom center */
	CGL = 0.1 m; /* length & width wise horizontal offsets */
	CGw = 0.1 m;
	
	L = fairingDia/sqrt(2) - 0.01 m; /* length meters */
	W = sqrt(fairingDia^2 - (L+0.01 m)^2) - 0.01 m; /* width meters */
	n = 4; /* n wheels */
	
	Contact(load) = sqrt( wheel.d^2/4 - ( wheel.d/2-sinkage(wgt=load) )^2 );
	contactArea(load) = Contact(load)*wheel.w;
	pressureMax = (old icebreaker.mass)*moon.G/
	  (2*contactArea(WheelLoadMax));
	slopeLmax = atan((L/2 - wheel.d/2 - CGL)/CGz);
	slopeWmax = atan((W/2 - CGw)/CGz);
	slopemax = max(slopeLmax, slopeWmax);

	weight = (old icebreaker.mass )*moon.G;
	load( theta = 0, wgt = locomotion.weight ) = wgt/n*cos( theta )*1.1; 
	/* 1.1 accounts for imbalance, cross slopes */
	wheelLoadMax = weight/2;
	  

	/* Total electric Driving power in W */

	P(theta=0 radian,V=mission.speed) = amplifier.P(theta,V)*n;

	/* Now calc electric power in for different conditions ??? */
	/* Drive sizing to include flat and slope conditions ??? */

	mass = 0.30/(1-0.30)*(old icebreaker.mass);
	power = P(theta=0,V=get_avg(mission.speed))*1.05 : 130 watt; /* 1.05 is margin */
	cost = 6;

	/* May be worth checking out flexible wheel for power/ size */
	Component Wheel{ /* rigid wheel model */
	  d = 0.6 m; /* m */
	  w = 0.3 m; /* m */
	  
	  /* for rigid wheel, sinkage equation */
	  sinkage(t,wgt=locomotion.weight) = ( meter )
	    ( (3 * load(t,wgt) / ( ( 3 - soil.n ) * ( soil.kc + W * soil.kphi )
				  * sqrt(D) ) )^( 2 / ( 2 * soil.n + 1 ) ) );
	  
	  /* Soil thrust */
	  H(t,wgt=locomotion.weight) = (soil.c*contactArea(load(t,wgt)) +
			   load(t,wgt)*tan(soil.phi));
	  

	  /* Compaction resistance */ /* LOKI fractional power of D */
/* 	  Rc(t,wgt) = ((3*load(t,wgt))^((2*soil.n+2)/(2*soil.n+1)) */
/* 		       /sqrt(D^((2*soil.n+2)/(2*soil.n+1)))) */
/* 	    / ((3-soil.n)^((2*soil.n+2)/(2*soil.n+1))*(soil.n+1) */
/* 	       *(soil.kc+W*soil.kphi)^(1/(2*soil.n+1))); */
	  Rc(t,wgt) = (3*load(t,wgt)/sqrt(D))^((2*soil.n+2)/(2*soil.n+1))
	    / ((3-soil.n)^((2*soil.n+2)/(2*soil.n+1))*(soil.n+1)
	       *(soil.kc+W*soil.kphi)^(1/(2*soil.n+1)));

	  /* Buldozing resistance */
	  alpha = 1.0;
	  Rb(t,wgt) = ( ( newton ) ( 0.5*alpha*W*sinkage(0,load(t,wgt))^2*
				     tan(pi/4 + soil.phi/2)^2 ) )
	    + 2*soil.c*W*sinkage(0,load(t,wgt))*
	    tan(pi/4+soil.phi/2);
	  
	  /* Rolling resistance */
	  Rr(t,wgt) = soil.fr*load(t,wgt);
	  
	  /* Gravitational resistance */
	  Rg(t,wgt=locomotion.weight) = wgt/n * sin(t);
	  
	  /* Acceleration resistance */
	  Ra(a=0) = (old icebreaker.mass)/n * a;
	  
	  /* Obstacle climbing resistance */
	  Ro(size=0) = 0;
	  
	  /* Total resistance excluding obstacle, steer, acceleration */
	  Rall(t=0 radian,wgt=locomotion.weight) = Rc(t,wgt)+Rb(t,wgt)+Rr(t,wgt)
	    +Rg(t,wgt);
	  
	  /* Drawbar Pull */
	  DP(t=0 radian,wgt=locomotion.weight) = H(t,wgt) - Rall(t,wgt);
	  
	  /* Driving Torque */
	  
	  defl = 0.0 m; /* deflection of wheel */
	  Tw(slope=0 radian,wgt=locomotion.weight) = Rall(slope,wgt)*(D/2-defl);
	  
	  Tmax = Tw(0,wheelLoadMax/load(wgt=1)); /* LOKI - NAN */ 
	  
	  NCycles = mission.distance/(Pi*D);

	  /* Wheel Design Constraints */
	  sinkage(t=0 radian,wgt=WheelLoadMax) <= 0.1*wheel.D;
	  constraint sinking = pressureMax <= soil.pressureMax;
	} /* Wheel */
			     
	Component Axle
	  {type wheel;
	   efficiency = 0.8;}
	Component Bearing{type wheel;
			  efficiency = 0.9;}
	Component SteerLink{}
	
						 
	Component Amplifier
	  {
	    /* electrical power input to amplifier */ /* Nm */
	    P(theta,V) = Wheel.Tw(theta)*
	      wheeldrive.omega(V)/efficiency/wheeldrive.Nu;
	    
	    /* At peak torque condition */
	    PTmax = wheel.Tmax*wheeldrive.omegapeak/efficiency/Wheeldrive.Nu;
	    
	    efficiency = 0.85;
	  }
	Component WheelDrive
	  {
	    inherited type wheel;
	    Component reduction1 
	      {
		efficiency = 0.7; /* 0.7 */
		ReductionRatio = 100; /* Reduction ratio */
	      }
	    Component reduction2
	      {
		efficiency = 0.7; /* 0.7 */
		ReductionRatio = 100; /* Reduction ratio */
	      }
	    safetyFactor = 1.5; /* for specing torque */
	    
	    /* Overall effeciency of wheel & drive train in delivering
	       power from motor output to terrain */
	    Nu = product(key efficiency of type wheel of locomotion);

	    /* Nominal drive train spec at output */
	    Torque = wheel.Tw; 
	    omega(V=mission.speed) = V/(wheel.D/2-wheel.defl);

	    /* Peak drive train spec at output */
	    TorquePeak = wheel.Tmax;
	    omegaPeak = get_min(mission.speed)/(wheel.D/2-wheel.defl);
	    
	    MotorNcycles = 10E6; /* real numbers needed */
	    MotorNCycles > wheel.Ncycles*2.0*ReductionRatio;
	  }

	Component SteerDrive
	  {
	    efficiency=0.3;
	  }

	sqrt(L^2 + W^2) <= fairingDia;
	2.2 * wheel.D <= L;
	slopemax >= spec.slopemax;

	/* the step/D ratio is really a function of (L+-CGL)/L, 
	   traction/weight & L/D, get something better for the hack */
	D >= obstacleMax/0.25; /* What is this factor really? */
      } /* locomotion */

      Component Structure{
	mass = 0.2*(old icebreaker.mass
		       /* except desc of locomotion ??? BUG??? */ 
	  - locomotion.mass) /* HACK ???.  need to except self */ ;
	cost = 2;
	Component Body{}
	Component Mast{ mass = 0.6*sum(key mass of type mastpayload) + 6 kg;}
	Component LanderSRMInterface {
	  mass = 0.005*((old icebreaker.mass)+lander.mass);
	  cost = 0.2;
	}
	Component LanderRoverinterface {mass = 0.005*(old icebreaker.mass);}
	Component SRMLauncherInterface {mass = 0.005*(old spacecraft.mass);}
      } /* Structure */
      
      Component PowerSystem {
	Component SolarArray { /* 4 mil cell, 3 mil coating */
	  W = 2.5 m; 
	  H = 1.25 m;
	  area = H * W; /* w x h */
	  mass = 0.75 kg/m2 * area * 2; /* 2 - structural overhead */
	  unitcost = 800 dollar/watt; /* 800 dollars/watt */
	  efficiency = 0.14; 
	  powerOut = area*( 1358 watt/m2 ) * efficiency; /* units LOKI */
	  cost = powerOut*unitcost + 0.5 dollar; /* .5 for DD&T */
	}
	Component Actuator{ /* Solar array 1 axis pointer */
	  mass = 5 kg;
	  distrib power = distribution(max=15,avg=5,min=0);
	  cost = 0.5;
	}
	Component BatteryPack{ /* HR190 - Yardney */
	  unitMass = 1 kg/whr /112; /* kg/Whr */
	  unitVolume = 0.256834 cc/Whr; /* cc/Whr */
	
	  energy = sortieDuration * old icebreaker.power;
	  mass = energy*unitMass;
	  volume = energy*unitVolume; /* cc */
	  cost = 2; /* includes engineering, DD&T, proto */
	}
	Component PMAD{ /* Power Management And Distribution */
	  cost = 1;
	  margin = 0.2; /* power margin for the overall system */
	  Component Charger{mass = 4 kg;}
	  Component Switch{mass = 1 kg;}
	  Component DCDC{
	    efficiency = 0.85;
	    mass = 2 kg;
	    dcpowerneed = old icebreaker.power - old locomotion.power
	      - old power;
	    /* actually, need to seperate dc-dc power from margin, num
	       correct though ?? */
	    power = dcpowerneed/efficiency * (1+margin) -  
	      dcpowerneed + locomotion.power*margin : 50 watt;
	  }
	  Component Harness{mass = 3 kg;}
	}/* PMAD */
      } /* PowerSystem */

      Component Thermal{
	cost = 2;
	Component EBox{}
	MLImass(area,layers) = area*layers*0.04 kg/m2; /* 1 mil (25 um - 1600 kg/m^3) kapton */ /* LOKI - units */
	Component MLIblanket{
	  area = 5 m2; 
	  layers = 10; 
	  mass = MLImass(area,layers);}
	Component MLIplumbing{
	  area = 1 m2; 
	  layers = 10;
	  mass = MLImass(area,layers);}
	Component HeatPipe{mass = 2 kg;}
	Component Radiator{mass = 3 kg;}
	Component Heaters{mass = 0.5 kg;}
      } /* Thermal */
      
      Component Computronics{
	Component Cage{mass = 4 kg; cost = 0.3;}
	
	/* once there ability to spawn or prototype boards,
	   spawn N boards, and tweak power on each to on/off ??? */
	Component Boards{
	  
	  mipsGTotal = 20; /* General purpose needed */
	  mipsOTotal = 140; /* Other mips, DSP or otherwise */
	  efficiency = 0.7;
	  int boardsActive = min(mipsGTotal/bmipsG/efficiencyG+1, 
				 mipsOTotal/bmipsDSP/efficiencyDSP+1);

	  int N = boardsActive+2;
	  mass = bmass*N; /* including heat sink, 5 of these */
	  power = bpower*boardsActive;
	  cost = 1.5e6 dollar + N*0.15e6 dollar;
	  Component Board {
	    bmass = 1 kg; /* including heat sink */
	    bpower = 15 watt;
	    bmipsG = 20;
	    efficiencyG = 0.5;
	    efficiencyDSP = 0.9;
	    bmipsDSP = 25*2;
	  }
	}
	Component ConnectorBus{
	  cost = 0.1;
	  mass = 0.02*200 kg; /* about 4kg for 200 */
	}
	Component DataRecorder{
	  cost = 0.8;
	  mass = 2 kg;
	  distrib power = distribution(max=15,avg=4,min=1);
	}
	Component Filters{ /* Analog filters & signal conditioners */
	  cost = 0.1;
	  mass = 1 kg;
	}
	Component SafetyCircuit{
	  cost = 0.2;
	  mass = 1 kg;
	}
      } /* Computronics */

      Component Comm
	{ /* If there is no satellite */
	  Component transciever
	    {
	      mass = 5 kg;
	      distrib power = distribution(max=40,avg=40,min=0) watt;
	    }
	  Component Antenna
	    { 
	      type mastpayload;
	      mass = 1 kg;
	    }
	  Component Cable
	    {
	      length = 5 m; /* meters */
	      mass = length*0.5 kg/m;
	    }
	  Component backup
	    {
	      mass = 3 kg;
	      distrib power = distribution(max=30,avg=0,min=0);
	    }
	} /* Comm(Earth) */

      Component Sensors{
	Component StarTracker{
	  mass = 0.6*2 kg;
	  power = 3.25*2 watt;
	  cost = 1.25;
	}
	Component IMU{
	  mass = 0.7 kg;
	  power = 5 watt;
	  cost = 0.25;
	}
	Component CameraPanosphere{
	  mass = 3 kg;
	  power =  15 watt; /* 15 W ??? */
	  cost = 2; 
	}
	Component SunTracker {}
	Component EarthTracker {}
	Component TerrainRadar {
	  driveFOV = 120 degree; /* what do i need to drive */
	  int Nfront = driveFOV/fov*2;
	  /* *2 for double coverage */
	  Nside = (180 degree-driveFOV)/30 degree; /* for single coverage on sides */
	  N = Nfront*2 + Nside*2; /* *2 for front & back */
	  
	  H = 0.8; /* meters */
	  Lookahead1 = H; /* Min lookahead */
	  LookAngle1 = atan(Lookahead1/H);
	  LookAngle2 = LookAngle1 + FOV;
	  Lookahead2 = tan(lookAngle2)*H;
	  
	  Xres = get_avg(mission.speed)/freq*4; /* X is along motion */
	  Yres = Xres;
	  
	  mass  = 0.4 kg * N;
	  power = 0.8 watt * (Nfront + Nside*2);
	  cost  = 2;

	  Component RadarUnit
	    {
	      freq = 20 hz; /* Hz */
	      FOV = 30 degree; /* 30 deg */
	    }
	}
	Component internalSensors{
	  mass = 2 kg;
	  power = 4 watt;
	  cost = 0.5;
	}
      } /* sensors */

      Component Software {
	Component OS {}
	Component Safeguarding 
	  {
	  }
	Component Messaging {}
      }
      
      Component SciencePackage {
	Component CryoDrill {
	  cost = 2;
	  mass = 4 kg;
	  distrib power = distribution(max=15,avg=0,min=0) watt;
	  Component IRFiber {}
	  Component IRTransciever {}
	}
	Component Microscope {}
	Component REGA {
	  cost = 0.5;
	  mass = 4 kg;
	  distrib power = distribution(max=20,avg=0,min=0) watt;
	}
	Component PanTilt {
	  cost = 1;
	  mass = 3 kg;
	  distrib power = distribution(max=10,avg=2,min=0) watt;
	}
	Component IRCamera {
	  cost = 0.2;
	  mass = 2 kg;
	  distrib power = distribution(max=5,avg=2,min=0) watt;
	}
	Component IRLaser {
	  cost = 0.2;
	  mass = 2 kg;
	  distrib power = distribution(max=10,avg=2,min=0) watt;
	}
	Component HiResCamera {
	  cost = 0.5;
	  mass = 1.5 kg;
	  distrib power = distribution(max=8,avg=5,min=0) watt;
	}
	Component Strobe {
	  cost = 0.5;
 	  mass = 1 kg;
	  distrib power = distribution(max=10,avg=4,min=0) watt;
	}
      } /* SciencePackage */
      
    } /* IceBreaker */

  Component Lander{
    transparent mass = sum( key mass of lander ) : 200 kg;
    cost = 15;
    Component Fuel
      {
	fuel_spent = fuel_spent + fuel_rate( old total_thrust ) * delta_time : 
	  0 kg, clock coriolis;

	fuel_rate( thrust ) = thrust / isp / 9.81 m/s2 : 0 kg/s, clock coriolis;

	Isp = 220 sec;
	dVmidcourse = 50 m/s; /* m/s, for a 1.5 deg max error, confirm */
	Mmidcourse = old spacecraft.mass - old dMmidcourse : 1500 kg;
	dMmidcourse = rocket.DeltaM(Mmidcourse, Isp, dVmidcourse) : 35 kg;
	
	// dVacs = ; /* m/s, attitude control during SRM burn */
	Macs = Mmidcourse - old dMacs;
	dMacs = Mmidcourse/1456*25 : 25 kg; /* kg */ /* LOKI -- needs ival */

	dVterminal = 200 m/s; /* m/s */
	Mterminal = Macs - old SRM.mass - old dMterminal;
	dMterminal = rocket.DeltaM(Mterminal, Isp, dVterminal) : 50 kg;

	Mhover = Mterminal + dMterminal;
	hangtime = 90 s; /* seconds of course */
	/* ratio of fuel mass/s to craft mass */
	dMhover = rocket.DeltaMhover(Mhover,Isp,hangtime,Moon.g) : 15 kg;
	/* kg, for hovering above surface */

	/* dMmargin = 0.0*mass : 25 kg;  kg, fuel margin */

	mass = sum (key "dm*" of fuel) - fuel_spent : 130 kg, 
	  clock coriolis, main_clock;
      }
    Component Propulsion {
	Component DE {
	    mass = 60 kg; /* Includes ACS & tubing as of now */
	  } /* Descent Engine */
	Component ACS {} /* Attitude Control System */
	Component BellyCamera {
	  mass = 2 kg;
	  cost = 0.25;
	}
	Component Radar {
	  mass = 3 kg;
	  cost = 0.5;
	}
	Component Software {}
	Component Disconnect {}
	Component Structure {}
	Component Electronics {}
      } /* propulsion */
  } /* Lander */
  
  Component SRM { /* Solid Rocket Motor */
    transparent mass = sum(key mass of SRM) : 974 kg;
    Cost = 250 megadollar/kg *old mass*2.2/1E6; /* in millions */ /* LOKI units */
    Component Fuel
      {
	dV = 2560 m/s;
	/* mass = rocket.deltaM(old spacecraft.Mass,
		      290 s, dV); */
	mass = old spacecraft.mass * 861 / 1500;
      }
    Component drySRM /* Thiokol Star 37FXP */
      {
	mass = 63.6 kg; 
      }
    Component LanderInterface 
      {
	Component Clamp { mass = 0.001 * ( old icebreaker.mass ); }
	Component Harness { mass = 2 kg; }
      }
  } /* SRM */
  } /* Spacecraft */

  Component DSN {cost = 1500 dollar/hour * duration;} 
  
  Component SystemEngineeringDDT {
    overhead = 2.0;
    Component Coreteam {
      personCost = 80000 dollar *Overhead;
      cost = personCost*work; /* 80k / engineer */
      work = 20*3;
    }
    Component facility {cost = 6; }
    Component IronBird {cost = 2; }
    Component prototype {cost = 2; }
    Component mobilityTestbed {cost = 0.6; }
    Component testing {
      Component equipment {cost = 1; }
      Component facility {cost = 0.5; }
      Component radiationFacility {cost = 0.2; }
      Component transportation {cost = 0.6; }
      Component burnInrisk {cost = 3; }
    } /* testing */
    
    Component Management {cost = 0.15*sum(key cost of lunarice 
			  except set( management.cost, contingency.cost ) );}
    Component Contingency {cost = 0.2*management.cost/0.15*1.15;}
    
    Component Software {
      unitcost = 65000 dollar * overhead;
      cost = unitcost*sum(key work); /* 60-70 + overhead */
      Component Simulation {work = 5*2;}
      Component CAD {work = 15*2;}
      Component Ground {work = 2*2;}
      Component Science {work = 2*2;}
      Component Test {work = 5*2;}
      Component Injection {work = 5*2;}
      Component Satellite {work = 2*2;}
      Component Lander {work = 5*2;}
      Component HousekeepingSafeguarding {work = 5*2;}
      Component Contingency {work = 5*2;}
    } /* software */
    
  } /* systemEngineeringDDT */
  
} /* lunarice */

/********************** Optional (sub) systems & components ****************/

Component Rocket {
  G = 9.81 m/s2; /* m/s^2 */
  DeltaM(Mf, Isp, dV) = Mf*(exp(dV/(Isp*earth.G))-1)/exp(dV/(Isp*earth.G));// kg

 /* Returns fuel needed given final Mass after burn, Isp & deltaV
    Do not forget to include rocket dry mass in your Mf, may need 
    iterations, so what does not?

    dV: Delta V 
    Vj: Jet velocity, Vj = Isp*G;
    Mf: Spacecraft mass 
    mu = exp(dV/Vj) 
    Propellant required (approx) = dV/Vj*mo 
    Propellent required (actual) = (mu - 1)/mu * mf 
    Isp: lbf/(lbmass/sec) [in Earth pounds force] 
    */

  DeltaMhover(Mf, Isp, hangtime, grav) = Mf/(1/(grav*hangtime/Isp/Earth.g)-1.0);
}

Component OrbiterStage {
  Component Margin {
    mass = 0.2*sum(key mass of OrbiterStage); /* same as power margin */
  }
  
  Component SRM {
    Component Fuel{mass = 33.52 kg;}
    Component rocket {
      mass = 7.5 kg;
      cost = 0.5;
    }
    Component interface {
      mass = 4 kg;
      cost = 0.1;
    }
  } /* SRM */
  Component Satellite {
    Component Solararray {
      mass = 2 kg;
      cost = 0.1;
    }
    Component Gyro {
      mass = 8 kg;
      power = 28;
      cost = 0.5;
    }
    Component computerBoard {
      mass = 2 kg;
      power = 10;
      cost = 0.2;
    }
    Component cameraElectronics {
      mass = 3 kg;
      distrib power = distribution(max=10,min=2);
      cost = 0.4;
    }
    Component CommMoon {
      mass = 3 kg;
      power = 20;
      cost = 0.5;
    }
    Component CommEarth {
      Component horn {
	mass = 1 kg;
      }
      Component comm {
	mass = 5 kg;
	power = 40;
	cost = 1;
      }
      Component gimbal {
	mass = 5 kg;
	power = 5;
	cost = 0.5;
      }
    } /* CommEarth */
    Component battery {
      mass = 5.5 kg;
      cost = 0.2;
    }
    Component PMAD {
      mass = 4 kg;
      cost = 0.3;
    }
    
    Component structure {
      cost = 1;
      Component support {
	mass = 0.2*sum(key mass of satellite);
      }
      Component bus {
	mass = 5 kg;
      }
      Component tankmount {
	mass = 0.3*sum(key mass of coldgastank);
      }
    } /* Structure */
    
    Component propulsion {
      cost = 0.5;
      Component coldgastank {
	Component coldgas {mass = 6 kg;}
	Component tank{mass = 2 kg;}
      } /* coldgastank */
      Component nozzles {mass = 2 kg;}
      Component actuators {mass = 2 kg;}
    } /* propulsion */
    
    Component mockup {cost = 0.6;}
  } /* satellite */
} /* orbiter stage */
  


Component CommSatellite{ /* If there is satellite */
  mass = 3 kg;
  power = 10; /* for 1 W output */
  cost = 0.5 dollar + backupEarth.cost;
  Component Transciever{}
  Component Modem{}
  Component Cable{}
  Component Antenna{}
  Component backupEarth{
    mass = 3 kg;
    distrib power = distribution(max=30,avg=0,min=0);
    cost = 1;
  }
}
