/*******************************************************************************
+
+  LEDA 3.5
+
+  face_map.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/
#ifndef LEDA_FACE_MAP_H
#define LEDA_FACE_MAP_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350043
#include <LEDA/REDEFINE_NAMES.h>
#endif


//------------------------------------------------------------------------------
// node maps
//
// S. Naeher (1996)
//------------------------------------------------------------------------------


/*{\Manpage {face_map} {E} {Face Maps} }*/

#include <LEDA/graph.h>

template <class E>
class face_map : public graph_map {

/*{\Mdefinition
An instance of the data type |\Mname| is a map for the faces of a graph
$G$, i.e., equivalent to $map\<face,E\>$ (cf. \ref{Maps}). It can be
used as a dynamic variant of the data type $face\_array$
(cf. \ref{Node Arrays}). }*/


  E def;

void clear_entry(GenPtr& x) const { LEDA_CLEAR(E,x); }
void copy_entry(GenPtr& x)  const { LEDA_COPY(E,x);  }
void init_entry(GenPtr& x)  const { x = leda_copy(def); }


  
public:

const char* elem_type_name() const { return LEDA_TYPE_NAME(E); }

/*{\Mcreation M }*/

face_map() : graph_map(nil,0,2) { init_def_entry(); }
/*{\Mcreate  introduces a variable |\Mvar| of type |\Mname| and initializes
             it to the map with empty domain. }*/

face_map(const graph& G) : graph_map(&G,G.max_face_index()+1,2) 
{ init_table();  init_def_entry(); }
/*{\Mcreate  introduces a variable |\Mvar| of type |\Mname| and initializes
             it with a mapping $m$ from the set of all faces of $G$ into 
             the set of variables of type $E$. The variables in the range 
             of $m$ are initialized by a call of the default constructor 
             of type $E$. }*/

face_map(const graph& G, E x) : graph_map(&G,G.max_face_index()+1,2) 
{ def = x; init_table();  init_def_entry(); }
/*{\Mcreate  introduces a variable |\Mvar| of type |\Mname| and initializes
             it with a mapping $m$ from the set of all faces of $G$ into 
             the set of variables of type $E$. The variables in the range 
             of $m$ are initialized with a copy of $x$. }*/

~face_map() { clear_table(); clear_def_entry(); }


/*{\Moperations 1.3 4.3 }*/


void init()  { graph_map::init(nil,1,2); }
/*{\Mop      makes |\Mvar| a face map with empty domain. }*/

void init(const graph& G)      
{ graph_map::init(&G,G.max_face_index()+1,2); 
  init_table(); }
/*{\Mop       makes |\Mvar| to a mapping $m$ from the set of all faces of $G$ 
              into the set of variables of type $E$. The variables in the 
              range of $m$ are initialized by a call of the default 
              constructor of type $E$. }*/

void init(const graph& G, E x) 
{ graph_map::init(&G,G.max_face_index()+1,2); 
  def = x;
  init_table(); }
/*{\Mop       makes |\Mvar| to a mapping $m$ from the set of all faces of $G$ 
              into the set of variables of type $E$. The variables in the 
              range of $m$ are initialized with a copy of $x$. }*/

const E& operator()(face f) const 
{  return LEDA_CONST_ACCESS(E,graph_map::map_read(f)); }

const E& operator[](face f) const 
{  return LEDA_CONST_ACCESS(E,graph_map::map_read(f)); }

E& operator[](face f) {  return LEDA_ACCESS(E,graph_map::map_access(f)); }
/*{\Marrop    returns the variable $M(f)$. }*/

};

/*{\Mimplementation
Node maps are implemented by an efficient hashing method based on the 
internal numbering of the faces. An access operation takes expected 
time $O(1)$. 
}*/



#if LEDA_ROOT_INCL_ID == 350043
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif
