//============================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//============================================================================

#include <iostream.h>
#include <stdlib.h>
#include "XppWindow.h"

//============================================================================
XppWindow::XppWindow(int w, int h, int x, int y, char *name)
{
  // Open X Window on root window
  Open( DefaultRootWindow(gXManager.AppDisplay()), w, h, x, y, name, 0);
  // note: if x || y < 0  then the window will be positioned with mouse
}

//============================================================================
XppWindow::XppWindow(XppWindow& win, int w, int h, int x, int y, int bw)
{
  // Open a subwindow in the XppWindow <win> with the specified dimensions
  Open(win.fWindow, w, h, x, y, "", bw);
}

//============================================================================
XppWindow::~XppWindow()    
{ 
  // Close the X Window
  Close();       
}

//============================================================================
void XppWindow::DrawPoint(int x, int y)         
{ 
  // Draw point at <x, y> 
  XDrawPoint( fDisplay, fDrawable, fGC, x, y ); 
}

//============================================================================
void XppWindow::DrawLine(int x1, int y1, int x2, int y2)
{
  // Draw line from <x1, y1> to <x2, y2> 
  XDrawLine( fDisplay, fDrawable, fGC, x1, y1, x2, y2 );
}

//============================================================================
void XppWindow::DrawRectangle(int x, int y, int w, int h)
{
  // Draw rectangle with top left corner at <x, y>, width <w>, and height <h> 
  XDrawRectangle( fDisplay, fDrawable, fGC, x, y, w, h );
}

//============================================================================
void XppWindow::DrawFilledRectangle(int x, int y, int w, int h)
{
  // Draw filled rectangle with top left corner at <x, y>, width <w>, and height <h> 
  XFillRectangle( fDisplay, fDrawable, fGC, x, y, w, h );
}

//=======================================================================
void XppWindow::DrawSquare(int x, int y, int r)
{
  // Draw square centered at <x, y> with sides of length 2<r> 
  XDrawRectangle( fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2 );
}

//============================================================================
void XppWindow::DrawFilledSquare(int x, int y, int r)
{
  // Draw filled square centered at <x, y> with sides of length 2<r>
  XFillRectangle( fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2 );
}

//============================================================================
void XppWindow::DrawArc(int x, int y, int r, int theta1, int theta2)
{
  XDrawArc(fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2, 
                                                   theta1*64, theta2*64 );
}

//============================================================================
void XppWindow::DrawFilledArc(int x, int y, int r, int theta1, int theta2)
{
  XFillArc(fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2, 
                                                   theta1 * 64, theta2 * 64 );
}

//============================================================================
void XppWindow::DrawCircle(int x, int y, int r)
{
  // Draw circle centered at <x, y> with radius <r> 
  XDrawArc(fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2, 0, 23040);
}

//============================================================================
void XppWindow::DrawFilledCircle(int x, int y, int r)
{
  XFillArc(fDisplay, fDrawable, fGC, x-r, y-r, r*2, r*2, 0, 23040);
}


//============================================================================
void XppWindow::DrawText(int x, int y, char *text)
{
  // Draw text in window 
  XDrawImageString(fDisplay, fDrawable, fGC, x, y, text, strlen(text));
}
    

//============================================================================
void XppWindow::SetAutoRedrawOnExpose(void)
{
  if(!fAutoRedrawOnExpose)
   {
     fPixmap = XCreatePixmap(fDisplay, fWindow, fWidth, fHeight, 8 );   
                                                    // 8 = the depth of pixmap
     // Clear the pixmap
     XSetForeground(fDisplay, fGC, fWhite);
     XFillRectangle(fDisplay, fPixmap, fGC, 0, 0 , fWidth, fHeight);    
     // Reset to foreground color
     XSetForeground(fDisplay, fGC, fBlack);                             

     fAutoRedrawOnExpose = 1;   // Set flag
     fDrawable = fPixmap;       // Use the pixmap as the drawable
   }
}



//============================================================================
void XppWindow::Clear(void)        
{
  // Clear the window 
  if(fAutoRedrawOnExpose)    
    {
      // Clear the pixmap
      XSetForeground(fDisplay, fGC, fWhite);
      XFillRectangle(fDisplay, fPixmap, fGC, 0, 0 , fWidth, fHeight);    
      // Reset to foreground color
      XSetForeground(fDisplay, fGC, fBlack);
    }
  else
    XClearWindow(fDisplay, fWindow);
}


//============================================================================
void XppWindow::SetDrawColor(char *colorname)
{
  // Set the foreground color to the color specified by the <colorname> string
  unsigned long color;
  color = FindColor(colorname);
  XSetForeground(fDisplay, fGC, color);
}

//============================================================================
void XppWindow::SetDrawBlack(void)
{
  // Set the foreground color to black
  XSetForeground(fDisplay, fGC, fBlack);
}

//============================================================================
void XppWindow::SetDrawWhite(void)
{
  // Set the foreground color to white
  XSetForeground(fDisplay, fGC, fWhite);
}

//============================================================================
void XppWindow::Flush(void)
{
  // Flush X display buffer 

  if(fAutoRedrawOnExpose)  // Copy pixmap onto window 
    XCopyArea(fDisplay,fPixmap,fWindow,fGC, 0,0,fWidth,fHeight,0,0);

  XFlush(fDisplay);      // Flush the display
}







//============================================================================
void XppWindow::SelectWindowEvents(long eventMask)
{
  // Set the event mask for this window and instruct server to 
  // deliver selected events

  fEventMask = eventMask;
  XSelectInput(fDisplay, fWindow, fEventMask);
}

//============================================================================
void XppWindow::Pause(void)
{
  // Wait until the user hits a mouse button or presses a key
  XEvent event;
  do {
    XNextEvent(fDisplay, &event);
  }
  while((event.type != ButtonPress) && (event.type != KeyPress));
}

//============================================================================

//.. This function needs to get commands applicable to this window only

void XppWindow::GetCommand(char& command)
{
  // Get single character command from keyboard (wait for input) 
  XEvent event;
  KeySym key;
  char text[10];

  strcpy(text, "");
  do 
    {
      XNextEvent(fDisplay, &event);
    }
  while(event.type != KeyPress);
  XLookupString((XKeyEvent *) &event, text, 10, &key, 0);
  command = text[0];
}

//============================================================================
int XppWindow::ScanCommand(char& command)
{
  // Check for keyboard command (real-time). 
  // Returns 1 if keypressed, 0 otherwise
  XEvent event;
  KeySym key;
  char text[10];

  if (XCheckWindowEvent(fDisplay, fWindow, KeyPressMask, &event)) 
    {
      strcpy(text, "");
      XLookupString((XKeyEvent *) &event, text, 10, &key, 0);
      command = text[0];
      //while (XEventsQueued(fDisplay, QueuedAfterReading)) 
      //  {
      //  XNextEvent(fDisplay, &event);
      //  }
      return(1);
    }
  else
    {
      return(0);
    }
}

//============================================================================
int XppWindow::ScanMouse(int& x, int& y)
{
  // Scan for mouse click (0 = no click, 1 = click) 
  XEvent event;

  if (XCheckWindowEvent(fDisplay, fWindow, ButtonPressMask, &event)) 
    {
      y = event.xbutton.x;
      x = event.xbutton.y;
      return(1);
    }
  else 
    {
      return(0);
    }
}

//============================================================================
int XppWindow::Exposed(void)
{
  // Returns 1 if window has been exposed, 0 otherwise 
  XEvent event;

  if (XCheckWindowEvent(fDisplay, fWindow, ExposureMask, &event)) 
    { return(1); }
  else 
    { return(0); }
}










                     // ====== Private member functions ======== //


//============================================================================
unsigned long XppWindow::FindColor(char *name)
{
  // Get the color specified by the <name> string  
  XColor exact, color;
  if (XAllocNamedColor(fDisplay, fCmap, name, &exact, &color) == 0)
    {cout << "XppWindow::FindColor -> could not allocate color"; exit(1);}
  return (color.pixel);
}

//============================================================================
void XppWindow::Open(Window win, int w, int h, int x,int y, char *name, int bw)
{
  // Open a  window on the specified XWindow <win> 
  // with top left corner at location <x, y>, 
  // width <w>, height <h>, and name <name> 

  fDisplay =  gXManager.AppDisplay();// Get the display for this application
  fScreen  =  gXManager.AppScreen(); // Get the screen for this application
  fCmap    =  gXManager.AppCmap();   // Get the color map for this application
      
  fWhite =   WhitePixel(fDisplay, fScreen ); // Get black & white pixel values 
  fBlack =   BlackPixel(fDisplay, fScreen ); // for this screen

  XSizeHints hint;                                    
  hint.x = x;          hint.y = y;
  hint.width = w;      hint.height = h;
  // Negative x or y implies position with mouse
  if( (x<0)||(y<0) ) hint.flags = PPosition | PSize;    
  else               hint.flags = USPosition | USSize;  

  fWindow = XCreateSimpleWindow(fDisplay, win,
				hint.x, hint.y, hint.width, hint.height,
				bw, fBlack, fWhite);

  XSetStandardProperties(fDisplay, fWindow, name, name, None, 0, 0, &hint);

  fGC   = XCreateGC(fDisplay, fWindow, 0, 0);
  XSetBackground(fDisplay, fGC, fWhite);
  XSetForeground(fDisplay, fGC, fBlack);
  XSetLineAttributes(fDisplay, fGC, 0, LineSolid, CapButt, JoinMiter);

  //fEventMask = ButtonPressMask | KeyPressMask | ExposureMask;
  //XSelectInput(fDisplay, fWindow, fEventMask);
  SelectWindowEvents(ButtonPressMask | KeyPressMask | ExposureMask);
  
  XMapRaised(fDisplay, fWindow);

  fWidth  = w;      fHeight = h;      // Save the width and height

  XEvent     event;
  do 
    {                                 // Get rid of all queued expose events
      XNextEvent(fDisplay, &event);
    }
  while(event.type != Expose);        

  fAutoRedrawOnExpose = 0;   // The window cannot redraw itself - default
  fDrawable = fWindow;     // Draw everything on the window itself - default
}


//============================================================================
void XppWindow::Close(void)
{
  // Close the window

  XFreeGC(fDisplay, fGC);
  XDestroyWindow(fDisplay, fWindow);
  if(fAutoRedrawOnExpose) XFreePixmap(fDisplay, fPixmap);
}



//============================================================================
// Redefinition of  the virtual XppClient member function
//  ProcessEvents(). 
// The XppWindow must respond to XEvents that are for this window only.
void XppWindow::ProcessEvents()
{
  XEvent event;
  
  if( XCheckWindowEvent(fDisplay, fWindow, fEventMask, &event) )
    {    // If there is any event for this window 
      if( fAutoRedrawOnExpose && (event.type == Expose) )
	HandleExposeIfAutoRedraw(event);   // Handle expose events if 
                                           // auto-redraw is set
      else
	HandleXEvent(event);          // Handle any other events
    }
}

//============================================================================
// Function that is called if an XEvent for this window is found 
void XppWindow::HandleXEvent(XEvent& event)
{
  // Handle X events in this virtual function

  return;  // Default behavior is to do nothing    
}


//============================================================================
void XppWindow::HandleExposeIfAutoRedraw(XEvent& event)
{
  int x = event.xexpose.x;
  int y = event.xexpose.y;
  int w = event.xexpose.width;
  int h = event.xexpose.height;
  
  XCopyArea(fDisplay, fPixmap, fWindow, fGC, x, y, w, h, x, y);
  XFlush(fDisplay);
}








