//============================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//============================================================================

#include <iostream.h>
#include <stdlib.h>
#include "XppCanvas.h"

extern "C" int abs(int);



//============================================================================
XppCanvas::XppCanvas(int w, int h, int x, int y, char *name)
  :XppWindow(w, h, x, y, name)
{ 
  SetDrawingCoordinates(0.0, w, h, 0.0); // Default cartesian coordinate system
}

//============================================================================
XppCanvas::XppCanvas(XppWindow& win, int w, int h, int x, int y)
  :XppWindow(win, w, h, x, y)
{ 
  SetDrawingCoordinates(0.0, w, h, 0.0); // Default cartesian coordinate system
}




//============================================================================
void XppCanvas::SetDrawingCoordinates(double left, double right, 
                                                       double down, double up)
{
  // Set up cartesian coordinate system.
  //  xaxis range = [<left>, <right>], yaxis range = [<down>,<up>]

  fLeft  = left;     // Horizontal axis leftmost value 
  fRight = right;    // Horizontal axis rightmost value  
  fDown  = down;     // Vertical axis   bottommost value
  fUp    = up;       // Vertical axis   topmost value

  // Make sure these coordinates are valid
  if ( (fLeft == fRight) || (fDown == fUp) )
    {cout << "Invalid arguments in XppCanvas::SetDrawingCoordinates"; exit(1);}

  SetScaling();      // Compute the scaling factors 
}

//============================================================================
void XppCanvas::DrawPoint(double x, double y)         
{ 
  // Draw point at <x, y> 
  XDrawPoint( fDisplay, fDrawable, fGC, WX(x), WY(y) ); 
}

//============================================================================
void XppCanvas::DrawLine(double x1, double y1, double x2, double y2)
{
  // Draw line from <x1, y1> to <x2, y2> 
  XDrawLine( fDisplay, fDrawable, fGC, WX(x1), WY(y1), WX(x2), WY(y2) );
}

//============================================================================
void XppCanvas::DrawRectangle(double x, double y, double w, double h)
{
  // Draw rectangle with top left corner at <x, y>, width <w>, and height <h> 
  XDrawRectangle( fDisplay, fDrawable, fGC, WX(x), WY(y), WLX(w), WLY(h) );
}

//============================================================================
void XppCanvas::DrawFilledRectangle(double x, double y, double w, double h)
{
  // Draw filled rectangle with top left corner at <x, y>, 
  // width <w>, and height <h> 
  XFillRectangle( fDisplay, fDrawable, fGC, WX(x), WY(y), WLX(w), WLY(h) );
}



#define  min(x,y)  (  ( (x) < (y) ) ? (x) : (y) )

//============================================================================
void XppCanvas::DrawRectangleToFit(double x1, double y1, double x2, double y2)
{
  // Draw rectangle to fit diagonally opposite points (x1,y1) and (x2,y2)

  int x1plot = WX(x1);   int y1plot = WY(y1);  // Get window coordinates
  int x2plot = WX(x2);   int y2plot = WY(y2);

  int xplot = min(x1plot, x2plot);         // Get upper left corner cordinates
  int yplot = min(y1plot, y2plot);

  int wplot = abs(x1plot - x2plot);            // Get width and length
  int hplot = abs(y1plot - y2plot);

  XDrawRectangle(fDisplay,fDrawable,fGC, xplot, yplot, wplot, hplot);
}

//============================================================================
void XppCanvas::DrawFilledRectangleToFit(double x1, double y1, 
                                                         double x2, double y2)
{
  // Draw filled rectangle to fit diagonally opposite points 
  // (x1,y1) and (x2,y2)

  int x1plot = WX(x1);   int y1plot = WY(y1);  // Get window coordinates
  int x2plot = WX(x2);   int y2plot = WY(y2);

  int xplot = min(x1plot, x2plot);        // Get upper left corner cordinates
  int yplot = min(y1plot, y2plot);

  int wplot = abs(x1plot - x2plot);            // Get width and length
  int hplot = abs(y1plot - y2plot);

  XFillRectangle(fDisplay,fDrawable,fGC, xplot, yplot, wplot, hplot);
}

//============================================================================
void XppCanvas::DrawSquare(double x, double y, double r)
{
  // Draw square centered at <x, y> with sides of length 2<r> 
  XDrawRectangle(fDisplay, fDrawable, fGC, WX(x-r), WY(y-r), WLX(r*2), WLY(r*2) );
}

//============================================================================
void XppCanvas::DrawFilledSquare(double x, double y, double r)
{
  // Draw filled square centered at <x, y> with sides of length 2<r>
  XFillRectangle(fDisplay, fDrawable, fGC, WX(x-r), WY(y-r), WLX(r*2), WLY(r*2) );
}

//============================================================================
void XppCanvas::DrawArc(double x, double y, double r, 
                                                 double theta1, double theta2)
{
  // Draw arc centered at <x, y> with radius <r> and sweeping from angle
  // <theta1> to angle <theta2> (specified in degrees) 
  int x1 = WX(x - r);  int x2 = WX(x + r);
  int y1 = WY(y - r);  int y2 = WY(y + r); 
  int ulx = min(x1, x2) ;  int uly = min(y1, y2) ;
  
  XDrawArc(fDisplay, fDrawable, fGC, ulx, uly, WLX(r*2), WLY(r*2), 
                                      (int)(theta1 * 64), (int)(theta2 * 64) );
}

//============================================================================
void XppCanvas::DrawFilledArc(double x, double y, double r, 
                                                 double theta1, double theta2)
{
  // Draw filled arc centered at <x, y> with radius <r> and sweeping from angle
  // <theta1> to angle <theta2> (specified in degrees) 
  int x1 = WX(x - r);  int x2 = WX(x + r);
  int y1 = WY(y - r);  int y2 = WY(y + r); 
  int ulx = min(x1, x2) ;  int uly = min(y1, y2) ;

  XFillArc(fDisplay, fDrawable, fGC, ulx, uly, WLX(r*2), WLY(r*2), 
                                        int(theta1 * 64), int(theta2 * 64) );
}

//============================================================================
void XppCanvas::DrawCircle(double x, double y, double r)
{
  // Draw circle centered at <x, y> with radius <r> 
  int x1 = WX(x - r);  int x2 = WX(x + r);
  int y1 = WY(y - r);  int y2 = WY(y + r); 
  int ulx = min(x1, x2) ;  int uly = min(y1, y2) ;

  XDrawArc(fDisplay, fDrawable, fGC, ulx, uly, WLX(r * 2), 
                                                        WLY(r * 2), 0, 23040);
}

//============================================================================
void XppCanvas::DrawFilledCircle(double x, double y, double r)
{
  // Draw filled circle centered at <x, y> with radius <r> 
  int x1 = WX(x - r);  int x2 = WX(x + r);
  int y1 = WY(y - r);  int y2 = WY(y + r); 
  int ulx = min(x1, x2) ;   int uly = min(y1, y2) ;

  XFillArc(fDisplay, fDrawable, fGC, ulx, uly, WLX(r * 2), 
                                                       WLY(r * 2), 0, 23040);
}

//============================================================================
void XppCanvas::DrawText(double x, double y, char *text)
{
  // Draw text in window 
  XDrawImageString(fDisplay, fDrawable, fGC, WX(x), WY(y), text, strlen(text));
}

//============================================================================
void XppCanvas::DrawXAxis(double atY)
{
  DrawLine(fLeft, atY, fRight, atY);    // Draw the xaxis <atY>
}

//============================================================================
void XppCanvas::DrawYAxis(double atX)
{
  DrawLine(atX, fDown, atX, fUp);       // Draw the yaxis <atX>
}

//============================================================================
void XppCanvas::DrawXYAxes(double atX, double atY)
{
  DrawYAxis(atX);                       // Draw x and y axes
  DrawXAxis(atY);
}

//============================================================================
int XppCanvas::ScanMouse(double& x, double& y)
{
  // Scan for mouse click (0 = no click, 1 = click) 
  XEvent event;

  if (XCheckWindowEvent(fDisplay, fWindow, ButtonPressMask, &event)) 
    {
      // Get mouse window position
      double px = event.xbutton.x;  
      double py = event.xbutton.y;  

      // Convert window coordinates to canvas cartesian coordinates
      x = XW(px);
      y = YW(py); 

      return(1);
    }
  else 
    {
      return(0);
    }
}


double XppCanvas::XW(double x) { return( fLeft + (fRight - fLeft)*x/fWidth ); }

double XppCanvas::YW(double y) { return( fUp + (fDown - fUp)*y/fHeight );     }







                    // ====== Private member functions ========





//============================================================================
void XppCanvas::SetScaling()
{
  // Compute scaling factors for drawing in the window
  fHScale = fWidth  / (fRight - fLeft) ;  
  fVScale = fHeight / (fDown - fUp);
}


//============================================================================
int XppCanvas::WLX( double lx)
{
  return( abs( (int)(lx  * fHScale) ) );
}

//============================================================================
int XppCanvas::WLY( double ly)
{
  return( abs( (int)(ly  * fVScale) ) );
}

//============================================================================
int XppCanvas::WX( double x )
{
  return ( (int)((x - fLeft) * fHScale) );
}

//============================================================================
int XppCanvas::WY( double y )
{
  return ( (int)(( y - fUp ) * fVScale) );
}





