//============================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//============================================================================

#include <math.h>
#include "SysPendulum.h"
#include "SysIntegration.h" 
#include "StdPi.h"

const double  g = 9.8;        // Gravity  (m/sec^2)             
const double  M = 1.0;        // Mass of the cart (Kg)          
const double  m = 1.0;        // Mass of the pendulum (Kg)      
const double  l = 1.0;        // Length of the pendulum (m)     

const double Mm  = M + m;     // Some handy consts
const double Mmg = Mm * g;
const double Mml = Mm * l;
const double ml  = m * l;

//============================================================================
SysPendulum::SysPendulum(void)
{
  fTheta     = 0.0;     fThetaDot  = 0.0;     fThetaDDot = 0.0;
  fH         = 0.0;     fHDot      = 0.0;     fHDDot     = 0.0;
  fForce     = 0.0;
}

//============================================================================
void SysPendulum::Reset(double t, double td, double h, double hd)
{
  fTheta     = t;      fThetaDot  = td;      fThetaDDot = 0.0;
  fH         = h;      fHDot      = hd;      fHDDot     = 0.0;
  fForce     = 0.0;    
}

//============================================================================
void SysPendulum::Step(double force, double stepSize)
{
    fTheta = fmod( fTheta, c2PI );     // Angle is Theta modulo 2pi 
    
    double sinTheta = sin( fTheta );
    double cosTheta = cos( fTheta );
 
    // Equations of motion for the pendulum and cart 
    fThetaDDot = Mmg*sinTheta + Mm*sinTheta*fHDot*fThetaDot - 
                cosTheta*(force + ml*fThetaDot*fThetaDot*sinTheta);
    fThetaDDot /= ( Mml - ml*cosTheta*cosTheta );
    fHDDot  = ( force + ml*( fThetaDot*fThetaDot*sinTheta - 
                fThetaDDot*cosTheta ) )/(Mm);

    // Integrate the accelerations and velocities 
    fThetaDot  = Euler(fThetaDot, fThetaDDot, stepSize);
    fTheta     = Euler(fTheta,    fThetaDot,  stepSize);
    fHDot      = Euler(fHDot,     fHDDot,     stepSize);
    fH         = Euler(fH,        fHDot,      stepSize);
    fForce = force;   
}






















