//============================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved.
//============================================================================

#include "SysNNetwork.h"
#include "SysIntegration.h"
#include "SysSigmoid.h"
#include "StdTypes.h"

//============================================================================
SysNNetwork::SysNNetwork(int numberOfNodes) 
 : fNumberOfNodes( numberOfNodes ),
   fTau( numberOfNodes ),                 // Array of timeconstants
   fBias( numberOfNodes ),                // Array of biases
   fInputScale( numberOfNodes ),          // Array of input scales
   fOutput( numberOfNodes ),              // Array of outputs
   fState( numberOfNodes ),               // Array of internal states
   fExternalInput( numberOfNodes ),       // Array of external inputs 
   fWeight( numberOfNodes, numberOfNodes ),  // Matrix of weights
   fConnect( numberOfNodes, numberOfNodes )  // Connectivity matrix
{
  for(int i = 0; i < fNumberOfNodes; i++)    // Make all nodes unconnected
    for(int j = 0; j < fNumberOfNodes; j++) 
      fConnect[i][j] = 0;			
}


//============================================================================
void SysNNetwork::Connect(int i, int j)
{
  fConnect[i][j] = 1;	 // the jth node is connected to the ith node
  fWeight[i][j]  = 0.0;  // default weight is zero
}

//============================================================================
void SysNNetwork::Disconnect(int i, int j)
{
  fConnect[i][j] = 0;	//  Disconnect connection from j to node i
  fWeight[i][j]  = 0.0;
}

//============================================================================
void SysNNetwork::Reset(void)
{
  for (int i = 0; i < fNumberOfNodes; i++)
    {
      fExternalInput[i] = 0.0;		
      fState[i]  = 0.0;
      fOutput[i] = Sigmoid( fState[i] - fBias[i] );
    }
}


#if 0 // Replaced by more efficient version. Code is here for documentation.
//============================================================================
void SysNNetwork::Step(double stepsize)
{
  double inputsum;  int i, j;
  for (i=0; i<fNumberOfNodes; i++)
    {
      inputsum = fInputScale[i] * fExternalInput[i];
      for (j=0; j<fNumberOfNodes; j++)  
	if(fConnect[i][j]) inputsum += fWeight[i][j] * fOutput[j];

      fState[i]=Euler(fState[i],(inputsum - fState[i])/fTau[i],stepsize);
    }
  for (i=0; i<fNumberOfNodes; i++)	
    fOutput[i] = Sigmoid( fState[i] - fBias[i] );
}	
#endif	


//============================================================================
void SysNNetwork::Step(double stepsize)
{
  double inputsum;  int i, j;
  for (i=0; i<fNumberOfNodes; i++)    // For each node
    {
      // Compute the net input to the node
      inputsum = fInputScale[i] * fExternalInput[i];
      double *w = fWeight[i];    // Pointer to ith row of weight matrix
      int    *c = fConnect[i];   // Pointer to ith row of connectivity matrix  
      for (j=0; j<fNumberOfNodes; j++)  
	if(c[j]) inputsum += w[j] * fOutput[j];

      // Compute the next state
      fState[i] = Euler(fState[i], (inputsum - fState[i])/fTau[i], stepsize);
    }
  // Compute the outputs
  for (i=0; i<fNumberOfNodes; i++)
    fOutput[i] = Sigmoid( fState[i] - fBias[i] );
}	
	

//============================================================================
void SysNNetwork::Print(ostream& os) const
{
  os << fNumberOfNodes << "\n";
  os << fTau        <<  "\n\n";
  os << fBias       <<  "\n\n";
  os << fInputScale <<  "\n\n";
  os << fConnect    <<  "\n";
  os << fWeight     <<  "\n";
}


//===========================================================================
ostream& operator << (ostream& os, const SysNNetwork& n)
{
  n.Print(os);
  return(os);
}


//===========================================================================
void LoadState(istream& is, SysNNetwork& n)
{
  LoadState(is, n.fNumberOfNodes);
  LoadState(is, n.fTau);
  LoadState(is, n.fBias);
  LoadState(is, n.fInputScale);
  LoadState(is, n.fConnect);
  LoadState(is, n.fWeight);

  LoadState(is, n.fOutput);
  LoadState(is, n.fState);
  LoadState(is, n.fExternalInput);
}

//===========================================================================
void SaveState(ostream& os, SysNNetwork& n)
{
  SaveState(os, n.fNumberOfNodes);  os << "\n";
  SaveState(os, n.fTau);            os << "\n";
  SaveState(os, n.fBias);           os << "\n";
  SaveState(os, n.fInputScale);     os << "\n";
  SaveState(os, n.fConnect);        os << "\n";
  SaveState(os, n.fWeight);         os << "\n";

  SaveState(os, n.fOutput);         os << "\n";
  SaveState(os, n.fState);          os << "\n";
  SaveState(os, n.fExternalInput);  os << "\n";
}















