//=========================================================================
// Copyright (c) 1995 Leslie Picardo.  All rights reserved
//=========================================================================

#ifndef __StdBitStringH__
#define __StdBitStringH__

#include "StdArray.h"

//=========================================================================
// A StdBitString implements an array of bits. eg - 0101001
// The bits are numbered from left to right starting with  bit 0
//=========================================================================

// The bits are packed into an array of AWords.
// AWord may be any one of the following data types. 

   typedef unsigned char  AWord;         // 1 byte  on a Sparc
// typedef unsigned short AWord;         // 2 bytes on a Sparc
// typedef unsigned int   AWord;         // 4 bytes on a Sparc

const int cBitsPerByte = 8;
const int cBitsPerAWord = cBitsPerByte * sizeof(AWord);

//.........................................................................

class StdBitString : private StdArray<AWord>
{
  public:
    // Constructors
    StdBitString(int numberOfBits = 1);
    StdBitString(StdBitString&, int bitLow, int bitHigh);    

    // Read access to data
    inline int NumberOfBits(void) const;
    inline int Size(void) const;
    inline int operator[] (int index) const;
    
    // Operations on individual bits
    inline void Set(int b);
    inline void Clear(int b);
    inline void Invert(int b);
	
    // Operations on entire bitstring
    inline void Set(void);
    inline void Clear(void);
    inline void Invert(void);

    inline StdBitString& operator &= (const StdBitString& s);
    inline StdBitString& operator |= (const StdBitString& s);
    inline StdBitString& operator ^= (const StdBitString& s);

    friend StdBitString operator & (const StdBitString&, const StdBitString&);
    friend StdBitString operator | (const StdBitString&, const StdBitString&);
    friend StdBitString operator ^ (const StdBitString&, const StdBitString&);

    // Changing the size
    void Resize(int numberOfBits);
    void ResizeSave(int numberOfBits);

    // Return a substring
    StdBitString SubString(int bitLow, int bitHigh);

    // Binary and Grey code conversions
    void BinaryToGrey(void); 
    void GreyToBinary(void); 
   
    // Printing the bitstring 
    void Print(ostream& os) const;  
    friend ostream& operator << (ostream&, const StdBitString&);

    // Saving and retrieving the state of the bitstring
    friend void LoadState(istream& is, StdBitString& b);
    friend void SaveState(ostream& os, StdBitString& b);

  private:
    int fNumberOfBits;
    int Index(int b)  const { return( b / cBitsPerAWord ); }
    int Offset(int b) const { return( b % cBitsPerAWord ); }
    AWord Mask(int b) const { return( 1 << Offset(b) ); }
};



// Non-member functions declarations ..........................................

// Binary and Grey code conversions
void ConvertBinaryToGrey(const StdBitString& binary, StdBitString& grey);
void ConvertGreyToBinary(const StdBitString& grey,   StdBitString& binary );
double ConvertBinaryToDecimal(const StdBitString& binary);
double ConvertBinaryToDecimal(const StdBitString& b, int b0, int b1);

// Inline function definitions ................................................

inline int StdBitString::NumberOfBits(void) const { return (fNumberOfBits); }
inline int StdBitString::Size(void)         const { return (fNumberOfBits); }

inline int StdBitString::operator[] (int b) const 
{ 
  assert( (b >= 0)&&(b<fNumberOfBits) ); 
  return( (fX[Index(b)] & Mask(b)) ? ~0 : 0 ); 
}

inline void StdBitString::Set(int b)       
{ assert( (b >= 0)&&(b<fNumberOfBits) );   fX[Index(b)] |=  Mask(b); }

inline void StdBitString::Clear(int b) 
{ assert( (b >= 0)&&(b<fNumberOfBits) );   fX[Index(b)] &= ~Mask(b); }

inline void StdBitString::Invert(int b)    
{ assert( (b >= 0)&&(b<fNumberOfBits) );   fX[Index(b)] ^=  Mask(b); }

inline void StdBitString::Set(void)    {for(int i=0;i<fSize;i++) fX[i]= ~0;   }
inline void StdBitString::Clear(void)  {for(int i=0;i<fSize;i++) fX[i]=  0;   }
inline void StdBitString::Invert(void) {for(int i=0;i<fSize;i++) fX[i]=~fX[i];}
inline StdBitString& StdBitString::operator &= (const StdBitString& s) 
{ return(*this = *this & s); }
    
inline StdBitString& StdBitString::operator |= (const StdBitString& s) 
{ return(*this = *this | s); }

inline StdBitString& StdBitString::operator ^= (const StdBitString& s) 
{ return(*this = *this ^ s); }
    
//inline StdBitString& StdBitString::operator ~  (void)
//{ Invert(); return (*this); }          //.. should return a copy ?

// ...........................................................................

#endif









