//=========================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//=========================================================================

#include "StdBitString.h"
#include "StdAssert.h"

// This code is machine dependant.

// A StdBitString contains bits that are packed into an array of words.

// In order to mask the b'th bit we use the left shift operator <<
// in the private member function  Mask(). This assumes that the 
// machine represents a 1 as ...00001. (technically it's called 
// Little-endian or Big-endian - Whatever.) 

// Bits are packed into AWords. Within each AWord the bits are arranged
// from right to left. If a AWord is an unsigned char (1 byte), for instance, 
// then the bit sequence would be 
//         AWord[2]         AWord[1]       AWord[0]     <=  *AWord
//        b[23]...b[16] ,  b[15]...b[8],  b[7]...b[0]

// If your machine represents a 1 as 10000... then the Mask()
// member function must be modified to use the right shift operator >>
// instead of the left shift operator.

// Private Member Functions
// Index(int b)  returns the index of the AWord containing the b'th bit.
// Offset(int b) returns the position of the bth bit within the AWord
//               containing the b'th bit.
// Mask(int b)   Is used to mask out all the bits except the b'th bit.



//=======================================================================
StdBitString::StdBitString(int n)
  :StdArray<AWord>( Index(n-1) + 1 ),          // Allocate required words
   fNumberOfBits(n)
{  assert(n>0);  }


//=======================================================================
StdBitString::StdBitString(StdBitString& s, int b1, int b2)
  :StdArray<AWord>( Index(b2-b1) + 1 ),       // Allocate required words
   fNumberOfBits(b2-b1+1)
{
  // Create a copy of the substring s from  bits [b1] till [b2].
  assert( (b2 > b1) ); 
  assert( (b1 >= 0) && (b2 < s.NumberOfBits()) );

  for(int i=0; i<fNumberOfBits; i++)
    if(s[b1+i]) Set(i); else Clear(i);
}


StdBitString StdBitString::SubString(int b1, int b2)
{
  // Return a substring from  bits [b1] till [b2]
  assert( (b2 > b1) ); 
  assert( (b1 >= 0) && (b2 < fNumberOfBits) );

  StdBitString temp( b2-b1+1 );    // The substring to return
  StdBitString& self = *this;      // This object
  
  for(int i=0; i<temp.NumberOfBits(); i++)  
     if(self[b1+i]) temp.Set(i); else temp.Clear(i); // Copy bits

  return(temp);
}





//=======================================================================
void StdBitString::Resize(int n)
{
  // Resize the bitstring to <n> bits
  assert(n>0); 
  StdArray<AWord>::Resize( Index(n-1) + 1 );   // Resize words
  fNumberOfBits = n;
}

//=======================================================================
void StdBitString::ResizeSave(int n)
{
  // Resize the bitstring to <n> bits and save old bits
  assert(n>0);
  StdArray<AWord>::ResizeSave( Index(n-1) + 1 );
  fNumberOfBits = n;
}

//=======================================================================
StdBitString operator & (const StdBitString& s1, const StdBitString& s2)
{
  // AND the two strings. The strings should be the same size.
  assert( s1.NumberOfBits() == s2.NumberOfBits() );

  StdBitString tmp( s1.NumberOfBits() );                // Create temporary
  for(int i=0; i<tmp.fSize; i++)  tmp.fX[i] = s1.fX[i] & s2.fX[i]; // AND
  return(tmp);
}

//=======================================================================
StdBitString operator | (const StdBitString& s1, const StdBitString& s2)
{
  // OR the two strings. The strings should be the same size.
  assert( s1.NumberOfBits() == s2.NumberOfBits() );

  StdBitString tmp( s1.NumberOfBits() );               // Create temporary
  for(int i=0; i<tmp.fSize; i++)  tmp.fX[i] = s1.fX[i] | s2.fX[i]; // OR
  return(tmp);
}

//=======================================================================
StdBitString operator ^ (const StdBitString& s1, const StdBitString& s2)
{
  // XOR the two strings. The strings should be the same size.
  assert( s1.NumberOfBits() == s2.NumberOfBits() );

  StdBitString tmp( s1.NumberOfBits() );                // Create temporary
  for(int i=0; i<tmp.fSize; i++)  tmp.fX[i] = s1.fX[i] ^ s2.fX[i]; // XOR
  return(tmp);
}


//=======================================================================
void StdBitString::Print(ostream& os) const
{
  // Print the bitstring in binary form - > b[0], b[1] ... b[n]
  for(int i = 0; i<fNumberOfBits; i++) 
    os << ( (fX[Index(i)] & Mask(i)) ? 1 : 0 );
}



//=======================================================================
void StdBitString::BinaryToGrey(void)
{
  // Assume that this bitstring has a binary encoding and convert it to  
  // grey code

  StdBitString  temp(*this);      // Create a temporary copy
  ConvertBinaryToGrey( temp, *this);   
}

//=======================================================================
void StdBitString::GreyToBinary(void)
{
  // Assume that this bitstring has a grey code encoding and convert it 
  // to binary

  StdBitString  temp(*this);     // Create a temporary copy
  ConvertGreyToBinary(temp, *this);
}









           ////// Overloaded non-member functions \\\\\\

//=======================================================================
void ConvertBinaryToGrey(const StdBitString& b, StdBitString& g)
{
  // Replace the StdBitString <g> with the grey code equivalent of the 
  // binary code in StdBitString <b>

  assert( b.NumberOfBits() == g.NumberOfBits());  // strings must be same size
 
  // To convert from Binary to Grey use
  //      g[i]  = b[i]             if i = 0
  //      g[i]  = b[i-1] XOR b[i]  if i > 0 
  // where g[i] is the ith grey code bit and b[i] the ith binary code bit
  // and the bitstrings are numbered from left to right. 

  if(b[0]) g.Set(0); else g.Clear(0);                 // b[0] = g[0]
  for(int i=1; i<g.NumberOfBits(); i++)
      if((b[i-1])^(b[i])) g.Set(i); else g.Clear(i);  // g[i] = b[i-1] XOR b[i]
}



//=======================================================================
void ConvertGreyToBinary(const StdBitString& g,StdBitString& b )
{
  // Replace the StdBitString <b> with the binary code equivalent of the 
  // grey code in StdBitString <g>

  assert( b.NumberOfBits() == g.NumberOfBits());  // strings must be same size

  // To convert from Grey to Binary use
  //    b[i] = g[i]             if i = 0
  //    b[i] = b[i-1] XOR g[i]  if i > 0
  // where g[i] is the ith grey code bit and b[i] the ith binary code bit
  // and the bitstrings are numbered from left to right. 

  if(g[0]) b.Set(0); else b.Clear(0);                // b[0] = g[0] 
  for(int i=1; i<b.NumberOfBits(); i++)                
    if((b[i-1])^(g[i])) b.Set(i); else b.Clear(i);   // b[i] = b[i-1] XOR g[i]
}


//======================================================================
double ConvertBinaryToDecimal(const StdBitString& b)
{
  // Return the decimal equivalent of the binary code in StdBitString b

  int size = b.NumberOfBits();         // The length of the binary code 
  double decimal = 0.0;                // The decimal value to return
  for(int i=0; i< size; i++)  
    {
      decimal = decimal*2;
      decimal += ( b[i] ? 1 : 0 );
    }
  return(decimal);
}

//======================================================================
double ConvertBinaryToDecimal(const StdBitString& b, int b0, int b1)
{
  // Return the decimal equivalent of the binary code in StdBitString b
  // from bit b0 till bit b1

  assert(b1 >= b0);

  double decimal = 0.0;                // The decimal value to return
  for(int i=b0; i<=b1; i++)  
    {
      decimal = decimal*2;
      decimal += ( b[i] ? 1 : 0 );
    }
  return(decimal);
}








//=======================================================================
ostream& operator << (ostream& os, const StdBitString& s)
{
  s.Print(os);
  return os;
}

//=======================================================================
void LoadState(istream& is, StdBitString& b)
{
  StdArray<AWord>& a = b;
  is >> hex;               // Load as hexadecimal
  LoadState(is, a);        // Load Base  
  is >> dec;               // Revert back to decimal
  is >> b.fNumberOfBits; assert(b.fNumberOfBits>0);
}

//=======================================================================
void SaveState(ostream& os, StdBitString& b)
{
  StdArray<AWord>& a = b;
  os << hex;         // Save in hexadecimal form  
  SaveState(os, a);  // Save Base
  os << dec;         // Revert to decimal
  os << b.fNumberOfBits << "\n";
}








