// ===========================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
// ===========================================================================

#include "StdArray.h"
#include "StdTypes.h"


//============================================================================
template<class ZZ>
StdArray<ZZ>::StdArray(int size)      // Create an array of the specified size
  :fX(0), fSize(size)
{ 
  assert(size>=0);                               
  if(size>0) { fX = new ZZ[size];  assert(fX);}  // New array   
}


//============================================================================
template<class ZZ>                              // Create a copy of the source 
StdArray<ZZ>::StdArray(const StdArray<ZZ>& source)    
  :fX(0), fSize(source.fSize)
{
  if(source.fSize>0) { fX = new ZZ[source.fSize];  assert(fX); }  // New array
  for(int i=0; i<fSize; i++) fX[i] = source.fX[i];        // Copy source array
}


//============================================================================
template<class ZZ>
StdArray<ZZ>::~StdArray(void)                            // Delete the array 
{
  delete [] fX;   // Delete array
}


//============================================================================
template<class ZZ> 
const StdArray<ZZ>& StdArray<ZZ>::operator= (const StdArray<ZZ>& source)
{                                                     // Assign source to this
  if (this != &source) 
    {
      if(fSize != source.fSize) Resize(source.fSize);    // Make same size
      for(int i=0; i<fSize; i++)  fX[i] = source.fX[i];  // Copy source array
    }
  return( *this);
}


//============================================================================
template<class ZZ>
void StdArray<ZZ>::Resize(int size)
{
  assert(size>=0);
  if(fSize == size) return;  // No need to resize

  ZZ* temp = 0;
  if(size>0) { temp = new ZZ[size];  assert(temp); }  // New array

  delete [] fX;   // Delete old array
  fX = temp;      // Use new array
  fSize = size;
}


//============================================================================
template<class ZZ>
void StdArray<ZZ>::ResizeSave(int size)
{
  assert(size>=0);
  if(fSize == size) return;   // No need to resize
  
  ZZ* temp = 0;
  if(size>0) { temp = new ZZ[size];  assert(temp); }  // New array

  int minSize = ( (fSize < size) ? fSize : size ); 
  for(int i=0; i<minSize; i++)  temp[i] = fX[i];     // Copy old array into new
      
  delete [] fX;       // Delete old array
  fX = temp;          // Use new array
  fSize = size;
}


//============================================================================
template<class ZZ>
void StdArray<ZZ>::Print(ostream& os) const
{
  for(int i=0; i<fSize; i++)
    os << fX[i] << " ";
}







         ////// Overloaded non member functions \\\\\\

//============================================================================
template<class ZZ>
ofstream& operator << (ofstream& ofs, const StdArray<ZZ>& a)
{
  ostream& os = ofs;  // Convert to an ostream
  os << a;            // Call ostream equivalent template function
  return(ofs);
}

//============================================================================
template<class ZZ>
void LoadState(ifstream& ifs, StdArray<ZZ>& a)
{
  istream& is = ifs;  // Convert to an istream
  LoadState(is, a);   // Call istream equivalent template function
}

//============================================================================
template<class ZZ>
void SaveState(ofstream& ofs, StdArray<ZZ>& a) 
{
  ostream& os = ofs; // Convert to an ostream
  SaveState(os, a);  // Call ostream equivalent template function
}

//============================================================================
template<class ZZ>
ostream& operator << (ostream& os, const StdArray<ZZ>& a)
{
  a.Print(os);   // Print the array
  return(os);
}

//============================================================================
template<class ZZ>
void LoadState(istream& is, StdArray<ZZ>& a)
{
  int size;
  is >> size;  assert(size>=0);    // Read array size from istream
  a.Resize(size);                  // Make enough space
  for(int i=0; i<a.Size(); i++) 
      LoadState(is, a[i]);           // LoadState of ith element
}

//============================================================================
template<class ZZ>
void SaveState(ostream& os, StdArray<ZZ>& a) 
{
  os << a.Size() << " ";          // Save size on ostream
  for(int i=0; i<a.Size(); i++)
    SaveState(os, a[i] );         // SaveState of the ith element
}




/*
Notes:

A] The template functions - 

 template<class ZZ> void SaveState(ofstream& ofs, StdArray<ZZ>& a) 
 template<class ZZ> void LoadState(ifstream& ifs, StdArray<ZZ>& a)
 template<class ZZ> ofstream& operator << (ofstream& ofs, const StdArray<ZZ>& a)

  are fstream duplicates of the iostream functions defined later. They
  are really unnecessary since an fstream should work wherever an
  iostream object is expected. However, according to the C++ language
  definition for templates, the compiler will only accept exact matches
  and hence these three workarounds are necessary.

B] If 'new' runs out of memory some compilers will return a null pointer.
   g++ does not return a null pointer - so the assert statements after
   'new' are useless. I have left them there 'cause they are harmless.

*/






