//============================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//============================================================================
#ifndef __StdTableH__
#define __StdTableH__

#include "StdVector.h"

//============================================================================
// StdTable is used to create a lookup table to approximate a function g(x).
//
// A StdTable can be created by sampling a function y = g(x) at <noIntervals>
// intervals in the x range  [<minX>, <maxX>]. 
// Alternatively, a StdTable can be constructed from a StdVector of y values that 
// represent values of the function g(x) sampled at equally spaced intervals 
// in the x range [<minX>, <maxX> ].
//
// StdTable uses linear interpolation on these stored samples to approximate 
// the function g(x).
//============================================================================

class StdTable : private StdVector
{
  friend class TTableView;
  public:
    // Constructors
    StdTable( double(*g)(double x), double minX, double maxX, int noIntervals);
    StdTable( StdVector& x,           double minX, double maxX); 
    StdTable(istream&);

    // Lookup operation 
    inline double operator () (double x) const;

    // Overloaded non-member functions
    friend void LoadState( istream& , StdTable&);
    friend void SaveState( ostream& , StdTable&);

   private:
    double fMin;         // The minimum x value
    double fMax;         // The maximum x value
    double fStepSize;    // The distance between two consecutive table x values
    double fInvStepSize; // 1/fStepsize for code optimization
};


inline double StdTable::operator () (double x) const
{ 
  double xn, dx, y1; int i;              // Return the value of y = f(x) 
  if      (x<fMin) return (fX[0]);       // if x < xmin saturate g(x) at g(xmin)
  else if (x>fMax) return (fX[fSize-1]); // if x > xmax saturate g(x) at g(xmax) 
  else {
    xn = (x - fMin);
    i  = int( xn*fInvStepSize );  // Index of left endpoint
    dx = xn - i * fStepSize;  
    y1 = fX[i];                
    return (y1 + (fX[i+1] - y1)*fInvStepSize*dx); 
  }
}


#endif







