//============================================================================
// Copyright (c) 1995 Leslie Picardo.  All rights reserved
//============================================================================
#include <math.h>

#include "StdAssert.h"
#include "StdTypes.h"
#include "StdTable.h"


//============================================================================
StdTable::StdTable( double(*g)(double x), double minX, double maxX, int noIntervals )
  :StdVector(noIntervals+1), fMin(minX), fMax(maxX)  
{
  // Construct a lookup table for the function => double g(double x). 
  // The function g is sampled at x values in the range [minX, maxX]
  // The x range is divided into <noIntervals> intervals and sampled at 
  // the endpoints of these intervals. 
  // The table stores these sampled values and uses linear interpolation 
  // between the stored points to approximate the function g.
  // Note that the number of sampled points = noIntervals + 1

  assert(maxX > minX);
  assert(noIntervals > 0);

  fStepSize = (maxX - minX)/noIntervals;
  fInvStepSize = (1/fStepSize);

  // Sample the function g and store the y values
  double z; int i;
  for(z=minX, i=0;  i<fSize;  i++, z+=fStepSize) 
    fX[i] = g(z);
}


//============================================================================
StdTable::StdTable( StdVector& x,  double minX, double maxX )
  :StdVector(x), fMin(minX), fMax(maxX)
{
  // Construct a lookup table using the y samples in the StdVector <x>.
  // The samples represent values of the function g(x) sampled at equally 
  // spaced intervals in the x range [<minX>, <maxX>].
  // Note that if there are n samples then this implies n-1 intervals

  assert(maxX > minX);
  assert(fSize > 1);
  
  int noIntervals = fSize - 1;  // no of intervals = No of samples - 1
  
  fStepSize = (maxX - minX)/noIntervals;
  fInvStepSize = (1/fStepSize);
}


//============================================================================
StdTable::StdTable(istream& is)
{
  LoadState(is, (*this) );
}


#if 0
// The lookup operation using the overloaded () operator is now optimized
// inlined code. This code is much easier to read since it shows 
// the original unoptimized code.
//============================================================================
// Original implementation of the lookup operation - unoptimized code.
double StdTable::operator () (double x) const 
{ 
  // Return the value of y = f(x) 

  double y;
  if      (x<fMin) y = fX[0];       // if x < xmin saturate g(x) at g(xmin)
  else if (x>fMax) y = fX[fSize-1]; // if x > xmax saturate g(x) at g(xmax) 
  else
    {
      // Get the endpoints of the interval containing x
      int i  = int( (x - fMin)/fStepSize );  // Index of left endpoint
      double x1 = fMin + i * fStepSize;       // The x value of left endpoint
      double y1 = fX[i];                      // The y value of left endpoint
      double x2 = x1 + fStepSize;             // The x value of right endpoint
      double y2 = fX[i+1];                    // The y value of right endpoint
      y = LinearInterpolate(x, x1, y1, x2, y2); // Interpolate using endpoints
    }
  return(y);
}

//============================================================================
double StdTable::
LinearInterpolate(double x, double x1, double y1, double x2, double y2)  const
{
  // Linear interpolate (x,y) given (x1,y1) and (x2,y2) as interval endpoints

  // return( y1 + ((y2-y1)/(x2-x1))*(x-x1) );  // Unoptimized computation
  return( y1 + (y2-y1)*fInvStepSize*(x-x1) );  // Optimize replacing 1/(x2-x1)
}
#endif

//============================================================================
void LoadState( istream& is, StdTable& t)
{
  // Recreate the state of the StdTable t from the istream is
  StdVector& v = t;            // Convert to base class
  LoadState( is, v );        // Load the base as hex
  LoadState( is, t.fMin );
  LoadState( is, t.fMax );
  LoadState( is, t.fStepSize );
  t.fInvStepSize = (1/t.fStepSize);
}

//============================================================================
void SaveState( ostream& os, StdTable& t)
{
 // Save the state of the StdTable t onto the ostream os

  StdVector& v = t;           // Convert to base class 
  SaveState( os, v );       // Save the base in hex
  SaveState( os, t.fMin ); 
  SaveState( os, t.fMax ); 
  SaveState( os, t.fStepSize); 
  os << "\n";
}














