//===========================================================================
// Copyright (c) 1995 Leslie Picardo.  All rights reserved
//===========================================================================
#include <math.h>        // declaration for sqrt and log
#include "StdRandom.h"
#include "StdAssert.h"

extern "C" long  random  (void);
extern "C" void  srandom (int seed);

// local global variables and functions
const double cgRandomMax =  2147483647.0;    // 2^31 - 1
double gX1, gX2, gSX;        
void GenerateNormals(void);

//===========================================================================
void RandomSeed(int seed) 
{ 
   srandom(seed);  // Seed the random generator
} 

//===========================================================================
double UniformRandom(double min, double max)
{  
  // Return a random double within the range [min, max]
  // The C lib function random() returns a random long between 0 and 2^31-1. 
  // This is scaled down to the range [min, max]
  return ( ( max - min ) * ( (double) random() )/cgRandomMax + min );
}

//===========================================================================
double GaussianRandom(double mean, double variance)
{
  GenerateNormals();  
  return( sqrt(variance) * gX1 +  mean );   
}

//===========================================================================
int   Probability(double p) 
{
  // Returns 1 with probability p, else 0
  assert( ( (p>=0.0)&&(p<=1.0) ) );
  return( (UniformRandom(0.0, 1.0) <= p) ? 1 : 0 );
}

//===========================================================================
int  RandomInt(int n)
{
  // Returns a random int in [0, n-1]
  assert( (n>0) );    
  return ( random()%n );
}

//===========================================================================
int  RandomInt(int min, int max)
{
  // Returns a random int in [min, max]
  assert( (max > min) );
  int x = RandomInt( (max - min) + 1 );  // Get a random int in [0, (max-min)]
  return ( x + min);
}


//===========================================================================
StdVector RandomUnitVector(int n)
{  
  // To generate a random unit vector, create a vector in which each
  // element is a random gaussian. Then normalize the vector magnitude to 1.
  // See "The Art of Computer Programming" by Donald Knuth Vol 2 

  StdVector u(n);    // Create an n-dimensional vector
   
  double r = 0.0;
  for(int i = 0; i < n; i += 2) // Generate the elements of a random vector
  {
    GenerateNormals();
    u[i]   = gX1;   if ((i + 1) == n) {r += gX1 * gX1; break;}
    u[i+1] = gX2;
    r += gSX;
  }
  r = sqrt(r);   // The magnitude to the vector
  u /= r;        // Normalize the random vector to unit magnitude
  return (u);    // Return the unit vector          
}





//===========================================================================
void GenerateNormals(void)
{
  // This function generates two random gaussian global variables gX1 and gX2 

  // This code implements the algorithm described in "The Art of
  // Computer Programming" by Donald Knuth Vol 2 (Pg 117)

  double v1, v2, s, d;                 
  do                                  
    {
      v1 = UniformRandom( -1.0, 1.0 );
      v2 = UniformRandom( -1.0, 1.0 );
      s = v1 * v1  +  v2 * v2;
    }
  while( s >= 1.0);

  d = sqrt( -2 * log(s)/s );

  gX1 = v1 * d;       // gX1 is a gaussian: zero mean and unit variance
  gX2 = v2 * d;       // gX2 is a gaussian: zero mean and unit variance
  gSX  = -2*log(s);   // gSX = gX1^2 + gX2^2
}











