// ===========================================================================
// Copyright (c) 1995 Leslie Picardo.  All rights reserved
// ===========================================================================
#include "StdPDensity.h"
#include "StdTypes.h"


// ===========================================================================
// if X is the r.v. whose probability density function f(x) is being estimated
// then
//                             x+D
//      P{ x < X < x+D } =  integral f(x) = f(x)*D  
//                             x
//
// we approximate P{ x < X < x+D } by  Ni/N  where Ni is the number of 
// data points in that interval and N the total number of data points.
// We can then approximate f(x) = (Ni/N)/D  where D is the length of the 
// interval.
//
// I keep two vectors fN and fY. fN keeps track of the number of datapoints 
// in the ith interval. fY is the estimate of the density function f(x) 
// in the ith interval and is computed from fN.
// For efficiency the density function fY is updated only when the 
// density function estimate is read. ie: a call to the overloaded 
// operator() function is made to get f(x).
// ===========================================================================


// ===========================================================================
StdPDensity::StdPDensity(double xmin, double xmax, int numberOfIntervals)
  : fN(numberOfIntervals), fY(numberOfIntervals)
{
  // Create a density function estimate f(x) of a random variable x
  // [xmin, xmax] is the domain of the function f(x) 

  assert(xmax > xmin);
  assert(numberOfIntervals > 0);
  
  fXMin = xmin;   // Set domain
  fXMax = xmax;
  fIntervalSize = (xmax - xmin)/numberOfIntervals; // Calculate interval length

  fN.Zeros(); // Initialize the vectors to zeros
  fY.Zeros(); 

  fNumberOfDataPoints = 0;         // There are no datapoints initially
  fNumberOfDiscardedPoints = 0;
  
  fUpdateFlag = 0;   // Flag indicates that density function table fY is valid
}


// ===========================================================================
StdPDensity::StdPDensity(istream& is)
{
  LoadState(is, (*this) );         // Load this object from the istream
}


// ===========================================================================
void StdPDensity::AddDataPoint(double x)
{
  // Reestimate the pdf given the new data point x

  if( (x >= fXMin) && (x < fXMax) )  // Ignore data points out of the domain
    {                                // of the density function
      int k = int( (x - fXMin)/fIntervalSize ); // Index of interval 
      fN[k] += 1.0;
      fNumberOfDataPoints++;
    }
  else
    fNumberOfDiscardedPoints++;    // Update counter if point is out of domain

  fUpdateFlag = 1;  // Flag indicates that fY needs to be updated to take
                    // into account the new data point 
}



// ===========================================================================
double StdPDensity::operator () (double x) 
{
  // Return the value of y = f(x)

  if(fUpdateFlag)  UpdateDensity(); 
  double y;
  if ( (x >= fXMin) && (x < fXMax) )  // We are within the range of the table
    {
      int i  = int( (x - fXMin)/fIntervalSize );  // Index of interval of x 
      y = fY[i];                                 // f(x) in the ith interval
    }
  else 
    y = 0.0;  // f(x) is set to 0.0 for x out of the range of the table

  return(y);
}


void StdPDensity::UpdateDensity(void)
{
  double scale = 1/(fNumberOfDataPoints * fIntervalSize);
  fY = fN*scale;     // Update the density 
  fUpdateFlag = 0;   // Reset flag
}



// ===========================================================================
void SaveState(ostream& os, StdPDensity& s)
{
  SaveState(os, s.fN);
  // The fY vector can be computed from fN so I don't save it
  SaveState(os, s.fXMin);
  SaveState(os, s.fXMax);
  SaveState(os, s.fIntervalSize);
  SaveState(os, s.fNumberOfDataPoints);
  SaveState(os, s.fNumberOfDiscardedPoints);
  os << "\n";

}


// ===========================================================================
void LoadState(istream& is, StdPDensity& s)
{
  LoadState(is, s.fN);
  LoadState(is, s.fXMin);
  LoadState(is, s.fXMax);
  LoadState(is, s.fIntervalSize);
  LoadState(is, s.fNumberOfDataPoints);
  LoadState(is, s.fNumberOfDiscardedPoints);

  s.fUpdateFlag = 1;
}


// ===========================================================================
void StdPDensity::SaveTable(ostream& os)
{
  // Save the density function so that it can be loaded as a TTable

  if(fUpdateFlag)  UpdateDensity(); 
  SaveState(os, fY );
  SaveState(os, fXMin );
  // The TTable stores samples at the endpoints of the intervals
  // whereas the StdPDensity stores one sample for each interval.
  // This implies that the TTable stores one extra datapoint. 
  // A simple fix is to shrink the range by one interval
  double xmax = fXMax - fIntervalSize;
  SaveState(os, xmax );
  SaveState(os, fIntervalSize );
  os << "\n";
}

















