//=========================================================================
// Copyright (c) 1994 Leslie Picardo
// All rights reserved
//=========================================================================
#include "StdMatrix.h"
#include "StdTypes.h"

//====================================================================
template <class ZZ>
StdMatrix<ZZ>::StdMatrix(int rows, int cols)
  : fX(0), fRows(rows), fCols(cols)
{
  assert( ((rows>0)&&(cols >0)) || ((rows==0)&&(cols==0)) );
  
  if( (fRows>0)&&(fCols>0) )
    {
      fX = new ZZ*[fRows]; assert(fX);
      for(int i=0; i<fRows; i++) { fX[i] = new ZZ[fCols]; assert(fX[i]); }
    }
}


//====================================================================
template <class ZZ>
StdMatrix<ZZ>::StdMatrix(const StdMatrix<ZZ>& source)
  :fX(0), fRows(source.fRows), fCols(source.fCols)
{
  if( (fRows>0)&&(fCols>0) )
    {
      fX = new ZZ*[fRows]; assert(fX);
      for(int i=0; i<fRows; i++) { fX[i] = new ZZ[fCols]; assert(fX[i]); }
    }

  for(int i=0; i<fRows; i++)          // Copy source matrix
    for(int j=0; j<fCols; j++)
      fX[i][j] = source.fX[i][j];
}


//====================================================================
template <class ZZ>
StdMatrix<ZZ>::~StdMatrix(void)
{
  for(int i=0; i<fRows; i++)  delete [] fX[i];
  delete [] fX;
}

//====================================================================
template <class ZZ>
const StdMatrix<ZZ>& StdMatrix<ZZ>::operator= (const StdMatrix<ZZ>& source)
{
  if(this != &source)  
    { 
      if( (fRows != source.fRows) || (fCols != source.fCols) )  
	Resize(source.fRows, source.fCols);
      
      for(int i=0; i<fRows; i++)      // Copy source matrix
	for(int j=0; j<fCols; j++)
	  fX[i][j] = source.fX[i][j];
    }
  return(*this);
}

//====================================================================
template <class ZZ>
void StdMatrix<ZZ>::Resize(int rows, int cols)
{
  assert( ((rows>0)&&(cols >0)) || ((rows==0)&&(cols==0)) );

  if ( (fRows == rows) && (fCols == cols) )  return;

  ZZ **temp = 0;                               // Make new matrix
  if( (rows>0)&&(cols>0) )
    {
      temp = new ZZ*[rows]; assert(temp);
      for(int i=0; i<rows; i++) { temp[i] = new ZZ[cols]; assert(temp[i]); }
    }

  for(int i=0; i<fRows; i++)  delete [] fX[i]; // Get rid of old matrix
  delete [] fX;

  fX = temp;       // Use the new matrix
  fRows = rows;
  fCols = cols;
}



//====================================================================
template <class ZZ>
void StdMatrix<ZZ>::ResizeSave(int rows, int cols)
{
  assert( ((rows>0)&&(cols >0)) || ((rows==0)&&(cols==0)) );

  if ( (fRows == rows) && (fCols == cols) )  return;

  ZZ **temp = 0;                             // Make new matrix
  if( (rows>0)&&(cols>0) )
    {
      temp = new ZZ*[rows]; assert(temp);
      for(int i=0; i<rows; i++) { temp[i] = new ZZ[cols]; assert(temp[i]); }
    }

  int minRows = ( (fRows < rows) ? fRows : rows);
  int minCols = ( (fCols < cols) ? fCols : cols);

  for(int i=0; i<minRows; i++)             // Copy the old matrix over
    for(int j=0; j<minCols; j++)
      temp[i][j] = fX[i][j];

  for(i=0; i<fRows; i++)  delete [] fX[i]; // Get rid of old matrix
  delete [] fX;

  fX = temp;                               // Use the new matrix
  fRows = rows;
  fCols = cols;
}


//====================================================================
template <class ZZ>
void StdMatrix<ZZ>::Print(ostream& os) const
{
  for(int i=0; i< fRows; i++)
    {
      for(int j=0; j< fCols; j++) os << fX[i][j] << " ";
      os << "\n";
    }
}





          //////  Overloaded non-member functions \\\\\\\


// These functions are fstream duplicates of the iostream functions
// defined below. They are necessary because the compiler will only 
// accept exact parameters matches during compilation.


//====================================================================
template <class ZZ>
ofstream& operator << (ofstream& ofs, const StdMatrix<ZZ>& m)
{
  ostream& os = ofs;
  os << m;
  return(ofs);
}

//====================================================================
template <class ZZ>
void LoadState(ifstream& ifs, StdMatrix<ZZ>& m)
{
  istream& is = ifs;
  LoadState(is, m);
}

//====================================================================
template <class ZZ>
void SaveState(ofstream& ofs, StdMatrix<ZZ>& m)
{
  ostream& os = ofs;
  SaveState(os, m);
}


//====================================================================
template <class ZZ>
ostream& operator << (ostream& os, const StdMatrix<ZZ>& m)
{
  m.Print(os);
  return(os);
}

//====================================================================
template <class ZZ>
void LoadState(istream& is, StdMatrix<ZZ>& m)
{
  int rows, cols;
  is >> rows >> cols;  
  assert( ((rows>0)&&(cols >0)) || ((rows==0)&&(cols==0)) );
  m.Resize(rows,cols);
  
  for(int i=0; i<m.Rows(); i++)          
      for(int j=0; j<m.Cols(); j++)
        LoadState(is, m[i][j]);
}

//====================================================================
template <class ZZ>
void SaveState(ostream& os, StdMatrix<ZZ>& m)
{
  os << m.Rows() << " " << m.Cols() << "\n";

  for(int i=0; i<m.Rows(); i++)
    {          
      for(int j=0; j<m.Cols(); j++)
        SaveState(os, m[i][j]);
      os << "\n";
    }
}














