// ==========================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
// ==========================================================================

#include <math.h>
#include "StdRandom.h"
#include "StdPi.h"
#include "StdSqr.h"
#include "StdTypes.h"
#include "EvPendulumPerf.h"


const int    cgNrParameters = 2;  // Number of parameters to search

const double cgStepSize  = 0.01;         // Integration stepsize
const double cgSimulationTime = 10.0;    // Length of simulation

double cgP;
//double cgI;
double cgD;


// ==========================================================================
EvPendulumPerformance::EvPendulumPerformance(void)
: fPendulum( ),                   // Make a pendulum
  fNetwork()    // Create the neural network
{
  fSearchDimension = cgNrParameters;
  fBestPerformance = -1e30;
}


// ==========================================================================
EvPendulumPerformance::EvPendulumPerformance(char *filename)
: fPendulum(), fNetwork()
{
  fSearchDimension = cgNrParameters;
  fBestPerformance = -1e30;
  ifstream ifs(filename);
  LoadState(ifs, cgP);
  //LoadState(ifs, cgI);
  LoadState(ifs, cgD);
}

// ==========================================================================
void EvPendulumPerformance::CopyParametersToNetwork( const StdArray<double>& x )
{
  // Search vector x is mapped onto the neural network parameters
  cgP = 10*x[0]; if (cgP > 20.0) cgP = 20.0;
  cgD = 10*x[1]; if (cgD > 20.0) cgD = 20.0; 
  //cgI = x[2];

}
 

// ==========================================================================
double EvPendulumPerformance::Evaluate(const StdArray<double>& x)
{
  CopyParametersToNetwork(x);
  
  double performanceValue = Simulate( cgSimulationTime );
  if( performanceValue > fBestPerformance )
    {
      ofstream ofs("network.best");
      SaveState( ofs , cgP );
      //SaveState( ofs , cgI );
      SaveState( ofs , cgD );
      ofs << "\n" << performanceValue << ": performance \n"; 

      fBestPerformance = performanceValue;
    }

  return( performanceValue );
}





// ==========================================================================
double EvPendulumPerformance::Simulate(double simulationTime)
{
  // Simulate the pendulum network system  
  double performanceValue = 0.0;

  //Initialize the pendulum: 0.1 rad = 6 degs
  fPendulum.Reset( 0.1, 0.0, 0.0, 0.0);  

  for (double time = 0.0; time < simulationTime; time += cgStepSize)
    {
      Step( cgStepSize );
      // compute performance
      performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
      //performanceValue += 0.005*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;
    }

  return( performanceValue ); // Return performance
}


// ==========================================================================
void  EvPendulumPerformance::Step(double stepSize)
{
  double force = cgP*fPendulum.Theta() + cgD* fPendulum.ThetaDot();
  fPendulum.Step( force, stepSize ); // Integrate pendulum
}













