// ==========================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
// ==========================================================================
#include <math.h>
#include "StdRandom.h"
#include "StdPi.h"
#include "StdSqr.h"
#include "EvPendulumPerf.h"


const int    cgN = 5; 
const int    cgNrNetworkNodes = 2*cgN; 
const int    cgNrParameters = cgN*(3+cgN) - 1; 
//const int    cgNrParameters = cgN*(4+cgN) - 1;  
//const int    cgNrParameters = 4*cgN-2;  // Number of parameters to search
//const int    cgNrParameters = 3*cgN-2;  // Number of parameters to search

const double cgForceGain = 40.0;         // Gain for the force
// If the pendulum is kept at theta = 45degrees, thetadot = 0.0, then the 
// force needed to keep thetaddot = 0.0 is computed as f = (M+m)*g*tan(theta).
// With M = m = 1kg this works out to f = 20N. 

const double cgStepSize  = 0.01;         // Integration stepsize
const double cgSimulationTime = 10.0;    // Length of simulation
//const double cgInitialTheta   = cPI;     // Initial pendulum theta

//const double cgTauMap     = 1.0;  // Maps [-1,1] to timeconstant range 
const double cgBiasMap    = 5.0;  // Maps [-1,1] to bias range [-5,5]
const double cgInScaleMap = 5.0;  // Maps [-1,1] to input_scale range [-5,5]
const double cgWeightMap  = 5.0;  // Maps [-1,1] to weight range [-5,5]

// ==========================================================================
EvPendulumPerformance::EvPendulumPerformance(void)
: fPendulum( ),                   // Make a pendulum
  fNetwork( cgNrNetworkNodes )    // Create the neural network
{
  fSearchDimension = cgNrParameters;
  fBestPerformance = -1e30;

  for(int i = 0; i < cgN; i++)  // Set the network connections
    {
      for(int j = 0; j < cgN; j++)
	{
	  fNetwork.Connect(i,j);      // left side fully connected
	  fNetwork.Connect(j,i); 
	  fNetwork.Connect(cgN+i, cgN+j); // right side fully connected
	  fNetwork.Connect(cgN+j, cgN+i);
	}
      fNetwork.Connect(i, i+cgN);       // Connect corresponding nodes
      fNetwork.Connect(i+cgN, i);       // on left and right side
    }
}


#if 0
// ==========================================================================
EvPendulumPerformance::EvPendulumPerformance(void)
: fPendulum( ),                   // Make a pendulum
  fNetwork( cgNrNetworkNodes )    // Create the neural network
{
  fSearchDimension = cgNrParameters;
  fBestPerformance = -1e30;

  for(int i = 0; i < (cgN-1); i++)  // Set the feedforward network connections
    {
      fNetwork.Connect( cgN-1, i); 
      fNetwork.Connect( 2*cgN-1, cgN+i); 
    }
}
#endif

// ==========================================================================
EvPendulumPerformance::EvPendulumPerformance(char *filename)
: fPendulum(), fNetwork()
{
  fSearchDimension = cgNrParameters;
  fBestPerformance = -1e30;
  ifstream ifs(filename);
  LoadState(ifs, fNetwork);
}


// ==========================================================================
void EvPendulumPerformance::CopyParametersToNetwork( const StdArray<double>& x )
{
  // Search vector x is mapped onto the neural network parameters
  int i, j, k; 

  k = 0; //index to go down the search vector x
  
//  for (i = 0; i < cgN; i++) 
//    { fNetwork.TimeConstant(i) = fNetwork.TimeConstant(i+cgN) 
//                                 = fabs( cgTauMap * x[k++] ) + 0.1; }
  
  for (i = 0; i < cgN; i++) 
    { fNetwork.TimeConstant(i) = fNetwork.TimeConstant(i+cgN) = 0.1; }
  
  for (i = 0; i < cgN; i++) 
    { fNetwork.Bias(i) = fNetwork.Bias(i+cgN) = cgBiasMap * x[k++]; }
  
  for (i = 0; i < (cgN-1); i++) 
    { fNetwork.InputScale(i)=fNetwork.InputScale(i+cgN) = cgInScaleMap * x[k++];}
  
  fNetwork.InputScale(cgN-1) = fNetwork.InputScale(2*cgN-1) = 0.0;
  
  for(i = 0; i < cgN; i++)
    {
      for(j = 0; j < cgN; j++)
	fNetwork.Weight(i,j) = fNetwork.Weight(i+cgN,j+cgN) = cgWeightMap * x[k++];
      fNetwork.Weight(i, i+cgN) = fNetwork.Weight(i+cgN,i) = cgWeightMap * x[k++];
    }
}


#if 0
// ==========================================================================
void EvPendulumPerformance::CopyParametersToNetwork( const StdArray<double>& x )
{
  // Search vector x is mapped onto the neural network parameters
  int i, j, k; 

  k = 0; // index to go down the search vector x

//  for (i = 0; i < cgN; i++) 
//    { fNetwork.TimeConstant(i) = fNetwork.TimeConstant(i+cgN) 
//                                 = fabs( cgTauMap * x[k++] ) + 0.1; }
  
  for (i = 0; i < cgN; i++) 
    { fNetwork.TimeConstant(i) = fNetwork.TimeConstant(i+cgN) =  0.1; }

  for (i = 0; i < cgN; i++) 
    { fNetwork.Bias(i) = fNetwork.Bias(i+cgN) = cgBiasMap * x[k++]; }
  
  for (i = 0; i < (cgN-1); i++) 
    { fNetwork.InputScale(i)=fNetwork.InputScale(i+cgN) = cgInScaleMap * x[k++];}
  
  fNetwork.InputScale(cgN-1) = fNetwork.InputScale(2*cgN-1) = 0.0;
  
  for(i = 0; i < (cgN-1); i++)
    {
      fNetwork.Weight(cgN-1, i) = fNetwork.Weight(2*cgN-1,i+cgN) = cgWeightMap * x[k++];
    }
}
#endif
 

// ==========================================================================
double EvPendulumPerformance::Evaluate(const StdArray<double>& x)
{
  CopyParametersToNetwork(x);
  
  double performanceValue = Simulate( cgSimulationTime );
  if( performanceValue > fBestPerformance )
    {
      ofstream ofs("network.best");
      SaveState( ofs , fNetwork );
      ofs << "\n" << performanceValue << ": performance \n"; 

      fBestPerformance = performanceValue;
    }

  return( performanceValue );
}





// ==========================================================================
double EvPendulumPerformance::Simulate(double simulationTime)
{
  // Simulate the pendulum network system  
  double performanceValue = 0.0;

  fNetwork.Reset();	// Initialize the network                    
  for (double t = 0.0; t < 1.0; t += cgStepSize)
      fNetwork.Step( cgStepSize );         // Let the transients die out

  //Initialize the pendulum: 0.1 rad = 6 degs
  //fPendulum.Reset( UniformRandom(0.01, +0.05), 0.0, 0.0, 0.0);  
  fPendulum.Reset( cPI, 0.0, 0.0, 0.0);  

  for (double time = 0.0; time < simulationTime; time += cgStepSize)
    {
      Step( cgStepSize );
      // compute performance
      performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
      performanceValue += 0.1*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;
    }

  return( performanceValue ); // Return performance
}

#if 0
// ==========================================================================
void  EvPendulumPerformance::Step(double stepSize)
{
  fNetwork.ExternalInput(0) = fPendulum.Theta();   // Inputs to left side 
  fNetwork.ExternalInput(1) = fPendulum.ThetaDot();
  
  fNetwork.ExternalInput(5) = -fPendulum.Theta();  // Inputs to right side 
  fNetwork.ExternalInput(6) = -fPendulum.ThetaDot();
  
  fNetwork.Step( stepSize );         // Integrate network
  double force =  cgForceGain * ( fNetwork.Output(cgN-1) - fNetwork.Output(2*cgN-1) );
  fPendulum.Step( force, stepSize ); // Integrate pendulum
}
#endif

#if 0
// ==========================================================================
void  EvPendulumPerformance::Step(double stepSize)
{
  fNetwork.ExternalInput(0) = fPendulum.Theta();   // Inputs to left side 
  fNetwork.ExternalInput(1) = fPendulum.ThetaDot();
  fNetwork.ExternalInput(2) = fPendulum.X();
  fNetwork.ExternalInput(3) = fPendulum.XDot();
  
  fNetwork.ExternalInput(5) = -fPendulum.Theta();  // Inputs to right side 
  fNetwork.ExternalInput(6) = -fPendulum.ThetaDot();
  fNetwork.ExternalInput(7) = -fPendulum.X();
  fNetwork.ExternalInput(8) = -fPendulum.XDot();
  
  fNetwork.Step( stepSize );         // Integrate network
  double force =  cgForceGain * ( fNetwork.Output(4) - fNetwork.Output(9) );
  fPendulum.Step( force, stepSize ); // Integrate pendulum
}
#endif


// ==========================================================================
void  EvPendulumPerformance::Step(double stepSize)
{
  double sinTheta =  sin( fPendulum.Theta() );
  double cosTheta =  cos( fPendulum.Theta() );
  // Note sin(-theta) = -sin(theta) 
  //      cos(-theta) =  cos(theta)

  fNetwork.ExternalInput(0) = sinTheta;	// Inputs to left side 
  fNetwork.ExternalInput(1) = cosTheta; 
  fNetwork.ExternalInput(2) = fPendulum.ThetaDot();
  fNetwork.ExternalInput(3) = fPendulum.X();
                              //fPendulum.XDot();
  
  fNetwork.ExternalInput(5) = -( sinTheta ); // Inputs to right side
  fNetwork.ExternalInput(6) = cosTheta ; 
  fNetwork.ExternalInput(7) = -( fPendulum.ThetaDot() );
  fNetwork.ExternalInput(8) = -( fPendulum.X() ); 
                               //-( fPendulum.XDot() );
  
  fNetwork.Step( stepSize ); // Integrate network
  double force =  cgForceGain * ( fNetwork.Output(4) - fNetwork.Output(9) );
  fPendulum.Step( force, stepSize ); // Integrate pendulum
}



#if 0
// Old Simulate function
// ==========================================================================
double EvPendulumPerformance::Simulate(double simulationTime)
{
  // Simulate the pendulum network system  
  double performanceValue = 0.0;

  fNetwork.Reset();	// Initialize the network                    
  // Initialize the pendulum: 0.1 rad = 6 degs

  //fPendulum.Reset( UniformRandom(cPI-0.2, cPI+0.2), 0.0, 0.0, 0.0);  
  
  for (double time = 0.0; time < simulationTime/2.0; time += cgStepSize)
    {
      Step( cgStepSize );
      // compute performance
      performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
      performanceValue += 0.005*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;
    }
  

  fNetwork.Reset();	// Initialize the network                    
  // Initialize the pendulum: 0.1 rad = 6 degs
  fPendulum.Reset( UniformRandom(-0.2, +0.2), 0.0, 0.0, 0.0);  
  
  for (time = 0.0; time < simulationTime/2.0; time += cgStepSize)
    {
      Step( cgStepSize );
      // compute performance
      performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
      performanceValue += 0.005*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;
    }

  return( performanceValue ); // Return performance
}

#endif


#if 0
      // With this performance function  randomwalk and randomsampling
      // cannot do any better than keep the pole hanging down. It looks 
      // like there are large regions of maxima from which it is too 
      // difficult to get out off. Maybe a simulated annealing algorithm would
      // work best here 
      //performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
      //performanceValue += Sqr( 2*exp(-fabs(fPendulum.X()))    ) * cgStepSize;
      
    //Pendulum stays up but then drifts away from x=0.0
    //performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
    //performanceValue += 0.001*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;

    // Pendulum stays around x=0.0 but doesn't go up
    //performanceValue += Sqr( cos( fPendulum.Theta() ) + 1.0 ) * cgStepSize;
    //performanceValue += 0.01*Sqr( 2*exp(-fabs(fPendulum.X()))) * cgStepSize;

#endif


















