//====================================================================
// Copyright (c) 1995 Leslie Picardo.  All rights reserved
//====================================================================

#include "EvBitGenome.h"
#include "StdRandom.h"
#include "StdSwap.h"

//====================================================================
EvBitGenome::EvBitGenome(int numberOfBits)
  :StdBitString( numberOfBits )
{ }


//====================================================================
EvBitGenome::EvBitGenome( const StdBitString& source)
  :StdBitString(source)
{ }


//====================================================================
void EvBitGenome::Randomize(void)
{
  // For each bit in the genome flip a coin (probability 1/2) 
  // and set the bit to 1 or 0 

  for(int b=0; b<NumberOfBits(); b++)   // Consider the b'th bit
    if( Probability(0.5) ) Set(b); else Clear(b); 
}


//====================================================================
void  EvBitGenome::Mutate(float p)
{
  // Invert each bit of the genome with probability p 

  for(int b=0; b<NumberOfBits(); b++)  // For each bit in the genome
     if( Probability(p) )  Invert(b);

  //... Note that instead of bit inversion we could define mutation 
  //... to be the replacement of a bit by a random bit
}



//===========================================================================
void OnePointCrossover(const EvBitGenome& p1, const EvBitGenome& p2,
                             EvBitGenome& c1,       EvBitGenome& c2  )
{
  // Take two parent genomes p1 & p2 and perform single point crossover
  // to create two new child genomes c1 & c2.

  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );
  
  // Generate a random crossover point. RandomInt(x) returns [0,x-1]
  int crossoverPoint = RandomInt( minSize );
      
  // Now create the children 
  
  c1 = p1;  // First make clones
  c2 = p2;
  // Now swap the portion of the genomes marked for crossing
  for(int k=0; k<=crossoverPoint; k++)  
    { 
      if(p2[k]) c1.Set(k); else c1.Clear(k);  // =>  c1[k] = p2[k]
      if(p1[k]) c2.Set(k); else c2.Clear(k);  // =>  c2[k] = p1[k]
    }
}




//===========================================================================
void TwoPointCrossover(const EvBitGenome& p1, const EvBitGenome& p2,
                             EvBitGenome& c1,       EvBitGenome& c2 )
{
  // Take two parent genomes p1 & p2 and perform two point crossover
  // to create two new child genomes c1 & c2.

  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );

  // Generate two random crossover points. RandomInt(x) returns [0,x-1]
  int crossoverPoint1 = RandomInt(  minSize );
  int crossoverPoint2 = RandomInt(  minSize );

  // Ensure that the first crossover point is before the second
  if(crossoverPoint1 > crossoverPoint2) Swap(crossoverPoint1, crossoverPoint2);

  // Now create the children 

  c1 = p1;  // First make clones
  c2 = p2;
  // Now swap the portion of the genomes marked for crossing
  for(int k = crossoverPoint1; k <= crossoverPoint2; k++)
    { 
      if(p2[k]) c1.Set(k); else c1.Clear(k);   // => c1[k] = p2[k]
      if(p1[k]) c2.Set(k); else c2.Clear(k);   // => c2[k] = p1[k]
    }
}



//===========================================================================
void
UniformCrossover(const EvBitGenome& p1, const EvBitGenome& p2, 
                       EvBitGenome& c1,       EvBitGenome& c2,  float p)
{
  // Take two parent genomes p1 & p2 and perform uniform crossover
  // to create two new child genomes c1 & c2.
  // p is the probability of crossing a gene.
      
  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );

  c1 = p1;  // First make clones
  c2 = p2;

  for(int k=0; k<minSize; k++)  // Consider each gene
    {
      if(Probability(p))  // with probability p
	{
	  if(p2[k]) c1.Set(k); else c1.Clear(k); // Swap the genes
	  if(p1[k]) c2.Set(k); else c2.Clear(k);
	}
    }
}



//====================================================================
ostream& operator << (ostream& os, const EvBitGenome& g)
{
  g.Print(os);  // print the bits
  return(os);
}

//====================================================================
void LoadState(istream& is, EvBitGenome& g)
{
  StdBitString& base = g;  // Convert to base class
  LoadState(is, base);     // Call base class equivalent
}

//====================================================================
void SaveState(ostream& os, EvBitGenome& g)
{
  StdBitString& base = g;  // Convert to base class
  SaveState(os, base);     // Call base class equivalent 
}










