//============================================================================
// Copyright (c) 1995 Leslie  Picardo. All rights reserved
//============================================================================

#include "EvRealGenome.h"
#include "StdRandom.h"
#include "StdSwap.h"       // def'n of Swap(int, int)

//===========================================================================
EvRealGenome::EvRealGenome(int size)
  :StdVector(size)                     // Make vector of genes
{  }


//===========================================================================
void EvRealGenome::Randomize(double min, double max)
{
  // Randomize the genes in the genome within the [min, max] range
  for(int i=0; i<fSize; i++) 
    fX[i] = UniformRandom(min, max);
}


//===========================================================================
void  EvRealGenome::Mutate(double mutationVariance, double mutationMean)
{
  // Add a random vector to the genome

  // The length of the vector   
  double magnitude = GaussianRandom(mutationMean, mutationVariance);
  *this += magnitude * RandomUnitVector( fSize ); // Add a random vector
}




//===========================================================================
void OnePointCrossover(const EvRealGenome& p1, const EvRealGenome& p2,
                             EvRealGenome& c1,       EvRealGenome& c2  )
{
  // Take two parent genomes p1 & p2 and perform single point crossover
  // to generate two child genomes c1 & c2.

  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );
  
  // Generate a random crossover point. RandomInt(x) returns [0,x-1]
  int crossoverPoint = RandomInt( minSize );
      
  // Now create the children 
  
  c1 = p1;  // First make clones
  c2 = p2;
  // Now swap the portion of the genomes marked for crossing
  for(int k=0; k<=crossoverPoint; k++)  
    { 
      c1[k] = p2[k];  
      c2[k] = p1[k]; 
    }
}




//===========================================================================
void TwoPointCrossover(const EvRealGenome& p1, const EvRealGenome& p2,
                             EvRealGenome& c1,       EvRealGenome& c2 )
{
  // Take two parent genomes p1 & p2 and perform two point crossover
  // to generate two child genomes c1 & c2.

  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );

  // Generate two random crossover points. RandomInt(x) returns [0,x-1]
  int crossoverPoint1 = RandomInt(  minSize );
  int crossoverPoint2 = RandomInt(  minSize );

  // Ensure that the first crossover point is before the second
  if(crossoverPoint1 > crossoverPoint2) Swap(crossoverPoint1, crossoverPoint2);

  // Now create the children 

  c1 = p1;  // First make clones
  c2 = p2;
  // Now swap the portion of the genomes marked for crossing
  for(int k = crossoverPoint1; k <= crossoverPoint2; k++)
    { 
      c1[k] = p2[k];  
      c2[k] = p1[k]; 
    }
}



//===========================================================================
void  UniformCrossover(const EvRealGenome& p1, const EvRealGenome& p2, 
                             EvRealGenome& c1,       EvRealGenome& c2,  
                       double p)
{
  // Take two parent genomes p1 & p2 and perform uniform crossover
  // to generate two child genomes c1 & c2.
  // p is the probability of crossing a gene.
      
  // Get the size of the smaller genome
  int minSize = ( (p1.Size() < p2.Size()) ? p1.Size() : p2.Size() );

  c1 = p1;  // First make clones
  c2 = p2;

  for(int k=0; k<minSize; k++)  // Consider each gene
    {
      if(Probability(p))  // with probability p
	{
	  c1[k] = p2[k];
	  c2[k] = p1[k];  // Swap the genes
	}
    }
}


//===========================================================================
ostream& operator << (ostream& os, const EvRealGenome& g)
{
  g.Print(os);   // print the genome
  return(os);
}

//===========================================================================
void LoadState(istream& is, EvRealGenome& g)
{
  StdVector& base = g;   // Convert to base class
  LoadState(is, base); // Call function on base
}


//===========================================================================
void SaveState(ostream& os, EvRealGenome& g)
{
  StdVector& base = g;   // Convert to base class
  SaveState(os, base); // Call function on base
}



#if 0     

 -== Defunct code =- 

//===========================================================================
void  Mutate(EvRealGenome& genome, double mutationVariance, 
                                                         double mutationMean)
{
  // Add a random vector to the genome
  genome.Mutate(mutationVariance, mutationMean);
}


#endif










