//===========================================================================
// Copyright (c) 1995 Leslie Picardo. All rights reserved
//===========================================================================

#include <math.h>

#include "EvPopulation.h"
#include "StdRandom.h"
#include "StdSqr.h"


//===========================================================================
template<class ZZ> 
EvPopulation<ZZ>::EvPopulation(int populationSize)
  :StdArray<ZZ>(populationSize)                       // Create array of ZZ's
{  }


//===========================================================================
template<class ZZ> 
void EvPopulation<ZZ>::Print(ostream& os) const
{
  for(int k=0; k<fSize;k++)               // Print each individual on a line
    os << k << " " << fX[k] << "\n";
}

//===========================================================================
template<class ZZ> 
void EvPopulation<ZZ>::SelectFitnessProportionate( EvPopulation<ZZ>& s ) 
{
  // Replace members of this population with individuals from the <s>ource
  // population using fitness proportionate selection.
  // This function assumes that the individual performance values are 
  // positive. Negative performance values are treated as zero.
  // No translation or scaling of performance is done.

  int i, k;
  int sSize = s.Size();    // size of source population

  // Compute fitness of the ith individual in the population
  for(i=0; i<sSize; i++)
    s[i].Fitness() = ((s[i].Performance() > 0) ? s[i].Performance() : 0.0);  
  
  // Compute cummulative Fitness 
  for(i=1; i<sSize; i++)  s[i].Fitness() += s[i-1].Fitness();
  
  // Select individuals from source into this population
  double rV;                            // A random throw on the roulette wheel
  double maxV = s[sSize-1].Fitness();   // Size of the roulette wheel
  for(i=0; i<fSize; i++)  // For each member of this population
    {
      // Spin the roulette wheel 
      rV = UniformRandom(0.0, 1.0) * maxV;

      // Get the selected individual index
      for (k=0; k<sSize; k++) { if ( rV <= s[k].Fitness() ) break; }

      fX[i] = s[k];  // Copy the selected individual
    }
}


//===========================================================================
template<class ZZ> 
EvPopulation<ZZ>  SelectFitnessProportionate( EvPopulation<ZZ>& s)
{
  // Returns a population consisting of members that have been selected
  // from the <s>ource population. The returned population is the same
  // size as the <s>ource population. 
  // This function assumes that the individual performance values are 
  // positive. Negative performance values are treated as zero.
  // No translation or scaling of performance is done.

  EvPopulation<ZZ>  temp(s.Size());     // Create a temp population 
  temp.SelectFitnessProportionate(s);  // Select members of s
  return (temp);                       // return the temp population
}


//===========================================================================
template<class ZZ> ostream& operator << (ostream& os, const EvPopulation<ZZ>& p)
{
  p.Print(os);
  return(os);
}

//===========================================================================
template<class ZZ> void LoadState(ifstream& ifs, EvPopulation<ZZ>& p)
{ 
  istream& is = ifs;  // Convert to an istream
  LoadState(is, p);   // Call istream equivalent template function
}

//===========================================================================
template<class ZZ> void SaveState(ofstream& ofs, EvPopulation<ZZ>& p)
{
  ostream& os = ofs; // Convert to an ostream
  SaveState(os, p);  // Call ostream equivalent template function
}


//===========================================================================
template<class ZZ> void LoadState(istream& is,  EvPopulation<ZZ>& p)
{
  StdArray<ZZ>& base = p;   // Refer to base class
  LoadState(is, base);    // Load the base 
}


//===========================================================================
template<class ZZ> void SaveState(ostream& os,  EvPopulation<ZZ>& p)
{
  StdArray<ZZ>& base = p;   // Refer to base class
  SaveState(os, base);  // Save the base 
}

// These functions have to be redeclared here or g++ will not compile
// the LoadState and SaveState templates. This is a compiler fix
template<class ZZ> void LoadState(istream&, StdArray<ZZ>&);
template<class ZZ> void SaveState(ostream&, StdArray<ZZ>&);


//===========================================================================
template<class ZZ> 
void EvPopulation<ZZ>::Randomize(void)
{
  for(int k=0; k<fSize;k++)       // For each member of the population
    fX[k].Randomize();            // Create a random individual
}


//===========================================================================
template<class ZZ> 
double EvPopulation<ZZ>::Average(void)
{
  double average = 0.0;               // return the average performance
  for(int k=0; k<fSize;k++)           // value of individuals in the population
    average += fX[k].Performance(); 
  return( average/fSize );
}

//===========================================================================
template<class ZZ> 
double EvPopulation<ZZ>::StdDev(void)
{
  double mean = Average(); // Get the average performance value
  double stddev = 0.0;

  for(int k=0; k<fSize;k++)           // value of individuals in the population
    stddev += Sqr( mean - fX[k].Performance() ); 

  return( sqrt(stddev/fSize) );
}







//===========================================================================
template<class ZZ> 
double EvPopulation<ZZ>::Best(void)
{
  double best = -1e30;                // return the best performance value
  for(int k=0; k<fSize;k++)           // of individuals in the population
    if( fX[k].Performance() > best ) best = fX[k].Performance();
  return( best );
}


//===========================================================================
template<class ZZ> 
void EvPopulation<ZZ>::Evaluate(void)
{
  for(int k=0; k<fSize;k++)       // For each member of the population
    fX[k].Evaluate();             // Evaluate the bugger
}



















