/*****************************************************************************
STATISTICAL LINE-LINKER (ANSI C)
================================
 THIS VERSION OF THE LINE-LINKER HAS BEEN DIVIDED INTO 3 MODULES:
 LINKER, GLUER AND SUPPORT. THIS VERSION IS NOW MAINTAINED AS THE
 CURRENT VERSION.


ALGORITHM:
==========
      THE BASIC IDEA IS: LINE LINKING IS PERFORMED IN TWO STEPS.


I) FIRST STEP: CHAIN LINKING AND LINE FITTING
=============================================

1) What it does:
----------------

  - Takes an image, segments it in
    chains: a chain is a sequence of pixels that is thin.
    By definition, a pixel is in a chain if and only if it 
    is connected to two neighbors.
  - Creates an image that contains the chains and 
    the connections. By definition, a connection is a group
    of pixels that joins more than two neighbors. 
    The reason for the definition
    of chains/connections is that a chain pixel can be
    followed while a connection pixel can't.
  - Uses the chain description to fit least-squares lines, under
    assumption of Gaussian noise.

  Description of chains is stored in structure CHAIN;
  description of connections is stored in structure
  CONNECTION. The vector of CHAINS and the vector of CONNECTIONS is
  generated on the fly.

  The image IMAGE_OUT contains a 
  summary of all results through the values of pixels.
  Pixels on the background receive value 0, while
  pixels in chains receive POSITIVE labels and 
  pixels in connections receive NEGATIVE labels.
  The positive labels index the pixels in the CHAIN structure;
  the negative labels index the pixels in the CONNECTION structure.

  The algorithm assumes that a thinning operation was
  already performed, otherwise lots and lots of 
  connections are created with few meaninful chains
  among them.

2) Description of the algorithm:
--------------------------------
 
  For every pixel (i,j) that has not been labelled yet:
 
  - If the pixel has no unlabelled neighbor (it is 
    alone in the image):
   
    - if the pixel has no connection-type neighbor, then
      it is a chain of one pixel;
    - if the pixel has one connection-type neighbor, then
      it is a chain of one pixel attached to the 
      connection-type neighbor; 
    - if the pixel has two connection-type neighbors, then
      it is a chain of one pixel attached to the two 
      connection-type neighbors;
    - if the pixel has more than two connection-type 
      neighbors, then it is just a connection that connects
      all the neighbors.

  - If the pixel has one unlabelled neighbor: 

    - if the pixel has no connection-type neighbor, then it
      is the beginning of a chain; the chain must be 
      followed until a connection is created or found;
    - if the pixel has one connection-type neighbor, then
      it is the beginning of a chain that is attached to
      the connection-type neighbor; the chain must be
      followed until a connection is created or found;
    - if the pixel has more than one connection-type 
      neighbor, then the pixel is a connection between the
      connection neighbors.

  - If the pixel has two unlabelled neighbors:

    - if the pixel has no connection-type neighbor, then it is a
      pixel in the middle of a chain; start a chain and follow
      it in both directions;
    - if the pixel has at least one connection-type neighbor, then
      create a connection between the neighbor connections.
     
  - If the pixel has more than two unlabelled pixels, then 
    create a connection that connects all connections around 
    (if there is one).

3) Additional Considerations:
-----------------------------

 As the chains are followed and formed, two things are done:

 - Lines are fit to the chains. The fit is a least-squares type of
    fit, with parameters and residuals estimated recursively 
    (based on Bar-Shalom/Fortmann exposition). A chi-square test
    is used to break lines inside the chains. At the end, each
    chain contains a linked list of all the lines that are included
    in the chain. This linked list has pointers to the first and
    last lines, which correspond to the lines that begin and end
    the chain respectively.
    
    In order to understand how lines are represented, it is 
    important to realize that the coordinate system is as follows:
      * Origin: left top pixel in the image.
      * X axis: vertical, from origin down.
      * Y axis: horizontal, from origin to right.
    This coordinate system mimics the indexes of images in the 
    GIL library.

    Two parameters are crucial to the algorithm:
      * VARIANCE_OF_PIXEL: is the variance of the noise around each
        pixel; variance in x and y directions are considered equal.
      * SIGNIFICANCE_LEVEL_BREAK_LINES: gives the level for the
        test that decides whether a line should be broken or not.
	This variable is an integer valued {0,...,6}. The meaning is:
	 > Value 0: level of 0.01
	 > Value 1: level of 0.05
	 > Value 2: level of 0.1
 	 > Value 3: level of 0.15
	 > Value 4: level of 0.2
	 > Value 5: level of 0.25
	 > Value 6: level of 0.35

 - The program creates a second output
     image, IMAGE_CHAINS, with chain-codes for all pixels. This 
     image may be used if there is a need for intensive processing
     that involves following chains again and again; in this case,
     it is necessary to acess one pixel of IMAGE_CHAINS instead of
     a block of nine pixels from IMAGE_IN. 
     
     The chain-code mechanism is fairly simple: 
      * the chain-code for a pixel is an unsigned integer (a byte
       of 8 bits)
      * for each one of the eight pixels (surrounding the given
       pixel) that are occupied, the chain-code receives has 
       a bit set to one. The order is:
       [SOUTH_EAST,SOUTH_WEST,NORTH_EAST,NORTH_WEST,EAST,WEST,SOUTH,NORTH] 
      * With the chain-code, it is obvious to obtain the following
       information: 
       > Whether the pixel is a chain or connection: using the 
          function query_chain_connection; basically, if more than
	  two bits in a chain-code are equal to 1, then the
	  pixel is a chain; otherwise it is a connection.
       > Given a chain-code, find the next pixel from the given
          pixel; then this information may be erased from the 
	  chain-code by using an XOR or AND operation.      


II) SECOND STEP: CONNECTING NEIGHBOR CHAINS
===========================================

1) What it does:
----------------

 - Picks up the structures CHAINS and CONNECTIONS;
 - For each chain with length larger than 1:
   * Verifies the start of the chain. If the chain has a zero 
      chain->second_chain_start (it means the chain is not connected to
      any other chain), then:
       > If the chain starts in a connection, tries to connect chain
        with all the other chains in connection.
       > If the chain starts alone, searches a region around the start 
        of the chain. If any chain is found in this region, creates  a
	ghost-connection. A ghost-connection is identical to a 
	connection (also stored in CONNECTIONS), but a ghost-connection
	has size 0: no pixel in the image corresponds to a ghost-connection.
	A ghost-connection is used only to connect chains through gaps.
	Then it tries to connect the initial chain to  any chain found in this
	region.
   * Verifies the end of the chain. If the chain has a zero 
      chain->second_chain_end (it means the chain is not connected to
      any other chain), then:
       > If the chain ends in a connection, tries to connect chain
        with all the other chains in connection.
       > If the chain ends alone, searches a region around the end 
        of the chain. If any chain is found in this region, creates  a
	ghost-connection. A ghost-connection is identical to a 
	connection (also stored in CONNECTIONS), but a ghost-connection
	has size 0: no pixel in the image corresponds to a ghost-connection.
	A ghost-connection is used only to connect chains through gaps.
	Then it tries to connect the initial chain to  any chain found in this
	region.

 As a result, all chains are checked (start and end) for possible connections.
 It is important to understand the two different actions taken at the 
 second step:
      a) First action: jumping gaps.
         Many chains are already connected by the end of the first step.
         The second sted enlarges this set by trying to jump gaps between
	 chains (creating ghost-connections).
      b) Second action: detecting similar chains.
         Every connection connects more than two chains; it may be that
	 two of the chains in a connection are very similar and should be
	 considered a unique straight line. The algorithm links them (but
	 does not try to fit any new line, since there is a connection
	 between the chains and any fitting may be erroneous).
       
2) Description of the algorithm:
--------------------------------

 The algorithm looks at every chain, first at its start, then at its 
 end. In both cases the algorithm proceeds in the same way:

 - The second_chain_start (or second_chain_end) variable in the chain
   must be zero. 
   Extremities of chains that have second_chain_start (or second_chain_end)
   different from zero are NEVER considered again by the algorithm. This
   is clearly sub-optimal, since a chain may be connectable to more
   than one chain and the algorithm always locks in the first option it
   finds. But this sub-optimality is not likely to produce wrong results
   in almost any situation, and the optimal solution (looking for all
   possibilities) is highly non-efficient computationally.

 - There are two possibilities for the extremity of the chain:

   - It is connected to a connection. In this case all chains that are
     connected to this connection are analysed. If any chain produces
     a better AIC, both chains are put together.

   - It is not connected to a connection. In this case a region is explored

 - In case a chain is connected to another chain, the other chain is now
   analysed by the same process; the idea is that the algorithm tries to
   extend the checking as much as possible. This is done for a simple
   reason: if someone wants to progressively fit lines to sequences of
   chains, it can be done easily. The idea would be make sequences of 
   lines "stronger" as they proceed; it is not done now. 


3) Additional Considerations:
-----------------------------

 Here is how two neighbor chains are considered "connected":
 - The relevant lines that belong to the chains are picked up. If the
   chains are connected in their ends, the last line is picked up; if
   the chains are connected in their starts, the first line is picked up.
 - A line is fit to all the points that belong to both of these lines.
 - The Aikake's information criterium is evaluated for both cases:
   * First case: points are fit by two lines (4 parameters are estimated).
   * Second case: points are fit by one line (2 parameters are estimated).
   The AIC is given by:
       AIC = - 2 (likelihood of data under model) + 
               2 (number of parameters estimated in the model)
   which in a Gaussian model gives:
       AIC = SSe + n log(2\pi) + number of parameters (SSe = Square Errors).
 - The case that has the minimum AIC is choosen.
 When a connection contains several chains, then all the combinations of 
 one chain with the other chains is tested; the combination that has
 minimum AIC is chosen.

 Two parameters define the behavior of the function: 
  - VARIANCE_OF_PIXEL: as in the first step, it gives the variance of the
    Gaussian error of the pixels in x and y coordinates.
  - SIZE_EXPLORATION: it defines the size of the region that is explored 
    around the extremity of a chain (when the chain ends alone). The region is
    a square with sides (2 * SIZE_EXPLORATION + 1); the extremity of the chain 
    is in the center of the square.

III) DATA STRUCTURES
====================

1) Images

 Three images are used: 
 - IMAGE_IN: A GIL integer image. Contains the initial image. 
   Background points must be zero;
   non-background points (the edgels) must be non-zero (no other constraint).
 - IMAGE_OUT: A GIL integer image. It is cleared at the beginning.
   At the end, contain information about chains and connections 
   (background pixels are zero).
 - IMAGE_CHAINS: A GIL integer image. It is cleared at the beginning.
   At the end, contain information about chain-codes.
   (background pixels are zero).

2) Structure Elements

 This is a general purpose structure that can be thought as a 
 vector for general types. It contains a TOTAL and a MAX field
 (current number of elements and maximum allowed number of elements).
 It also contains a vector of ELEMENTS, which are general purpose units
 of data.

 struct element {
  int id;    * An arbitrary counter *
  int type;  * Indicates the type of DATA; may be CHAIN, 
		CONNECTION or GHOST_CONNECTION *
  Data data; * A pointer to any data (void *) *
 };
 typedef struct element Element;
  
 struct elements {
  int total;          * The total number of elements in the vector *
  int max;            * Maximum number of elements (length of the vector *
  Element *elements;  * Pointer to a vector of Element units *
 };
 typedef struct elements Elements;

3) Chains and Connections

 Two data structures are fundamental: the vector of Chains and the 
 vector of Connections. Each one of these vectors is implemented as
 a ELEMENTS structure; what changes is the content of the field DATA
 in the structure. Two different structure serve this purpose: one 
 for connections (structure CONNECTION) and one for chains (structure
 CHAIN). Note that the structure CHAIN is accompanied by the 
 structure LINE.

 struct connection {
  int x, y;                    * Initial point of the connection *
  int size;                    * Number of pixels in the connection *
  int number_connected_chains; * Number of chains connected by connection *
  int number_processed_chains; * Should be equal to the previous *
  int *connected_chains;       * A list of the chains that are connected
				  by the connection; only the number of the
				  chains are stored, not pointers 
				  to the chains *
 };
 typedef struct connection Connection;

 struct chain {
  int xs, ys;     * Initial point of the chain *
  int ds;         * Direction of the chain in the first point; 
		    check linker3.h to see meaning of 
		    numbers (1 = NORTH, 2 = SOUTH, etc...) *
  int xe, ye;     * Last point of the chain *
  int de;         * Direction of the chain in the last point *
  int first_chain_start, second_chain_start; * Flags that tell how the 
						chain is connected in the 
						first point; see table before
						function glue_chains to 
						understand meaning *
  int first_chain_end, second_chain_end; * Flags that tell how the 
						chain is connected in the 
						last point; see table before
						function glue_chains to 
						understand meaning *
  int length;    * Number of pixels in the chain *
  int intensity_summation; * Summation of intensities of all pixels 
			      in the chain *
  LinkedList *line; * Linked list of lines in the chain *
 };
 typedef struct chain Chain;

 struct line {
  int n;        * Number of points in the line *
  float to, tf; * Time to (initial) and final (tf) for estimator of line *
  float a, b;   * Parameters for line in X direction *
  float sigma_a, sigma_b, cov_ab; * Variances for parameters in X direction *
  float c, d;   * Parameters for line in Y direction *
  float sigma_c, sigma_d, cov_cd; * Variances for parameters in Y direction *
  float St, Stt, Sxt, Syt; * Summation of time (t), time square (tt), 
			      time . X (xt) and time . Y (yt) *
  float Sx, Sxx, Sy, Syy;  * Summation of X, X^2 (xx), Y and Y^2 (yy) *
  float residual_ab, residual_cd; * Residuals for X line and Y line *
 };
 typedef struct line Line;

4) Additional Data Structures: Image Points and Linked Lists

 An auxiliary structure is used in order to manipulate pixels as units
 of data:

 struct image_point { int x, y };
 typedef struct image_point ImagePoint;

 Some auxiliary structures are necessary in order to handle linked lists
 efficiently. The code for linked lists is stored separately.

 struct nodelinked {
  int number_of_node;
  struct nodelinked *father;
  struct nodelinked *child;
  Data data;
 };
 typedef struct nodelinked NodeLinked;

 struct linkedlist {
  int id;
  int number_of_nodes;
  NodeLinked *first_node;
  NodeLinked *last_node;
 };
 typedef struct linkedlist LinkedList;

5) Data Structure for Reduced Information

 In order to facilitate usage of the chains and connections generated 
 by the program, the excessive information contained in the chain 
 structure defined above is filtered out by the function filter_all_chains.
 This function picks the vectors CHAINS and generates
 another vector, FILTER_CHAINS. 

 For each chain in CHAINS there will be a structure FilterChain in
 FILTER_CHAINS, in the same position. The structure FilterChain is
 similar to the structure Chain; differences are:
  a) All flags are replaced by PREVIOUS and NEXT flags: PREVIOUS
     contains the number of the chain (positive) or connection 
     (negative) that is attached to the beginning of the current
     chain; NEXT contains the number of the chain (positive) or
     connection (negative) that is attached to the end of the
     current chain.
  b) Instead of a linked list of lines, there is a vector of 
     lines and a field containing the number of lines.

 struct filter_chain {
  int xs, ys; 
  int ds;
  int xe, ye;
  int de;
  int previous, next;
  int length, intensity_summation;
  int number_of_lines;
  FilterLine *filter_lines;
 };
 typedef struct filter_chain FilterChain;
 
 And the filtered lines used in FilterChain are defined as:

 struct filter_line {
  float xo, yo, xf, yf;  * Initial and final coordinates of line *
  float A, B, C; * Parameters for line in form  A x + B y + C = 0 *
 };
 typedef struct filter_line FilterLine;

***************************************************************
HISTORY:
 - Oct 6 1993, Fabio Cozman created LINKER1.c; it contains 
   only a simplified first step (does not generated 
   IMAGE_CHAINS; does not fit lines; connections are single
   pixels.
 - Jan 13 1994, Fabio Cozman created LINKER2.c from LINKER1.c;
   modifications: 
   IMAGE_CHAINS is added; 
   Elements structure is added (chains and connections are
        Elements now); 
   connections are grown and may include more than one pixel;
   straight lines are fit;
   second step is added.
 - Oct 23 1994, Fabio Cozman includes the filter_chain and
   filter_line structures in order to prepare a release 
   version. LINKER3.c is already in use (a better version of
   LINKER2.c but almost the same).
 - May 24 1995, Fabio Cozman divides linker3.c into 3 files:
   linker5.c, gluer5.c and linker5-support.c. 
**************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "general.h"
#include "linked.h"
#include "linker5.h"
#include "chi-square.h"

/*****************************************************************
******************************************************************
  THE FUNCTIONS THAT FOLLOW ARE USED IN THE FIRST STEP:
  CHAIN LINKING AND LINE FITTING.
******************************************************************
*****************************************************************/

/***************************************************************
 Function that initializes the basic fields in the Chains and
 Connections structures.
Author: Fabio Cozman
Date: Oct 23 94
***************************************************************/

void initialize_chains_connections(Elements *chains, Elements *connections,
                                   int max_elements)
{
chains->total = 0;
connections->total = 0;
chains->max = max_elements;
connections->max = max_elements;
chains->elements = (Element *)malloc( (chains->max) * sizeof(Element) );
connections->elements =
  (Element *)malloc( (connections->max) * sizeof(Element) );
}

/***************************************************************
 Function that performs the central loop of chain linking.
 In order to avoid complications, auxiliar images are cleared.

 Structures CHAINS and CONNECTIONS are not initialized;
 they MUST be initialized before (totals set to 0, max_elements
 set to appropriate number and elements must be allocated).
 THIS IS IMPORTANT.

 Goes over all pixels of IMAGE_IN; if a pixel is non-zero
 (zero is background), then calls the appropriate action,
 depending on:
   - the pixel is alone in the image;
   - the pixel has a unique neighbor;
   - the pixel has two neighbors;
   - the pixel has more neighbors.
Author: Fabio Cozman
Date: Oct 93
****************************************************************/

void linking(FG_Image *image_in, FG_Image *image_out, FG_Image *image_chains, 
	     FG_Subimage bounds, 
	     Elements *chains, Elements *connections,
	     float variance_of_pixel, int significance_level_break_lines)
{
Chain *temp_chain = NULL;
int i, j;
int temp_i, temp_j;
int rs, re, cs, ce;
int number_unexplored, number_connections;
int buf_unexplored[8], buf_connections[8];

/****** ERROR HANDLING *******/
if (image_in == NULL) error("linking", 0);
if (image_out == NULL) error("linking", 1);
if (image_chains == NULL) error("linking", 2);
if (chains == NULL) error("linking", 3);
if (connections == NULL) error("linking", 4);
if (chains->total != 0) error("linking", 5);
if (connections->total != 0) error("linking", 6);

/****** Initialization: Images are Cleared and Appropriate
               Bounds to Algorithm are Set       ********/
fg_i_clrimg(image_out);
fg_i_clrimg(image_chains);
rs = bounds.rs + 1; re = bounds.re;
cs = bounds.cs + 1; ce = bounds.ce;

/****** Beginning of Main Loop *****/
for (i=rs; i<re; i++)
  for (j=cs; j<ce; j++) {
    if ( status_of_pixel(image_in, image_out, image_chains, bounds,
			 &number_unexplored, &number_connections, 
			 buf_unexplored, buf_connections, i, j, 0) ) {

#ifdef DEBUG
printf("\n New event at (%d,%d);  %d unlabelled, %d connection-type",
       i, j, number_unexplored, number_connections);
if (number_unexplored > 0) printf("\n  Unlabelled: ");
for (k=0;k<number_unexplored; k++)  printf(" [ %d ]", buf_unexplored[k]);
if (number_connections > 0) printf("\n  Connections: ");
for (k=0; k<number_connections; k++) printf(" [ %d ]", buf_connections[k]); 
printf("\n");
#endif

if ( (number_unexplored + number_connections) > 2 )
  handle_connection(image_in, image_out, image_chains, bounds, connections,
		    buf_connections, number_connections, i, j, 0);
else {
temp_chain = create_chain(chains, i, j);
fg_i_iputpixel(image_out, i, j, chains->total);
update_line((Line *)(temp_chain->lines->first_node->data), i, j,
	    variance_of_pixel, TO_END);
temp_chain->length = 1;
temp_chain->intensity_summation = fg_i_igetpixel(image_in, i, j);

switch (number_unexplored) {
  /***** First case: number of unlabelled neighbors is zero *****/
 case 0:
  switch (number_connections) {
   case 0: /***** A pixel alone *****/     break;
   case 1: /**** A pixel close to a connection ***/
      connect_one(connections, temp_chain, buf_connections[0], 
		  chains->total, TO_END);
     break;
   case 2: /**** A pixel close to two connections ****/
     connect_one(connections, temp_chain, buf_connections[0], 
		 chains->total, TO_START);
     if (buf_connections[0] != buf_connections[1]) 
       connect_one(connections, temp_chain, buf_connections[1], 
		   chains->total, TO_END);
     else 
       temp_chain->first_chain_end = buf_connections[0];
     break;
   default: error("linking", 7); 
   }
  break;
  /****** Second case: number of unlabelled neighbors is one ***/
 case 1:
  temp_chain->ds = buf_unexplored[0]; /** Adjust initial direction of chain **/
  switch (number_connections) {
    case 0: /**** A pixel that starts a chain ****/
    follow(image_in, image_out, image_chains, bounds, chains, connections, 
	   temp_chain, chains->total, i, j, 
	   buf_unexplored[0], buf_unexplored[0],
	   variance_of_pixel, significance_level_break_lines, TO_END);
     break;
   case 1:
    connect_one(connections, temp_chain, buf_connections[0], 
		chains->total, TO_START);
    follow(image_in, image_out, image_chains, bounds, chains, connections,
	   temp_chain, chains->total, i, j, 
	   buf_unexplored[0], buf_unexplored[0],
	   variance_of_pixel, significance_level_break_lines, TO_END);
     break;
   default: error("linking", 8);
   }
  if (temp_chain->length == 1) /* Check whether chain has more than a pixel */
    { temp_chain->ds = temp_chain->de = NO_DIRECTION; }
  break;
  /***** Third case: number of unlabelled neighbors is two ****/
 case 2:
  switch (number_connections) {
    case 0:
     temp_i = i; temp_j = j;
     next_one(&temp_i, &temp_j, buf_unexplored[0]);
     follow(image_in, image_out, image_chains, bounds,  chains, connections,
	    temp_chain, chains->total, i,j, 
	    buf_unexplored[1], invert_direction(buf_unexplored[0]),
	    variance_of_pixel, significance_level_break_lines, TO_START);
     if ( fg_i_igetpixel(image_out, temp_i, temp_j) ==  0) 
       follow(image_in, image_out, image_chains, bounds, chains, connections,
	      temp_chain, chains->total, i, j, 
	      buf_unexplored[0], invert_direction(buf_unexplored[1]),
	      variance_of_pixel, significance_level_break_lines, TO_END);
     else { /*** Chain is closed through itself or through connection ***/
       temp_chain->first_chain_end = - connections->total;
       temp_chain->de = buf_unexplored[1];
     }
     verify_whether_closed_curve(temp_chain, chains->total);
     break;
    default: error("linking", 9);
   }
  if (temp_chain->length == 1) /* Check whether chain has more than a pixel */
    { temp_chain->ds = temp_chain->de = NO_DIRECTION; }
  break;
  /***** Fourth case: number of unlabelled neighbors is larger than two ****/
 default: error("linking", 10); 
} } } }
}

/*************************************************************
 Function that checks the neighbors of a given pixel (i,j)
 and discovers how many unlabelled pixels and connection-type
 there are around. It modifies the contents of
 number_unexplored, number_connections and fill the
 buf_unexplored (with the directions to the unlabelled pixels)
 and the buf_connection (with the code of the connection - this
 is a NEGATIVE number!). 
 The value of id_chain is zero if the checking of labelled
 pixels around is absolute (no labelled pixel is possible); 
 otherwise, pixels with label equal to id_chain are accepted.
 The routine builds the chain-codes for the pixels on the 
 fly.
 It is important also to notice that the checks on the bounds
 of the image are made in this function.
Author: Fabio Cozman
Date: Oct 8 93
*************************************************************/

int status_of_pixel(FG_Image *image_in, FG_Image *image_out, 
		    FG_Image *image_chains, FG_Subimage bounds,
		    int *number_unexplored, int *number_connections,
		    int buf_unexplored[], int buf_connections[], 
		    int i, int j, int id_chain)
{
unsigned int chain_code = 0;
int rs, re, cs, ce; 

(*number_unexplored) = (*number_connections) = 0;
rs = bounds.rs + 1; re = bounds.re - 1;
cs = bounds.cs + 1; ce = bounds.ce - 1;

  /******* If the pixel is still unexplored ******/
if ( (fg_i_igetpixel(image_in, i, j) != 0) &&
     (fg_i_igetpixel(image_out, i, j) == 0) ) {
     /***** If the North pixel is not zero in the image *****/
  if ( (i > rs) && (fg_i_igetpixel(image_in, i-1, j) != 0) ) {
    if (analyse_pixel_value(fg_i_igetpixel(image_out, i-1, j), NORTH,
			    number_unexplored, number_connections,
			    buf_unexplored, buf_connections, id_chain)) 
      error("status_of_pixel", 1);
    else chain_code = ( chain_code | 1 ); }
     /***** If the South pixel is not zero in the image *****/
  if ( (i < re) && (fg_i_igetpixel(image_in, i+1, j) != 0) ) {
    if (analyse_pixel_value(fg_i_igetpixel(image_out, i+1, j), SOUTH,
			    number_unexplored, number_connections,
			    buf_unexplored, buf_connections, id_chain))
      error("status_of_pixel", 2);
    else chain_code = ( chain_code | 2 ); }
     /***** If the West pixel is not zero in the image *****/
  if ( (j > cs) && (fg_i_igetpixel(image_in, i, j-1) != 0) ) {
     if (analyse_pixel_value(fg_i_igetpixel(image_out, i, j-1), WEST,
			     number_unexplored, number_connections,
			     buf_unexplored, buf_connections, id_chain)) 
       error("status_of_pixel", 3); 
     else  chain_code = ( chain_code | 4 ); }
     /***** If the East pixel is not zero in the image *****/
  if ( (j < ce) && (fg_i_igetpixel(image_in, i, j+1) != 0) ) {
     if (analyse_pixel_value(fg_i_igetpixel(image_out, i, j+1), EAST,
			     number_unexplored, number_connections,
			     buf_unexplored, buf_connections, id_chain))
       error("status_of_pixel", 4); 
    else   chain_code = ( chain_code | 8 ); }
     /***** If the Norht-West pixel is not zero in the image *****/
  if ( (i > rs) && (j > cs) && (fg_i_igetpixel(image_in, i-1, j-1) != 0 ) ) {
     if (analyse_pixel_value(fg_i_igetpixel(image_out, i-1, j-1), NORTH_WEST,
                             number_unexplored, number_connections,
                             buf_unexplored, buf_connections, id_chain))
       error("status_of_pixel", 5);
     else       chain_code = ( chain_code | 16 ); }
     /***** If the North-East pixel is not zero in the image *****/
  if ( (i > rs) && (j < ce) && (fg_i_igetpixel(image_in, i-1, j+1) != 0) ) {
     if (analyse_pixel_value(fg_i_igetpixel(image_out, i-1, j+1), NORTH_EAST,
                             number_unexplored, number_connections,
                             buf_unexplored, buf_connections, id_chain)) 
       error("status_of_pixel", 6);
     else       chain_code = ( chain_code | 32 ); }
     /***** If the South-West pixel is not zero in the image *****/
  if ( (i < re) && (j > cs) && (fg_i_igetpixel(image_in, i+1, j-1) != 0) ) {
     if (analyse_pixel_value(fg_i_igetpixel(image_out, i+1, j-1), SOUTH_WEST,
			     number_unexplored, number_connections,
			     buf_unexplored, buf_connections, id_chain)) 
       error("status_of_pixel", 7);
     else       chain_code = ( chain_code | 64 ); }
     /***** If the South-East pixel is not zero in the image *****/
  if ( (i < re) && (j < ce) && (fg_i_igetpixel(image_in, i+1, j+1) != 0) ) {
      if (analyse_pixel_value(fg_i_igetpixel(image_out, i+1, j+1), SOUTH_EAST,
			      number_unexplored, number_connections,
			      buf_unexplored, buf_connections, id_chain))
	error("status_of_pixel", 8);
      else       chain_code = ( chain_code | 128 ); }
     /***** Put chain-code and return ******/
  fg_i_iputpixel(image_chains, i, j, chain_code);
  return(1);
}
else
  return(0);
}

/****************************************************************
 Function that, for a given value and a given direction, verifies
 whether there is a unlabelled pixel or a connection in the 
 direction from the given pixel (in both cases returns 0; 
 otherwise returns 1.
 If there is a unlabelled pixel, then put the direction in
 buf_unexplored; if there is a connection-type pixel, then put
 the value of connection in buf_connection.
Author: Fabio Cozman
Date: Oct 8 1993
***************************************************************/

int analyse_pixel_value(int value, int direction, 
                        int *number_unexplored, int *number_connections, 
                        int buf_unexplored[], int buf_connections[],
                        int id_chain)
{
if (value == 0) {
  buf_unexplored[ ( *number_unexplored ) ] = direction;
  (*number_unexplored)++;
}
else {
  if (value < 0) {
    buf_connections[ (*number_connections) ] = value;
    (*number_connections)++;
  }
  else 
    if ( (id_chain == 0) || (id_chain != value) ) return(1);
}
return(0);
} 
 
/**************************************************************
 Function that starts and follows a chain in the image; 
 the chains ends at any pixel that has more than one 
 unlabelled neighbor OR any connection.
 It returns the new value of total_connections; this value
 is changed only if a new_connection is created (i.e., if the
 chain ends not by having no further points).
Author: Fabio Cozman
Date: Oct 6 93
**************************************************************/

void follow(FG_Image *image_in, FG_Image *image_out, FG_Image *image_chains, 
	    FG_Subimage bounds, 
	    Elements *chains, Elements *connections, 
	    Chain *temp_chain, int id_chain, int i, int j, 
	    int direction, int previous_direction,
	    float variance_of_pixel, int significance_level_break_lines, 
	    int to_end)
{
int temp_i, temp_j, aux_i, aux_j;
int flag_chain = 1;
int number_connections, number_unexplored;
int buf_unexplored[8], buf_connections[8];
Line *current_line = NULL;
Line *aux_line = NULL;
float limit_residual;
   
         /**************** ERROR HANDLING ************/
if (image_in == NULL) error("follow", 0);
if (image_out == NULL) error("follow", 1);
if (image_chains == NULL) error("follow", 2);
if (chains == NULL) error("follow", 3);
if (chains->elements == NULL) error("follow", 4);
if (connections == NULL) error("follow", 5);
if (connections->elements == NULL) error("follow", 6);
if (temp_chain == NULL) error("follow", 7);
if (id_chain < 0) error("follow", 8);

aux_i = temp_i = i; aux_j = temp_j = j;
next_one(&temp_i, &temp_j, direction);
if (to_end)
  current_line = (Line *)(temp_chain->lines->last_node->data);
else
  current_line = (Line *)(temp_chain->lines->first_node->data);

         /****** Follow the chain ***********/
while (flag_chain) {    /*** While the chain continues ****/
status_of_pixel(image_in, image_out, image_chains, bounds,
		&number_unexplored, &number_connections, 
		buf_unexplored, buf_connections, temp_i, temp_j, id_chain);
            /** Increases the length and intensity summation of the chain **/
temp_chain->length++;
temp_chain->intensity_summation += fg_i_igetpixel(image_in, temp_i, temp_j);
           /*** If the pixel is the last one in the chain ****/
if ( (number_unexplored == 0) && (number_connections == 0) ) {
  fg_i_iputpixel(image_out, temp_i, temp_j, id_chain);
  update_line(current_line, temp_i, temp_j, variance_of_pixel, to_end);
  fill_to_end(temp_chain, temp_i, temp_j, direction, to_end);
  flag_chain = 0;
}
else {      /****** If the pixel has only one unlabelled neighbor ****/
if ( (number_unexplored == 1) && (number_connections == 0) ) {
  update_line(current_line, temp_i, temp_j, variance_of_pixel, to_end);
  limit_residual = chi_square_critical_value(current_line->n - 2, 
					     significance_level_break_lines);
                     /*** Significance test that breaks lines ***/
  if ((current_line->residual_ab/variance_of_pixel > limit_residual) ||
      (current_line->residual_cd/variance_of_pixel > limit_residual) ) {
    current_line = create_line();
    insert_unsorted_linked_list(temp_chain->lines, (Data)current_line, to_end);
  }
  fg_i_iputpixel(image_out, temp_i, temp_j, id_chain);
  aux_i = temp_i;   aux_j = temp_j;
  next_one(&temp_i, &temp_j, buf_unexplored[0]);
  previous_direction = direction;
  direction = buf_unexplored[0];
}
else {     /*** If the pixel meets one connection ***/
if ( (number_unexplored == 0) && (number_connections == 1) ) {
  fg_i_iputpixel(image_out, temp_i, temp_j, id_chain);
  update_line(current_line, temp_i, temp_j, variance_of_pixel, to_end);
  connect_one(connections, temp_chain, buf_connections[0],
	      chains->total, to_end);
  fill_to_end(temp_chain, temp_i, temp_j, direction, to_end);
  flag_chain = 0;
}
else {   /*** If the pixel has several unlabelled neighbors ***/
if ( (number_unexplored >= 1) && (number_connections == 0) ) {
  aux_line = (to_end ? (Line *)(temp_chain->lines->last_node->data) :
	               (Line *)(temp_chain->lines->first_node->data));
  if (aux_line->n == 0) 
    delete_extremity_unsorted_linked_list(temp_chain->lines, to_end); 
  handle_connection(image_in, image_out, image_chains, bounds, 
		    connections, buf_connections, 
		    number_connections, temp_i, temp_j, id_chain);
  connect_one(connections, temp_chain, - connections->total, 
	      chains->total, to_end);
  temp_chain->length--;
  temp_chain->intensity_summation -= fg_i_igetpixel(image_in, temp_i, temp_j);
  fill_to_end(temp_chain, aux_i, aux_j, previous_direction, to_end);
  flag_chain = 0;
}
else {  error("follow", 9); }
} } } }
}

/******************************************************************
 Function that takes a chain and a point (x,y) and puts the point
 in the chain as starting or ending point, depending on the value
 of the flag TO_END.
Author: Fabio Cozman
Date: Jan 29 94
*****************************************************************/

void fill_to_end(Chain *temp_chain, int x, int y, int direction, int to_end)
{
if (temp_chain == NULL) error("fill_to_end", 1);

if (to_end == TO_END) { 
  temp_chain->xe = x;  
  temp_chain->ye = y;
  temp_chain->de = invert_direction(direction);
}
else {
  if (to_end != TO_START) error("fill_to_end", 2);
  temp_chain->xs = x;  
  temp_chain->ys = y;
  temp_chain->ds = invert_direction(direction);
}
}

/*********************************************************************
 Function that takes:
  vector of connections and number of connection => takes a connection
  a chain
 and assigns
  the chain as one of the chains of the connection.
  the connection as one of the connections of the chain (using TO_END).
Author: Fabio Cozman
Date: Oct 6 93
*********************************************************************/

void connect_one(Elements *connections, Chain *temp_chain, 
		 int index_connection_to, int index_chain, int to_end)
{
Connection *connection_to = NULL;

         /**************** ERROR HANDLING *******************/
if (connections == NULL) error("connect_one", 0);
if (connections->elements == NULL) error("connect_one", 1);
if (index_connection_to >= 0) error("connect_one", 2);
if (temp_chain == NULL) error("connect_one", 3);

         /**** Adjust the connection ******/
connection_to = 
  (Connection *)(connections->elements[ ( - index_connection_to ) ].data);
if ( (connection_to->number_processed_chains) >= 
    (connection_to->number_connected_chains) ) error("connect_one", 4);  
connection_to->connected_chains[ (connection_to->number_processed_chains) ] = 
  index_chain;
(connection_to->number_processed_chains)++;
         /**** Adjust the chain *********/
if (to_end)
     temp_chain->first_chain_end = index_connection_to;
else
    temp_chain->first_chain_start = index_connection_to;
}

/*******************************************************************
 Function that creates a connection and fills the parameters of
 the connection.
 Return the new total_connections.
Author: Fabio Cozman
Date: Oct 6 93
*******************************************************************/

void handle_connection(FG_Image *image_in, FG_Image *image_out, 
                       FG_Image *image_chains, FG_Subimage bounds,
		       Elements *connections, 
		       int buf_connections[], int number_connections, 
                       int i, int j, int id_chain)
{
int total_connections;
Connection *temp_connection = NULL;
LinkedList *extremities_of_connection = NULL;
NodeLinked *node = NULL;
ImagePoint *image_point = NULL;

         /**************** ERROR HANDLING ************/
if (connections == NULL) error("handle_connection", 0);
if (connections->elements == NULL) error("handle_connection", 1);
if (image_out == NULL) error("handle_connection", 2);
if (image_in == NULL) error("handle_connection", 3);
if (image_chains == NULL) error("handle_connection", 4);

          /******* Increment total connections ********/
increment_total_elements(connections);
total_connections = connections->total;
          /******* Creates a connection structure *******/
if ( (temp_connection = (Connection *)(malloc( sizeof(Connection) ))) == NULL )
  error("handle_connection", 5);
          /******** Attaches connection structure in connections vector ***/
connections->elements[ total_connections ].id = - total_connections;
connections->elements[ total_connections ].type = CONNECTION;
connections->elements[ total_connections ].data = (Data)temp_connection;
          /******* Fill new connection parameters ******/
temp_connection->x = i;  
temp_connection->y = j;
temp_connection->size = 0;
temp_connection->number_processed_chains = 0;
temp_connection->number_connected_chains = ((id_chain == 0) ? 0 : 1);
          /****** Initialize list of extremities of connection *******/
extremities_of_connection = (LinkedList *)malloc(sizeof(LinkedList));
extremities_of_connection->id = 0;
extremities_of_connection->number_of_nodes = 0;
extremities_of_connection->first_node = 
  extremities_of_connection->last_node = NULL;
          /******* Grow connection ********/
fg_i_iputpixel(image_out, i, j, - total_connections);
temp_connection->size = labelline(image_in, image_out, image_chains, bounds,
				  i, j, - total_connections, 1, id_chain, 
				  &(temp_connection->number_connected_chains),
				  extremities_of_connection);

if (temp_connection->number_connected_chains > 0)
  temp_connection->connected_chains = 
    (int *)(malloc((temp_connection->number_connected_chains) * sizeof(int) ));
          /******* Clear all extremities of the connection *******/
node = extremities_of_connection->first_node;
while (node != NULL) {
  image_point = (ImagePoint *)(node->data);
  fg_i_iputpixel(image_chains, image_point->x, image_point->y, 0);
  node = node->child;
}
free_linked_list_and_data(extremities_of_connection);
}

/***************************************************************
 Function that grows a connection recursively, looking at all
 the eight directions (for 8-connectivity). The function 
 returns the size of the grown connection in pixels; on the
 fly, the number of unlabelled chains that are connected to
 the connection is computed. The value of ID_CHAIN is the
 value of a labelled chain that is tolerated in the neighborhood
 of the connection; if ID_CHAIN is zero, no labelled chain
 is tolerated. Chain-codes are 
 formed on the fly; connection stops when there are no more
 pixels or a pixel with only two neighbors is found.
 Function labelline is adapted from CRE.C (Krotkov's code).
 For speed, this function has no ERROR HANDLING.
Author: Fabio Cozman, adapted from CRE.
Date: Jan 30 1994
**************************************************************/

int labelline(FG_Image *image_in, FG_Image *image_out, FG_Image *image_chains, 
	      FG_Subimage bounds, 
	      int startx, int starty, int label, int cnt, int id_chain, 
	      int *number_connected_chains, 
	      LinkedList *extremities_of_connection)
{
int i, j, k, l;
int rs, re, cs, ce;

rs = bounds.rs; re = bounds.re;
cs = bounds.cs; ce = bounds.ce;

/* horizontal (column direction), increased */
for (i = (starty + 1);  ((i <= ce) && 
        (continue_connection_test(image_in, image_out, image_chains, bounds, 
				  startx, i, id_chain, number_connected_chains,
				  extremities_of_connection))); i++);
if (i > (starty + 1)) {
  for (j = (starty + 1); j<i; j++) 
    fg_i_iputpixel(image_out, startx, j, label);
  cnt = cnt + i - starty - 1;
  for (j = (starty + 1); j<i; j++)
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    startx, j, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* horizontal (column direction), decreased */
for (i = (starty - 1); ((i >= cs) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  startx, i, id_chain, number_connected_chains,
				  extremities_of_connection))); i--);
if (i < (starty - 1)) {
  for (j = (starty - 1); j>i; j--)
	fg_i_iputpixel(image_out, startx, j, label);
  cnt = cnt + starty - i - 1;
  for (j = (starty - 1); j > i; j--)
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    startx, j, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}   
    
/* vertical (row direction), increased */
for (i = (startx + 1); ((i <= re) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, starty, id_chain, number_connected_chains,
				  extremities_of_connection))); i++);
if (i > (startx + 1)) {
  for (j = (startx + 1); j < i; j++)
	fg_i_iputpixel(image_out, j, starty, label);
  cnt = cnt + i - startx - 1;
  for (j = (startx + 1); j < i; j++)
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    j, starty, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* vertical (row direction), decreased */
for (i = (startx - 1); ((i >= rs) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, starty, id_chain, number_connected_chains,
				  extremities_of_connection))); i--);
if (i < (startx - 1)) {
  for (j = (startx - 1); j>i; j--)
	fg_i_iputpixel(image_out, j, starty, label);
  cnt = cnt + startx - i - 1;
  for (j = (startx - 1); j > i; j--)
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    j, starty, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* diagonal (row and column), right increase */
for (i = (startx + 1), j = (starty + 1); ((i <= re) && (j <= ce) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, j, id_chain, number_connected_chains, 
				  extremities_of_connection))); i++, j++);
if ((i > (startx + 1)) && (j > (starty + 1))) {
  for (k = (startx + 1), l = (starty + 1); (k<i) && (l<j); k++, l++) 
	fg_i_iputpixel(image_out, k, l, label);
  cnt = cnt + i - startx - 1;
  for (k = (startx + 1), l = (starty + 1); (k<i) && (l<j); k++, l++) 
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    k, l, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* diagonal (row and column), right decrease */
for (i = (startx - 1), j = (starty + 1); ((i >= rs) && (j <= ce) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, j, id_chain, number_connected_chains,
				  extremities_of_connection))); i--, j++);
if ((i < (startx - 1)) && (j > (starty + 1))) {
  for (k = (startx - 1), l = (starty + 1); (k>i) && (l<j); k--, l++) 
	fg_i_iputpixel(image_out, k, l, label);
  cnt = cnt + startx - i - 1;
  for (k = (startx - 1), l = (starty + 1); (k>i) && (l<j); k--, l++) 
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    k, l, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* diagonal (row and column), left increase */
for (i = (startx + 1), j = (starty - 1); ((i <= re) && (j >= cs) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, j, id_chain, number_connected_chains, 
				  extremities_of_connection))); i++, j--);
if ((i > (startx + 1)) && (j < (starty - 1))) {
  for (k = (startx + 1), l = (starty - 1); (k<i) && (l>j); k++, l--) 
	fg_i_iputpixel(image_out, k, l, label);
  cnt = cnt + i - startx - 1;
  for (k = (startx + 1), l = (starty - 1); (k<i) && (l>j); k++, l--) 
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    k, l, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

/* diagonal (row and column), left decrease */
for (i = (startx - 1), j = (starty - 1); ((i >= rs) && (j >= cs) && 
	(continue_connection_test(image_in, image_out, image_chains, bounds, 
				  i, j, id_chain, number_connected_chains,
				  extremities_of_connection))); i--, j--);
if ((i < (startx - 1)) && (j < (starty - 1))) {
  for (k = (startx - 1), l = (starty - 1); (k>i) && (l>j); k--, l--) 
	fg_i_iputpixel(image_out, k, l, label);
  cnt = cnt + startx - i - 1;
  for (k = (startx - 1), l = (starty - 1); (k>i) && (l>j); k--, l--) 
    cnt = labelline(image_in, image_out, image_chains, bounds, 
		    k, l, label, cnt, id_chain, number_connected_chains,
		    extremities_of_connection);
}

return(cnt);
}

/************************************************************
 Function that checks whether a connection ends or not. 
 If the current pixel is empty, nothing needs to be checked.
 If not, the  check is simply to verify whether or not 
 the current pixel has more than two neighbors; 
 if it has, then it is in the connection; if not, the 
 connection stops there.
 Returns 1 if connection continues; 0 if not.
 For speed, this function has no ERROR HANDLING.
 - Some extra speed could be obtained by using the chain_code
   and function query_number_bits; if a chain pixel already
   detected were found, all readings from image_in would be
   avoided. This extra gain would be small and it is not
   implemented here.
Author: Fabio Cozman
Date: Jan 31 94
***********************************************************/

int continue_connection_test(FG_Image *image_in, FG_Image *image_out, 
			     FG_Image *image_chains, FG_Subimage bounds,
			     int x, int y, int id_chain, 
			     int *number_connected_chains,
			     LinkedList *extremities_of_connection)
{
int pixel;
int rs, re, cs, ce;
int number_neighbors = 0;
int chain_code = 0;
ImagePoint *point;

if ((id_chain != 0) && ((pixel = fg_i_igetpixel(image_out, x, y)) > 0))
  if (pixel != id_chain) error("continue_connection_test", 0);

if ((fg_i_igetpixel(image_in, x, y) == 0) || 
    (fg_i_igetpixel(image_out, x, y) < 0))
  return(0);

rs = bounds.rs + 1; re = bounds.re - 1;
cs = bounds.cs + 1; ce = bounds.ce - 1;

if ((x < rs) || (x > re) || (y < cs) || (y > ce)) return(0);

if ((x > rs) && (fg_i_igetpixel(image_in, x-1, y) != 0)) /** NORTH **/
  { number_neighbors++; chain_code = (chain_code | 1); }
if ((x < re) && (fg_i_igetpixel(image_in, x+1, y) != 0)) /** SOUTH **/
  { number_neighbors++; chain_code = (chain_code | 2); }
if ((y > cs) && (fg_i_igetpixel(image_in, x, y-1) != 0)) /** WEST **/
  { number_neighbors++; chain_code = (chain_code | 4); }
if ((y < ce) && (fg_i_igetpixel(image_in, x, y+1) != 0)) /** EAST **/
  { number_neighbors++; chain_code = (chain_code | 8); }
if ((x > rs) && (y > cs) &&
    (fg_i_igetpixel(image_in, x-1, y-1) != 0)) /** NORTH_WEST **/
  { number_neighbors++; chain_code = (chain_code | 16); }
if ((x > rs) && (y < ce) &&
    (fg_i_igetpixel(image_in, x-1, y+1) != 0)) /** NORTH_EAST **/
  { number_neighbors++; chain_code = (chain_code | 32); }
if ((x < re) && (y > cs) && 
    (fg_i_igetpixel(image_in, x+1, y-1) != 0)) /** SOUTH_WEST **/
  { number_neighbors++; chain_code = (chain_code | 64); }
if ((x < re) && (y < ce) && 
    (fg_i_igetpixel(image_in, x+1, y+1) != 0)) /** SOUTH_EAST **/
  { number_neighbors++; chain_code = (chain_code | 128); }

if (number_neighbors == 0) error("continue_connection_test", 1); 
if ((number_neighbors == 1) || (number_neighbors == 2))  { 
  if (fg_i_igetpixel(image_chains, x, y) == 0) {
    (*number_connected_chains)++; 
    fg_i_iputpixel(image_chains, x, y, chain_code); 
    point = (ImagePoint *)malloc(sizeof(ImagePoint));
    point->x = x;
    point->y = y;
    insert_unsorted_linked_list(extremities_of_connection,(Data)point,TO_END);
  }
  return(0);
}
else { fg_i_iputpixel(image_chains, x, y, chain_code); return(1); }
}

/******************************************************************
 Function that detects whether a chain is actually a closed curve
 (when the first and last pixels are neighbors) and fixes this
 by setting correctly the temp_chain->second_chain_{start,end} variables.
Author: Fabio Cozman
Date: Feb 10 94
*******************************************************************/

void verify_whether_closed_curve(Chain *chain, int id_chain)
{
if ((chain->length > 3) && 
    (abs(chain->xs - chain->xe) < 2) && (abs(chain->ys - chain->ye) < 2)) {
  chain->first_chain_start =  chain->second_chain_start = 
    chain->first_chain_end =  chain->second_chain_end   = id_chain; 
}
}

